"use client";

export type ClientPortalAuth = {
  clientCode: string;
  homePage: string;
  authorizedAt: string;
};

const CLIENT_PORTAL_AUTH_STORAGE_KEY = "clientPortalAuth";

const normalizePortalPath = (homePage: string) => {
  if (!homePage) return "/";
  if (homePage.startsWith("/")) return homePage;
  if (homePage.startsWith("app/")) {
    return `/${homePage.replace(/^app\//, "")}`;
  }
  return `/${homePage}`;
};

export function storeClientPortalAuth(clientCode: string, homePage: string) {
  if (typeof window === "undefined") return;
  const payload: ClientPortalAuth = {
    clientCode,
    homePage: normalizePortalPath(homePage),
    authorizedAt: new Date().toISOString(),
  };
  window.localStorage.setItem(CLIENT_PORTAL_AUTH_STORAGE_KEY, JSON.stringify(payload));
}

export function getClientPortalAuth() {
  if (typeof window === "undefined") return null;
  const raw = window.localStorage.getItem(CLIENT_PORTAL_AUTH_STORAGE_KEY);
  if (!raw) return null;
  try {
    const parsed = JSON.parse(raw) as ClientPortalAuth;
    if (!parsed?.clientCode || !parsed?.homePage) return null;
    return {
      ...parsed,
      homePage: normalizePortalPath(parsed.homePage),
    };
  } catch {
    return null;
  }
}

export function isMeiwaCorePortalAuth(auth: ClientPortalAuth | null) {
  if (!auth) return false;
  return normalizePortalPath(auth.homePage).startsWith("/meiwa-core");
}

export function resolvePortalHomePath(homePage: string) {
  return normalizePortalPath(homePage);
}