"use client";

import { useEffect, useMemo, useState } from 'react';
import { useRouter } from 'next/navigation';
import { format } from 'date-fns';
import { es } from 'date-fns/locale';
import { supabase, type Database } from '@/lib/supabase';
import { DashboardLayout } from '@/components/dashboard/dashboard-layout';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Archive, ArrowLeftCircle, Clock3, RefreshCcw } from 'lucide-react';
import { toast } from 'sonner';

type Task = Database['public']['Tables']['tasks']['Row'];

const priorityColors = {
  low: 'bg-green-100 text-green-800',
  medium: 'bg-yellow-100 text-yellow-800',
  high: 'bg-red-100 text-red-800',
};

const priorityLabels = {
  low: 'Baja',
  medium: 'Media',
  high: 'Alta',
};

export default function ArchivePage() {
  const router = useRouter();
  const [tasks, setTasks] = useState<Task[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    let isMounted = true;

    const loadData = async () => {
      const {
        data: { session },
      } = await supabase.auth.getSession();

      if (!session?.user) {
        router.push('/');
        return;
      }

      if (!isMounted) return;
      await fetchArchivedTasks(session.user.id);
      if (isMounted) {
        setLoading(false);
      }
    };

    loadData();

    const {
      data: { subscription },
    } = supabase.auth.onAuthStateChange((_event, session) => {
      if (!session?.user) {
        router.push('/');
      } else {
        fetchArchivedTasks(session.user.id);
      }
    });

    return () => {
      isMounted = false;
      subscription.unsubscribe();
    };
  }, [router]);

  const fetchArchivedTasks = async (userId: string) => {
    const { data, error } = await supabase
      .from('tasks')
      .select('*')
      .eq('user_id', userId)
      .eq('completed', true)
      .order('updated_at', { ascending: false });

    if (error) {
      console.error('Error fetching archived tasks:', error);
      toast.error('No se pudieron cargar las tareas archivadas');
      return;
    }

    setTasks(data || []);
  };

  const handleRestore = async (taskId: string) => {
    const { error } = await supabase
      .from('tasks')
      .update({ completed: false, updated_at: new Date().toISOString() })
      .eq('id', taskId);

    if (error) {
      toast.error('No se pudo restaurar la tarea');
      return;
    }

    setTasks((prev) => prev.filter((task) => task.id !== taskId));
    toast.success('Tarea movida a pendientes');
  };

  const archivedCount = tasks.length;
  const latestRestored = useMemo(() => tasks[0], [tasks]);

  return (
    <DashboardLayout>
      <div className="space-y-6">
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Archivo</h1>
            <p className="text-gray-600">
              Consulta las tareas completadas y restáuralas si es necesario.
            </p>
          </div>
          <Button variant="outline" className="gap-2" onClick={() => router.push('/meiwa-core/dashboard')}>
            <ArrowLeftCircle className="h-4 w-4" />
            Volver al dashboard
          </Button>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-4">
          <Card>
            <CardHeader className="pb-3">
              <CardTitle className="flex items-center gap-2 text-lg">
                <Archive className="h-5 w-5 text-blue-600" />
                Total archivadas
              </CardTitle>
              <CardDescription>Tareas completadas que permanecen en el archivo.</CardDescription>
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold text-gray-900">{archivedCount}</p>
            </CardContent>
          </Card>

          <Card className="lg:col-span-2">
            <CardHeader className="pb-3">
              <CardTitle className="text-lg">Última actividad</CardTitle>
              <CardDescription>
                La tarea archivada más reciente aparece aquí para referencia rápida.
              </CardDescription>
            </CardHeader>
            <CardContent>
              {latestRestored ? (
                <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-3 p-4 border rounded-lg bg-white">
                  <div className="space-y-1">
                    <p className="text-sm text-gray-600">Tarea más reciente</p>
                    <p className="text-lg font-semibold text-gray-900">{latestRestored.title}</p>
                    {latestRestored.due_date && (
                      <div className="flex items-center gap-2 text-sm text-gray-600">
                        <Clock3 className="h-4 w-4" />
                        {format(new Date(latestRestored.due_date), "dd 'de' MMMM, yyyy", { locale: es })}
                      </div>
                    )}
                  </div>
                  <Button
                    variant="outline"
                    className="gap-2"
                    onClick={() => handleRestore(latestRestored.id)}
                  >
                    <RefreshCcw className="h-4 w-4" />
                    Restaurar
                  </Button>
                </div>
              ) : (
                <p className="text-gray-500">Aún no tienes tareas en el archivo.</p>
              )}
            </CardContent>
          </Card>
        </div>

        <Card>
          <CardHeader>
            <CardTitle className="text-xl">Tareas en el archivo</CardTitle>
            <CardDescription>
              Estas tareas ya se completaron. Puedes dejarlas aquí para consulta o restaurarlas.
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-3">
            {loading ? (
              <div className="flex items-center justify-center py-10 text-gray-500">
                <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-blue-600 mr-3" />
                Cargando archivo...
              </div>
            ) : tasks.length === 0 ? (
              <div className="text-center text-gray-500 py-8">
                No hay tareas archivadas todavía.
              </div>
            ) : (
              tasks.map((task) => (
                <div
                  key={task.id}
                  className="p-4 rounded-lg border bg-white shadow-sm flex flex-col sm:flex-row sm:items-start sm:justify-between gap-3"
                >
                  <div className="space-y-2 min-w-0">
                    <div className="flex flex-wrap items-center gap-2">
                      <Badge className={priorityColors[task.priority]}>
                        {priorityLabels[task.priority]}
                      </Badge>
                      <Badge variant="outline" className="text-xs">Completada</Badge>
                    </div>
                    <h3 className="text-lg font-semibold text-gray-900 break-words">{task.title}</h3>
                    {task.description && (
                      <p className="text-sm text-gray-600 whitespace-pre-line break-words">
                        {task.description}
                      </p>
                    )}
                    <p className="text-xs text-gray-500">
                      Archivada el {format(new Date(task.updated_at), "dd 'de' MMMM, yyyy", { locale: es })}
                    </p>
                  </div>
                  <div className="flex items-center gap-2 self-end sm:self-center">
                    <Button variant="outline" size="sm" className="gap-2" onClick={() => handleRestore(task.id)}>
                      <RefreshCcw className="h-4 w-4" />
                      Restaurar
                    </Button>
                  </div>
                </div>
              ))
            )}
          </CardContent>
        </Card>
      </div>
    </DashboardLayout>
  );
}
