"use client";

import { useEffect, useState } from 'react';
import { useRouter } from 'next/navigation';
import { supabase } from '@/lib/supabase';
import { User } from '@supabase/supabase-js';
import { DashboardLayout } from '@/components/dashboard/dashboard-layout';
import { TaskList } from '@/components/tasks/task-list';
import { TaskForm } from '@/components/tasks/task-form';
import { StatsCards } from '@/components/dashboard/stats-cards';
import { Button } from '@/components/ui/button';
import { Plus } from 'lucide-react';
import { useState as useTaskModal } from 'react';

export default function Dashboard() {
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [showTaskForm, setShowTaskForm] = useState(false);
  const [refreshKey, setRefreshKey] = useState(0);  
  const router = useRouter();

  useEffect(() => {
    const getSession = async () => {
      const { data: { session } } = await supabase.auth.getSession();
      
      if (!session?.user) {
        router.push('/');
        return;
      }
      
      setUser(session.user);
      setLoading(false);
    };

    getSession();

    const {
      data: { subscription },
    } = supabase.auth.onAuthStateChange((_event, session) => {
      if (!session?.user) {
        router.push('/');
      } else {
        setUser(session.user);
      }
    });

    return () => subscription.unsubscribe();
  }, [router]);

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (!user) {
    return null;
  }

  return (
    <DashboardLayout>
      <div className="space-y-6">
        {/* Header */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Dashboard</h1>
            <p className="text-gray-600">
              Bienvenido de vuelta, {user.user_metadata?.full_name || user.email}
            </p>
          </div>
          <Button onClick={() => setShowTaskForm(true)} className="flex items-center gap-2">
            <Plus className="h-4 w-4" />
            Nueva Tarea
          </Button>
        </div>
        <div className="space-y-10">
          <section className="rounded-3xl bg-gradient-to-br from-blue-500 via-blue-400 to-emerald-300 px-8 py-10 text-white shadow-lg">
            <div className="max-w-2xl space-y-4">
              <p className="text-sm font-semibold uppercase tracking-[0.5em] text-white/90">
                Sistema de Gestión de Tareas
              </p>
            </div>
          </section>

      </div>
        {/* Stats */}
        <StatsCards />

        {/* Tasks */}
        <div className="bg-white rounded-lg shadow-sm border">
          <div className="p-6 border-b">
            <h2 className="text-lg font-semibold text-gray-900">Mis Tareas</h2>
          </div>
          <TaskList refreshKey={refreshKey} />
        </div>

        {/* Task Form Modal */}
        {showTaskForm && (
          <TaskForm 
            onClose={() => setShowTaskForm(false)}
            onSuccess={() => {
              setShowTaskForm(false);
              setRefreshKey((prev) => prev + 1);
            }}
          />
        )}
      </div>
    </DashboardLayout>
  );
}