"use client";

import { FormEvent, useCallback, useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import { supabase } from "@/lib/supabase";
import { User } from "@supabase/supabase-js";
import { DashboardLayout } from "@/components/dashboard/dashboard-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Textarea } from "@/components/ui/textarea";
import { CalendarIcon, ClipboardList, Loader2, Send, UserRound } from "lucide-react";
import { TaskItem } from "@/components/tasks/task-item";
import { UnifiedTask } from "@/components/tasks/types";

type Team = {
  id: string;
  name: string;
  owner_id: string;
  created_at: string;
};

type TeamMember = {
  id: string;
  member_id: string;
  owner_id: string;
  team_id: string;
  role: string | null;
  status: "pending" | "active" | "removed";
  profile: ProfileSummary | null;
};

type ProfileSummary = {
  id: string;
  full_name: string | null;
  email: string | null;
  avatar_url: string | null;
};

type TeamTask = {
  id: string;
  title: string;
  description: string | null;
  status: "pending" | "in_progress" | "completed";
  priority: "low" | "medium" | "high";
  link_url: string | null;  
  due_date: string | null;
  recurrence_pattern: "daily" | "weekly" | "monthly" | "yearly" | null;
  recurrence_interval: number | null;
  recurrence_end_date: string | null;
  member_id: string;
  assigned_by: string;
  team_id: string | null;  
  completed?: boolean;  
  created_at: string;
  assignee: {
    full_name: string | null;
    avatar_url: string | null;
    email?: string | null;
  } | null;
  assigner: {
    full_name: string | null;
    avatar_url: string | null;
    email?: string | null;
  } | null;
  team: {
    name: string | null;
  } | null;  
};

type TaskTab = "byMe" | "toMe";
type StatusFilter = "all" | "pending" | "in_progress" | "completed";
type DueFilter = "all" | "overdue" | "upcoming";

const statusLabels: Record<TeamTask["status"], { label: string; color: string }> = {
  pending: { label: "Pendiente", color: "bg-amber-100 text-amber-700" },
  in_progress: { label: "En progreso", color: "bg-blue-100 text-blue-700" },
  completed: { label: "Completada", color: "bg-emerald-100 text-emerald-700" },
};

const recurrenceLabels: Record<NonNullable<TeamTask["recurrence_pattern"]>, string> = {
  daily: "Diaria",
  weekly: "Semanal",
  monthly: "Mensual",
  yearly: "Anual",
};

export default function TeamTasksPage() {
  const router = useRouter();
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [teams, setTeams] = useState<Team[]>([]);
  const [selectedTeamId, setSelectedTeamId] = useState("");
  const [members, setMembers] = useState<TeamMember[]>([]);
  const [membersLoading, setMembersLoading] = useState(false);
  const [assignedByMe, setAssignedByMe] = useState<TeamTask[]>([]);
  const [assignedToMe, setAssignedToMe] = useState<TeamTask[]>([]);
  const [taskTitle, setTaskTitle] = useState("");
  const [taskDescription, setTaskDescription] = useState("");
  const [taskPriority, setTaskPriority] = useState<TeamTask["priority"]>("medium");
  const [taskLink, setTaskLink] = useState("");  
  const [taskDueDate, setTaskDueDate] = useState("");
  const [taskRecurrencePattern, setTaskRecurrencePattern] = useState<TeamTask["recurrence_pattern"] | "none">(
    "none"
  );
  const [taskRecurrenceInterval, setTaskRecurrenceInterval] = useState("1");
  const [selectedMember, setSelectedMember] = useState("");
  const [assignLoading, setAssignLoading] = useState(false);
  const [error, setError] = useState("");
  const [activeTab, setActiveTab] = useState<TaskTab>("byMe");
  const [tasksLoading, setTasksLoading] = useState(false);
  const [statusFilter, setStatusFilter] = useState<StatusFilter>("all");
  const [dueFilter, setDueFilter] = useState<DueFilter>("all");

  const getFriendlyErrorMessage = useCallback((supabaseError: { message?: string } | null, fallback: string) => {
    const message = supabaseError?.message?.toLowerCase() ?? "";

    if (message.includes("networkerror") || message.includes("fetch failed")) {
      return "No pudimos conectar con el servidor. Verifica tu conexión e inténtalo de nuevo.";
    }

    return fallback;
  }, []);

  const selectedTeam = useMemo(
    () => teams.find((team) => team.id === selectedTeamId) ?? null,
    [selectedTeamId, teams]
  );

  const memberIds = useMemo(() => members.map((member) => member.member_id), [members]);
  const isMemberOfTeam = useMemo(() => memberIds.includes(user?.id ?? ""), [memberIds, user?.id]);

  const filteredAssignedByMe = useMemo(() => {
    return assignedByMe.filter((task) => {
      const matchesStatus = statusFilter === "all" || task.status === statusFilter;
      const matchesDue =
        dueFilter === "all" ||
        (dueFilter === "overdue" && task.due_date && new Date(task.due_date) < new Date()) ||
        (dueFilter === "upcoming" && task.due_date && new Date(task.due_date) >= new Date());
      return matchesStatus && matchesDue;
    });
  }, [assignedByMe, dueFilter, statusFilter]);

  const filteredAssignedToMe = useMemo(() => {
    return assignedToMe.filter((task) => {
      const matchesStatus = statusFilter === "all" || task.status === statusFilter;
      const matchesDue =
        dueFilter === "all" ||
        (dueFilter === "overdue" && task.due_date && new Date(task.due_date) < new Date()) ||
        (dueFilter === "upcoming" && task.due_date && new Date(task.due_date) >= new Date());
      return matchesStatus && matchesDue;
    });
  }, [assignedToMe, dueFilter, statusFilter]);

  const mapTeamTaskToUnified = useCallback(
    (task: TeamTask): UnifiedTask => ({
      id: task.id,
      title: task.title,
      description: task.description,
      priority: task.priority,      
      link_url: task.link_url ?? null,
      due_date: task.due_date,
      recurrence_pattern: task.recurrence_pattern,
      recurrence_interval: task.recurrence_interval,
      created_at: task.created_at,
      source: "team",
      completed: task.status === "completed",
      status: task.status,
      team_name: task.team?.name ?? selectedTeam?.name ?? null,
      assigned_by_name: task.assigner?.full_name ?? task.assigner?.email ?? null,
      assigned_by_id: task.assigned_by,
      assignee_name: task.assignee?.full_name ?? task.assignee?.email ?? null,
      owner_id: task.member_id,
      parent_task_id: null,
      canEdit: task.assigned_by === user?.id || task.member_id === user?.id,
      canDelete: task.assigned_by === user?.id,
    }),
    [selectedTeam?.name, user?.id]
  );

  const sanitizeTeamUpdates = useCallback(
    (updates: Partial<UnifiedTask>): Partial<TeamTask> => {
      const allowedRecurrence: Array<TeamTask["recurrence_pattern"]> = ["daily", "weekly", "monthly", "yearly"];
      const safeRecurrence = allowedRecurrence.includes(updates.recurrence_pattern as TeamTask["recurrence_pattern"])
        ? (updates.recurrence_pattern as TeamTask["recurrence_pattern"])
        : null;

      return {
        title: updates.title,
        description: updates.description ?? null,
        priority: updates.priority as TeamTask["priority"] | undefined,
        link_url: updates.link_url ?? null,
        due_date: updates.due_date ?? null,
        status: updates.status as TeamTask["status"] | undefined,
        recurrence_pattern: safeRecurrence,
        recurrence_interval:
          typeof updates.recurrence_interval === "number" ? updates.recurrence_interval : null,
        completed: updates.completed,
      };
    },
    []
  );


  const fetchTeams = useCallback(async () => {
    if (!user) return;

    const { data, error: teamsError } = await supabase
      .from("teams")
      .select("*")
      .order("created_at", { ascending: true });

    if (teamsError) {
      console.warn("Error fetching teams", teamsError);
      setError(getFriendlyErrorMessage(teamsError, "No pudimos cargar la lista de equipos disponibles."));
      return;
    }

    const sortedTeams = (data ?? []).sort(
      (first, second) => new Date(first.created_at).getTime() - new Date(second.created_at).getTime()
    );

    setTeams(sortedTeams);

    if (sortedTeams.length > 0 && (!selectedTeamId || !sortedTeams.some((team) => team.id === selectedTeamId))) {
      setSelectedTeamId(sortedTeams[0].id);
    }

    if (sortedTeams.length === 0) {
      setSelectedTeamId("");
      setMembers([]);
      setAssignedByMe([]);
      setAssignedToMe([]);
    }
  }, [getFriendlyErrorMessage, selectedTeamId, user]);

  const fetchTeamMembers = useCallback(async () => {
    if (!user || !selectedTeamId) return;

    setMembersLoading(true);
    try {
      const { data, error: fetchError } = await supabase
        .from("team_members")
        .select("id, member_id, owner_id, team_id, role, status")
        .eq("team_id", selectedTeamId)
        .order("created_at", { ascending: false });

      if (fetchError) {
        console.warn("Error fetching team members", fetchError);
        setError(getFriendlyErrorMessage(fetchError, "No pudimos cargar los integrantes de tu equipo."));
        return;
      }

      const memberIdsForTeam = (data || []).map((member: any) => member.member_id);
      let profileById: Record<string, ProfileSummary> = {};

      if (memberIdsForTeam.length > 0) {
        const { data: profileData, error: profileError } = await supabase
          .from("profiles")
          .select("id, full_name, email, avatar_url")
          .in("id", memberIdsForTeam);

        if (profileError) {
          console.warn("Error fetching member profiles", profileError);
        } else {
          profileById = (profileData || []).reduce((acc: Record<string, ProfileSummary>, profile: any) => {
            acc[profile.id] = profile;
            return acc;
          }, {});
        }
      }

      const formatted: TeamMember[] = (data || []).map((member: any) => ({
        id: member.id,
        member_id: member.member_id,
        owner_id: member.owner_id,
        team_id: member.team_id,
        role: member.role,
        status: member.status,
        profile: profileById[member.member_id] ?? null,
      }));

      setMembers(formatted);
    } catch (cause) {
      console.warn("Error fetching team members", cause);
      setError(
        getFriendlyErrorMessage(
          { message: cause instanceof Error ? cause.message : String(cause) },
          "No pudimos cargar los integrantes de tu equipo."
        )
      );
    } finally {
      setMembersLoading(false);
    }
  }, [getFriendlyErrorMessage, selectedTeamId, user]);

  const fetchTasksForTab = useCallback(
    async (tab: TaskTab) => {
      if (!user || !selectedTeamId || membersLoading) return;

      setError("");
      setTasksLoading(true);

      const memberIdsForTeam = members.map((member) => member.member_id);

      if (memberIdsForTeam.length === 0) {
        setAssignedByMe([]);
        setAssignedToMe([]);
        setTasksLoading(false);
        return;
      }

      if (tab === "byMe") {
        const { data, error: byMeError } = await supabase
          .from("team_tasks")
          .select(
            `
              id,
              title,
              description,
              status,
              priority,
              link_url,                           
              due_date,
              recurrence_pattern,
              recurrence_interval,
              recurrence_end_date,
              member_id,
              assigned_by,
              team_id,              
              created_at,
              assignee:member_id(full_name, avatar_url, email),
              assigner:assigned_by(full_name, avatar_url, email),
              team:team_id(name)
            `
          )
          .eq("assigned_by", user.id)
          .eq("team_id", selectedTeamId)          
          .in("member_id", memberIdsForTeam)
          .order("created_at", { ascending: false })
          .limit(50);

        if (byMeError) {
          console.warn("Error fetching tasks assigned by user", byMeError);
          setError(getFriendlyErrorMessage(byMeError, "No pudimos cargar las tareas asignadas por ti."));
          setTasksLoading(false);
          return;
        }

        const mapTasks = (data: any[] | null) =>
          (data || []).map((task) => {
            const assignee = Array.isArray(task.assignee) ? task.assignee[0] ?? null : task.assignee ?? null;
            const assigner = Array.isArray(task.assigner) ? task.assigner[0] ?? null : task.assigner ?? null;
            const team = Array.isArray(task.team) ? task.team[0] ?? null : task.team ?? null;

            return {
              id: task.id,
              title: task.title,
              description: task.description,
              status: task.status,
              priority: task.priority,
              link_url: task.link_url ?? null,
              due_date: task.due_date,
              recurrence_pattern: task.recurrence_pattern,
              recurrence_interval: task.recurrence_interval,
              recurrence_end_date: task.recurrence_end_date,
              member_id: task.member_id,
              assigned_by: task.assigned_by,
              team_id: task.team_id,
              created_at: task.created_at,
              assignee,
              assigner,
              team,
            };
          });

        setAssignedByMe(mapTasks(data));
      } else {
        if (!memberIdsForTeam.includes(user.id)) {
          setAssignedToMe([]);
          setTasksLoading(false);
          return;
        }

        const { data, error: forMeError } = await supabase
          .from("team_tasks")
          .select(
            `
              id,
              title,
              description,
              status,
              priority,
              link_url,
              due_date,
              recurrence_pattern,
              recurrence_interval,
              recurrence_end_date,
              member_id,
              assigned_by,
              team_id,              
              created_at,
              assignee:member_id(full_name, avatar_url, email),
              assigner:assigned_by(full_name, avatar_url, email),
              team:team_id(name)
            `
          )
          .eq("member_id", user.id)
          .eq("team_id", selectedTeamId)          
          .order("created_at", { ascending: false })
          .limit(50);

        if (forMeError) {
          console.warn("Error fetching tasks assigned to user", forMeError);
          setError(getFriendlyErrorMessage(forMeError, "No pudimos cargar las tareas que te asignaron."));
          setTasksLoading(false);
          return;
        }

        const mapTasks = (data: any[] | null) =>
          (data || []).map((task) => {
            const assignee = Array.isArray(task.assignee) ? task.assignee[0] ?? null : task.assignee ?? null;
            const assigner = Array.isArray(task.assigner) ? task.assigner[0] ?? null : task.assigner ?? null;
            const team = Array.isArray(task.team) ? task.team[0] ?? null : task.team ?? null;

            return {
              id: task.id,
              title: task.title,
              description: task.description,
              status: task.status,
              priority: task.priority,
              link_url: task.link_url ?? null,                           
              due_date: task.due_date,
              recurrence_pattern: task.recurrence_pattern,
              recurrence_interval: task.recurrence_interval,
              recurrence_end_date: task.recurrence_end_date,
              member_id: task.member_id,
              assigned_by: task.assigned_by,
              team_id: task.team_id,              
              created_at: task.created_at,
              assignee,
              assigner,
              team,              
            };
          });

        const tasksForMe = mapTasks(data).filter((task) => memberIdsForTeam.includes(task.member_id));
        setAssignedToMe(tasksForMe);
      }

      setTasksLoading(false);
    },
    [getFriendlyErrorMessage, members, membersLoading, selectedTeamId, user]
  );

  useEffect(() => {
    const getSession = async () => {
      const {
        data: { session },
      } = await supabase.auth.getSession();

      if (!session?.user) {
        router.push("/");
        return;
      }

      setUser(session.user);
      setLoading(false);
    };

    getSession();

    const {
      data: { subscription },
    } = supabase.auth.onAuthStateChange((_event, session) => {
      if (!session?.user) {
        router.push("/");
      } else {
        setUser(session.user);
      }
    });

    return () => subscription.unsubscribe();
  }, [router]);

  useEffect(() => {
    if (!user) return;

    void fetchTeams();
  }, [fetchTeams, user]);

  useEffect(() => {
    if (!user || !selectedTeamId) return;

    setAssignedByMe([]);
    setAssignedToMe([]);
    void fetchTeamMembers();
  }, [fetchTeamMembers, selectedTeamId, user]);

  useEffect(() => {
    if (!selectedTeamId || membersLoading) return;
    void fetchTasksForTab(activeTab);
  }, [activeTab, fetchTasksForTab, membersLoading, selectedTeamId]);

  useEffect(() => {
    setSelectedMember("");
    setTaskPriority("medium");
    setTaskLink("");    
    setTaskRecurrencePattern("none");
    setTaskRecurrenceInterval("1");
  }, [selectedTeamId]);

  const handleAssignTask = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    setError("");

    if (!selectedTeamId) {
      setError("Selecciona un equipo para asignar tareas");
      return;
    }

    if (!selectedMember) {
      setError("Selecciona un miembro para asignar la tarea");
      return;
    }

    if (!user) {
      setError("No se pudo validar tu sesión");
      return;
    }

    if (!memberIds.includes(selectedMember)) {
      setError("Solo puedes asignar tareas a integrantes del equipo seleccionado.");
      return;
    }

    const recurrencePattern = taskRecurrencePattern === "none" ? null : taskRecurrencePattern;
    const recurrenceInterval = recurrencePattern ? Math.max(1, Number(taskRecurrenceInterval) || 1) : null;

    if (recurrencePattern && !taskDueDate) {
      setError("Agrega una fecha límite para las tareas recurrentes");
      return;
    }

    setAssignLoading(true);

    const { error: insertError } = await supabase.from("team_tasks").insert({
      member_id: selectedMember,
      assigned_by: user.id,
      title: taskTitle,
      description: taskDescription,
      due_date: taskDueDate || null,
      link_url: taskLink || null,      
      status: "pending",
      priority: taskPriority,
      recurrence_pattern: recurrencePattern,
      recurrence_interval: recurrenceInterval,
      team_id: selectedTeamId,      
    });

    if (insertError) {
      console.error("Error creating assignment", insertError);
      setError("No pudimos asignar la tarea. Inténtalo nuevamente.");
      setAssignLoading(false);
      return;
    }

    setTaskTitle("");
    setTaskDescription("");
    setTaskPriority("medium");
    setTaskLink("");       
    setTaskDueDate("");
    setTaskRecurrencePattern("none");
    setTaskRecurrenceInterval("1");
    setSelectedMember("");
    await fetchTasksForTab("byMe");
    setAssignLoading(false);
  };

  const handleTeamTaskUpdate = (taskId: string, updates: Partial<UnifiedTask>) => {
    const filteredUpdates = sanitizeTeamUpdates(updates);
    setAssignedByMe((prev) => prev.map((task) => (task.id === taskId ? { ...task, ...filteredUpdates } : task)));
    setAssignedToMe((prev) => prev.map((task) => (task.id === taskId ? { ...task, ...filteredUpdates } : task)));
  };

  const handleTeamTaskDelete = (taskId: string) => {
    setAssignedByMe((prev) => prev.filter((task) => task.id !== taskId));
    setAssignedToMe((prev) => prev.filter((task) => task.id !== taskId));
  };

  if (loading) {
    return (
      <DashboardLayout>
        <div className="space-y-6">
          <div className="flex flex-col gap-2">
            <h1 className="text-2xl font-bold text-gray-900">Tareas de equipo</h1>
            <p className="text-gray-600">Cargando información...</p>
          </div>
        </div>
      </DashboardLayout>
    );
  }

  if (!user) return null;

  return (
    <DashboardLayout>
      <div className="space-y-6">
        <div className="flex flex-col gap-2">
          <h1 className="text-2xl font-bold text-gray-900">Gestión de tareas por equipo</h1>
          <p className="text-gray-600">
            Consulta y asigna tareas únicamente para el equipo seleccionado. Los datos se cargan bajo demanda por
            pestaña.
          </p>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          <Card className="lg:col-span-2">
            <CardHeader>
              <CardTitle>Selecciona un equipo</CardTitle>
            </CardHeader>
            <CardContent className="flex flex-col gap-4">
              <Select value={selectedTeamId} onValueChange={setSelectedTeamId}>
                <SelectTrigger className="w-full lg:w-80">
                  <SelectValue placeholder="Equipo" />
                </SelectTrigger>
                <SelectContent>
                  {teams.map((team) => (
                    <SelectItem key={team.id} value={team.id}>
                      {team.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              {selectedTeam && (
                <p className="text-sm text-gray-600">
                  Solo verás tareas de los integrantes de <span className="font-semibold">{selectedTeam.name}</span>.
                </p>
              )}
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Filtros rápidos</CardTitle>
            </CardHeader>
            <CardContent className="grid grid-cols-1 gap-4">
              <div className="space-y-2">
                <Label>Filtrar por estado</Label>
                <Select value={statusFilter} onValueChange={(value: StatusFilter) => setStatusFilter(value)}>
                  <SelectTrigger>
                    <SelectValue placeholder="Estado" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    <SelectItem value="pending">Pendiente</SelectItem>
                    <SelectItem value="in_progress">En progreso</SelectItem>
                    <SelectItem value="completed">Completada</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Vencimiento</Label>
                <Select value={dueFilter} onValueChange={(value: DueFilter) => setDueFilter(value)}>
                  <SelectTrigger>
                    <SelectValue placeholder="Vencimiento" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    <SelectItem value="overdue">Vencidas</SelectItem>
                    <SelectItem value="upcoming">Próximas</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </CardContent>
          </Card>
        </div>

        {error && (
          <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
            {error}
          </div>
        )}

        <div className="grid grid-cols-1 xl:grid-cols-3 gap-6">
          <Card className="xl:col-span-1">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <ClipboardList className="h-5 w-5 text-blue-600" />
                Asignar nueva tarea
              </CardTitle>
            </CardHeader>
            <CardContent>
              <form className="space-y-4" onSubmit={handleAssignTask}>
                <div className="space-y-2">
                  <Label>Integrante del equipo</Label>
                  <Select value={selectedMember} onValueChange={setSelectedMember}>
                    <SelectTrigger>
                      <SelectValue placeholder="Selecciona un miembro" />
                    </SelectTrigger>
                    <SelectContent>
                      {members.length === 0 && (
                        <SelectItem value="no-members" disabled>
                          No hay integrantes en este equipo
                        </SelectItem>
                      )}
                      {members.map((member) => (
                        <SelectItem key={member.id} value={member.member_id}>
                          {member.profile?.full_name || member.profile?.email || "Integrante"}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div className="space-y-2">
                  <Label>Título de la tarea</Label>
                  <Input
                    value={taskTitle}
                    onChange={(event) => setTaskTitle(event.target.value)}
                    placeholder="Revisar brief, preparar demo..."
                    required
                  />
                </div>

                <div className="space-y-2">
                  <Label>Descripción</Label>
                  <Textarea
                    value={taskDescription}
                    onChange={(event) => setTaskDescription(event.target.value)}
                    placeholder="Agrega contexto o pasos necesarios para completar la tarea"
                  />
                </div>

                <div className="space-y-2">
                  <Label>Enlace</Label>
                  <Input
                    type="url"
                    value={taskLink}
                    onChange={(event) => setTaskLink(event.target.value)}
                    placeholder="https://..."
                  />
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label>Prioridad</Label>
                    <Select value={taskPriority} onValueChange={(value: TeamTask["priority"]) => setTaskPriority(value)}>
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="low">Baja</SelectItem>
                        <SelectItem value="medium">Media</SelectItem>
                        <SelectItem value="high">Alta</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                </div>

                <div className="space-y-2">                    
                    <Label>Fecha límite</Label>
                    <div className="relative">
                      <Input
                        type="date"
                        value={taskDueDate}
                        onChange={(event) => setTaskDueDate(event.target.value)}
                        className="pl-10"
                      />
                      <CalendarIcon className="h-4 w-4 absolute left-3 top-3 text-gray-500" />
                    </div>
                  </div>

                <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label>Recurrencia</Label>
                    <Select
                      value={taskRecurrencePattern ?? "none"}
                      onValueChange={(value) => setTaskRecurrencePattern(value as TeamTask["recurrence_pattern"] | "none")}
                    >
                      <SelectTrigger>
                        <SelectValue placeholder="Selecciona una recurrencia" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="none">Sin recurrencia</SelectItem>
                        <SelectItem value="daily">Diaria</SelectItem>
                        <SelectItem value="weekly">Semanal</SelectItem>
                        <SelectItem value="monthly">Mensual</SelectItem>
                        <SelectItem value="yearly">Anual</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div className="space-y-2">
                    <Label>Intervalo</Label>
                    <Input
                      type="number"
                      min={1}
                      disabled={taskRecurrencePattern === "none"}
                      value={taskRecurrenceInterval}
                      onChange={(event) => setTaskRecurrenceInterval(event.target.value)}
                    />
                    <p className="text-xs text-gray-500">
                      Configura cada cuántos días, semanas, meses o años se repetirá la tarea.
                    </p>
                  </div>
                </div>

                <div className="flex justify-end">
                  <Button type="submit" disabled={assignLoading || !selectedTeamId} className="flex items-center gap-2">
                    {assignLoading ? <Loader2 className="h-4 w-4 animate-spin" /> : <Send className="h-4 w-4" />}
                    {assignLoading ? "Asignando..." : "Asignar tarea"}
                  </Button>
                </div>
                {!selectedTeamId && (
                  <p className="text-xs text-amber-600">Selecciona un equipo antes de crear una tarea.</p>
                )}
              </form>
            </CardContent>
          </Card>

          <Card className="xl:col-span-2">
            <CardHeader>
              <div className="flex items-center justify-between gap-2 flex-wrap">
                <CardTitle className="flex items-center gap-2">
                  <UserRound className="h-5 w-5 text-blue-600" />
                  Tareas del equipo
                </CardTitle>
                <div className="flex items-center gap-2">
                  <Button variant={activeTab === "byMe" ? "default" : "outline"} size="sm" onClick={() => setActiveTab("byMe")}>
                    Asignadas por mí
                  </Button>
                  <Button variant={activeTab === "toMe" ? "default" : "outline"} size="sm" onClick={() => setActiveTab("toMe")}>
                    Asignadas a mí
                  </Button>
                  <Button variant="outline" size="sm" onClick={() => fetchTasksForTab(activeTab)} disabled={tasksLoading}>
                    Actualizar
                  </Button>
                </div>
              </div>
            </CardHeader>
            <CardContent className="space-y-3">
              {tasksLoading && (
                <div className="flex items-center gap-2 text-gray-600">
                  <Loader2 className="h-4 w-4 animate-spin" />
                  <span>Cargando tareas...</span>
                </div>
              )}

              {activeTab === "byMe" && filteredAssignedByMe.length === 0 && !tasksLoading && (
                <p className="text-gray-600">Aún no has asignado tareas a tu equipo seleccionado.</p>
              )}

              {activeTab === "toMe" && !isMemberOfTeam && (
                <p className="text-amber-700 text-sm">
                  No eres integrante de este equipo, por lo que no verás tareas asignadas a ti aquí.
                </p>
              )}

              {activeTab === "toMe" && filteredAssignedToMe.length === 0 && isMemberOfTeam && !tasksLoading && (
                <p className="text-gray-600">No tienes tareas asignadas en este equipo.</p>
              )}

              {(activeTab === "byMe" ? filteredAssignedByMe : filteredAssignedToMe).map((task) => (
                <TaskItem
                  key={task.id}
                  task={mapTeamTaskToUnified(task)}
                  onUpdate={handleTeamTaskUpdate}
                  onDelete={handleTeamTaskDelete}
                  onCreate={() => {}}
                />
              ))}
            </CardContent>
          </Card>
        </div>
      </div>
    </DashboardLayout>
  );
}