"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import {
  CalendarDays,
  CheckCircle2,
  ClipboardList,
  ListChecks,
  PieChart,
  Puzzle,
  UserCircle2,
  XCircle,
} from "lucide-react";
import { toast } from "sonner";
import type { User } from "@supabase/supabase-js";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { supabase } from "@/lib/supabase";

const responseOptions = [
  "Cumple correctamente",
  "Cumple, pero puede mejorar",
  "Cumple, pero hay riesgo futuro",
  "Cumple de forma destacada",
  "No cumple parcialmente",
  "No cumple totalmente o hay riesgo grave",
];

const pagePath = "/meiwa-core/meiwa-sgi/evaluation/internal-audit";
const allowedManagers = new Set(["ehs.mexico@meiwa.com.mx", "auxiliar.ehs@meiwa.com.mx"]);

const sections = [
  {
    id: "contexto",
    title: "4. Contexto",
    description: "Comprensión de la organización y del sistema de gestión ambiental.",
    groups: [
      {
        title: "4.1. Comprensión de la organización y de su contexto",
        questions: [
          "¿Dispone la organización de una metodología para el análisis, seguimiento y revisión del contexto interno y externo?",
          "¿Ha detectado la organización todas las condiciones ambientales capaces de afectar o de verse afectadas por la organización?",
          "¿Se han identificado requisitos legales y otros requisitos en relación con las necesidades y expectativas de las partes interesadas?",
        ],
      },
      {
        title: "4.3. Determinación del alcance del sistema de gestión ambiental",
        questions: [
          "¿Tiene documentado la organización el alcance del sistema de gestión?",
          "¿Se han delimitado claramente los límites físicos y las actividades del sistema?",
          "¿El alcance del sistema se encuentra disponible para las partes interesadas?",
        ],
      },
      {
        title: "4.4. Sistema de gestión ambiental",
        questions: [
          "¿Se han identificado todos los procesos necesarios y sus interacciones, incluyendo entradas, salidas y secuencia?",
          "¿Se han definido actividades de seguimiento e indicadores para el control de estos procesos?",
          "¿Se han identificado los recursos necesarios y las responsabilidades y autoridades de cada proceso?",
          "¿Se han definido los procesos teniendo en cuenta los riesgos y oportunidades?",
          "¿Se evidencia la mejora en el desempeño de los procesos y el sistema de gestión ambiental?",
        ],
      },
    ],
  },
  {
    id: "liderazgo",
    title: "5. Liderazgo",
    description: "Compromiso directivo, política ambiental y roles internos.",
    groups: [
      {
        title: "5.1. Liderazgo y compromiso",
        questions: [
          "¿Demuestra la dirección el liderazgo y compromiso respecto al sistema de gestión ambiental?",
          "¿Asume la alta dirección la responsabilidad y obligación de rendir cuentas en relación a la eficacia del sistema de gestión ambiental?",
          "¿Se asegura y promueva la dirección el cumplimiento de los requisitos del sistema de gestión?",
          "¿Se asegura la alta dirección del establecimiento de la política y los objetivos ambientales, de una forma compatible con la dirección estratégica y el contexto de la organización?",
        ],
      },
      {
        title: "5.2. Política ambiental",
        questions: [
          "¿Mantiene la organización una política ambiental apropiada al propósito y contexto de la organización, teniendo en cuenta los impactos ambientales de sus actividades, productos y servicios?",
          "¿Incluye la política los compromisos de protección del medio ambiente, incluida la prevención de la contaminación, de cumplimiento de requisitos legales y otros requisitos y de la mejora continua?",
          "¿Existe una relación entre la política y los objetivos ambientales?",
          "¿La política se encuentra disponible para las partes interesadas?",
          "¿La política es comunicada y entendida dentro de la organización?",
        ],
      },
      {
        title: "5.3. Roles, responsabilidades y autoridades en la organización",
        questions: [
          "¿Existe evidencia de la definición de responsabilidades y autoridades para cada uno de los roles de la organización?",
          "¿Estas responsabilidades y autoridades han sido comunicadas y entendidas en toda la organización?",
          "¿Ha asignado la alta dirección la responsabilidad para el aseguramiento del cumplimiento de los requisitos de la norma, el correcto funcionamiento de los procesos, etc.?",
        ],
      },
    ],
  },
  {
    id: "planeacion",
    title: "6. Planeación",
    description: "Riesgos, oportunidades y objetivos ambientales.",
    groups: [
      {
        title: "6.1. Acciones para abordar riesgos y oportunidades",
        questions: [],
        subsections: [
          {
            title: "6.1.1. Generalidades",
            questions: [
              "¿Se han identificado los riesgos y oportunidades relacionados con el análisis de contexto, las necesidades y expectativas de las partes interesadas y los procesos?",
              "¿Se han evaluado estos riesgos y oportunidades para determinar acciones proporcionales al impacto potencial?",
              "¿Se han planificado acciones para abordar los riesgos y las oportunidades?",
            ],
          },
          {
            title: "6.1.2. Aspectos ambientales",
            questions: [
              "¿Se han identificado todos los aspectos ambientales de las actividades, productos y servicios que puede controlar y de aquellos en los que puede influir, y sus impactos ambientales asociados, desde una perspectiva de ciclo de vida?",
              "¿Se han considerado las condiciones anormales y las situaciones de emergencia?",
              "¿Se ha realizado una evaluación de la significancia de los aspectos ambientales identificados?",
              "¿Se encuentran establecidos los criterios para la evaluación de aspectos ambientales?",
            ],
          },
          {
            title: "6.1.3. Requisitos legales y otros requisitos",
            questions: [
              "¿Se ha determinado la metodología de acceso a los requisitos legales y otros requisitos aplicables?",
              "¿Se ha identificado la aplicación de los requisitos legales en la organización?",
              "¿Se han tenido en cuenta los requisitos legales y otros requisitos en la implementación, mantenimiento y mejora del sistema de gestión ambiental?",
            ],
          },
          {
            title: "6.1.4. Planeación de acciones",
            questions: [
              "¿Se han planificado las acciones emprendidas relativas a los aspectos ambientales significativos, los requisitos legales y los riesgos y oportunidades identificados?",
            ],
          },
        ],
      },
      {
        title: "6.2. Objetivos ambientales y planeación para lograrlos",
        questions: [
          "¿Se han establecido objetivos coherentes con la política ambiental?",
          "¿Los objetivos están relacionados con los aspectos ambientales significativos, los requisitos legales y los riesgos y oportunidades?",
          "¿Los objetivos son medibles y disponen de metodología de seguimiento?",
          "¿La planeación de los objetivos contempla las actividades, los recursos, los plazos y las responsabilidades para su realización?",
          "¿Se han comunicado los objetivos en la organización en los niveles pertinentes?",
        ],
      },
    ],
  },
  {
    id: "soporte",
    title: "7. Soporte",
    description: "Recursos, competencia, comunicación e información documentada.",
    groups: [
      {
        title: "7.1. Recursos",
        questions: [
          "¿La organización dispone de los recursos necesarios para el correcto desempeño del sistema de gestión ambiental?",
          "¿La organización ha determinado y proporcionado las personas necesarias para la implementación eficaz del sistema de gestión ambiental?",
        ],
      },
      {
        title: "7.2. Competencia",
        questions: [
          "¿Se han determinado las competencias necesarias de las personas para realizar las tareas del sistema de gestión ambiental?",
          "¿Se han emprendido acciones para asegurar o mejorar la competencia del personal de la organización?",
          "¿Existen evidencias documentadas de la competencia necesaria?",
        ],
      },
      {
        title: "7.3. Toma de conciencia",
        questions: [
          "¿Se han realizado acciones para asegurar que las personas tomen conciencia de la política y los objetivos ambientales?",
          "¿Se ha comunicado su contribución a la eficacia del sistema y los beneficios de una mejora del desempeño?",
          "¿Se han realizado acciones para que las personas tomen conciencia de las consecuencias de incumplir los requisitos del sistema de gestión ambiental?",
        ],
      },
      {
        title: "7.4. Comunicación",
        questions: [
          "¿Se han determinado las comunicaciones internas y externas pertinentes al sistema de gestión ambiental?",
          "¿Se encuentra definido qué, cuándo, a quién, cómo y quién realiza cada comunicación?",
          "¿Existe una metodología para la comunicación tanto interna como externa?",
          "¿Responde la organización a las comunicaciones externas pertinentes?",
        ],
      },
      {
        title: "7.5. Información documentada",
        questions: [
          "¿Se ha identificado la documentación requerida por la norma y el propio sistema de gestión?",
          "¿La identificación y descripción de los documentos es apropiada?",
          "¿Se encuentra definido el formato y soporte de cada documento?",
          "¿Existe una metodología de revisión y aprobación adecuada?",
          "¿La documentación está disponible en los puntos de uso para su consulta?",
        ],
      },
      {
        title: "7.6. Información documentada (continuación)",
        questions: [
          "¿La documentación está protegida adecuadamente contra pérdida o uso inadecuado?",
          "¿Se han definido metodologías para la distribución, acceso, recuperación y uso de los documentos?",
          "¿Se contemplan actividades para el almacenamiento y preservación de los documentos (copias de seguridad)?",
          "¿Existe un control de cambios en los documentos del sistema?",
          "¿Se ha identificado la documentación de origen externo necesaria para el desempeño de los procesos?",
        ],
      },
    ],
  },
  {
    id: "operacion",
    title: "8. Operación",
    description: "Planeación, control operacional y respuesta ante emergencias.",
    groups: [
      {
        title: "8.1. Planeación y Control Operacional",
        questions: [
          "¿Se han identificado los procesos necesarios para cumplir los requisitos del sistema de gestión ambiental?",
          "¿Se han establecido criterios para la operación de los procesos?",
          "¿Se controlan los procesos contratados externamente?",
          "¿Se han definido los requisitos ambientales para la compra de productos y servicios?",
          "¿Se han comunicado los requisitos ambientales pertinentes a los proveedores externos?",
          "¿Se evalúa a sus proveedores y esta evaluación considera condiciones ambientales?",
        ],
        subsections: [
          {
            title: "Procesos de operaciones · Agua",
            questions: ["Registro de consumo mensual actualizado y factura."],
          },
          {
            title: "Procesos de operaciones · Energía",
            questions: ["Registro de balance energético mensual actualizado y su factura."],
          },
          {
            title: "Procesos de operaciones · Combustibles – Gasolina",
            questions: ["Registro de consumo mensual actualizado."],
          },
          {
            title: "Procesos de operaciones · Combustibles – Gases industriales",
            questions: ["Registro de consumo mensual actualizado."],
          },
          {
            title: "Procesos de operaciones · Residuos Sólidos Urbanos (RSU)",
            questions: [
              "Registro y plan de manejo: verificar existencia del documento, su enlace actualizado y control de cambios del registro principal.",
              "Permisos / proveedores: validar permisos vigentes y la evidencia asociada (PDF).",
              "Bitácora de ingreso: revisar residuo, fecha y departamento para trazabilidad de generación interna.",
              "Bitácora de salida: verificar cantidad, registro ambiental, transportista, autorizaciones y destino final para cumplimiento regulatorio.",
              "Facturas y manifiestos: corroborar que existan archivos cargados y correspondan con las salidas registradas.",
            ],
          },
          {
            title: "Procesos de operaciones · Residuos de Manejo Especial (RME)",
            questions: [
              "Registro y plan de manejo: documento y enlace vigente.",
              "Permisos / proveedores: permisos vigentes y evidencia PDF asociada.",
              "Bitácora de ingreso: residuo, fecha, cantidad (kg), código de peligrosidad, área/proceso y responsable.",
              "Bitácora de salida: cantidad, código de peligrosidad, número de manifiesto, registro ambiental, transportista y destino final.",
              "Facturas y manifiestos: existencia y trazabilidad documental.",
            ],
          },
          {
            title: "Procesos de operaciones · Residuos Peligrosos (RP)",
            questions: [
              "Registro y plan de manejo: documento y enlace actualizado.",
              "Permisos / proveedores: permisos vigentes y PDF de soporte.",
              "Bitácora de ingreso: residuo, fecha, cantidad (kg), código de peligrosidad, área/proceso y responsable.",
              "Bitácora de salida: cantidad, código de peligrosidad, manifiesto, registro ambiental, transportista, autorizaciones y destino final.",
              "Facturas y manifiestos: validar que los archivos existan y correspondan con la bitácora de salida.",
            ],
          },
          {
            title: "Procesos de operaciones · Servicios externalizados",
            questions: [
              "Registro de proveedores actualizado.",
              "Estado de evaluación: confirmar última evaluación, próxima evaluación y estatus (al día / vencida / pendiente).",
            ],
          },
          {
            title: "Procesos de operaciones · Calidad",
            questions: [
              "Registro de stock y scrap actualizado.",
              "Registro y seguimiento de QIR’s actualizado.",
            ],
          },
        ],
      },
      {
        title: "8.2. Preparación y Respuesta Ante Emergencias",
        questions: [
          "¿Se han establecido los procesos necesarios para prepararse y responder a situaciones potenciales de emergencia?",
          "¿Se dispone de medidas planificadas para la prevención y mitigación de los impactos ambientales adversos provocados por situaciones de emergencia?",
          "¿Se tienen previstas las medidas para la respuesta ante situaciones de emergencia?",
          "¿Existe una planificación de pruebas para las acciones de respuesta previstas?",
          "¿Se tiene en cuenta la comunicación con partes interesadas, cuando sea oportuno?",
        ],
      },
    ],
  },
  {
    id: "evaluacion",
    title: "9. Evaluación",
    description: "Seguimiento, auditorías internas y revisión por la dirección.",
    groups: [
      {
        title: "9.1. Seguimiento, medición, análisis y evaluación",
        questions: [],
        subsections: [
          {
            title: "9.1.2. Evaluación del cumplimiento",
            questions: [
              "¿La organización evalúa el desempeño y la eficacia del sistema de gestión ambiental?",
              "¿Se han identificado procesos y aspectos con necesidades de seguimiento y medición?",
              "¿Los equipos de medición se encuentran calibrados o verificados de manera adecuada?",
              "¿Se comunica externa e internamente la información pertinente a su desempeño ambiental?",
              "¿Existe una metodología y planificación para realizar la evaluación de cumplimiento de los requisitos legales y otros requisitos?",
            ],
          },
        ],
      },
      {
        title: "9.2. Auditoría Interna",
        questions: [
          "¿Las auditorías internas se realizan de forma planificada?",
          "¿Se garantiza la competencia e independencia de los auditores internos?",
          "¿El alcance de la auditoría y los métodos son apropiados para evaluar la eficacia del sistema de gestión ambiental?",
          "¿La dirección pertinente es informada de los resultados de auditoría?",
          "¿Se emprenden acciones para solventar los incumplimientos detectados en las auditorías internas?",
        ],
      },
      {
        title: "9.3. Revisión por la Dirección",
        questions: [
          "¿Se han incluido todas las entradas de la revisión presentes en la norma de referencia?",
          "¿Se han tratado todas las salidas necesarias requeridas por la norma de referencia?",
          "¿Existe una metodología definida y una planificación para la realización de las revisiones por la dirección?",
          "¿Se está empleando la revisión por la dirección como una herramienta de mejora del sistema de gestión ambiental?",
        ],
      },
    ],
  },
  {
    id: "mejora",
    title: "10. Mejora",
    description: "Acciones correctivas y mejora continua.",
    groups: [
      {
        title: "10.1. Generalidades",
        questions: [
          "¿La organización planifica acciones para la mejora de su desempeño ambiental y del desempeño del sistema de gestión ambiental?",
          "¿Se contemplan para la mejora las necesidades y expectativas de las partes interesadas?",
          "¿Se contemplan los riesgos y oportunidades para emprender acciones para la mejora?",
        ],
      },
      {
        title: "10.2. No Conformidad y Acción Correctiva",
        questions: [
          "¿Existe una metodología para el tratamiento de las no conformidades?",
          "¿Se está realizando análisis de las causas de las no conformidades para emprender acciones correctivas?",
          "¿Existe análisis de la repetitividad de las no conformidades para emprender acciones correctivas?",
          "¿La documentación de las no conformidades y acciones correctivas es adecuada para conocer las causas, responsabilidades, resultados y análisis de la eficacia?",
        ],
      },
      {
        title: "10.3. Mejora Continua",
        questions: [
          "¿La organización dispone de las herramientas adecuadas para favorecer la mejora continua (objetivos, acciones, salidas de la revisión, etc.)?",
          "¿Existen evidencias de estas mejoras planificadas por la organización?",
          "¿Las mejoras a emprender tienen en cuenta las necesidades y expectativas de las partes interesadas, el análisis de contexto y los riesgos y oportunidades?",
        ],
      },
    ],
  },
];

const sectionOptions = sections.map((section) => ({
  value: section.id,
  label: section.title.toUpperCase(),
}));

const classificationByResponse: Record<string, string> = {
  "Cumple correctamente": "Conformidad",
  "Cumple, pero puede mejorar": "Oportunidad de mejora",
  "Cumple, pero hay riesgo futuro": "Observación",
  "Cumple de forma destacada": "Buena práctica",
  "No cumple parcialmente": "No conformidad menor",
  "No cumple totalmente o hay riesgo grave": "No conformidad mayor",
};

const classificationDetailLabels = [
  { key: "Conformidad", label: "Conformidades" },
  { key: "Oportunidad de mejora", label: "Oportunidades de mejora" },
  { key: "Observación", label: "Observaciones" },
  { key: "Buena práctica", label: "Buenas prácticas" },
  { key: "No conformidad menor", label: "No conformidades menores" },
  { key: "No conformidad mayor", label: "No conformidades mayores" },
];

const nonComplianceResponses = new Set([
  "No cumple parcialmente",
  "No cumple totalmente o hay riesgo grave",
]);

type AuditResponseRecord = {
  section_id: string;
  group_title: string;
  question: string;
  response: string;
};

type AuditSessionSummary = {
  id: string;
  audit_date: string;
  auditors: string[] | null;
};

export default function InternalAuditPage() {
  const [user, setUser] = useState<User | null>(null);
  const [selectedSection, setSelectedSection] = useState(sections[0].id);
  const [now, setNow] = useState(() => new Date());
  const [statsLastAuditDate, setStatsLastAuditDate] = useState<Date | null>(null);
  const [auditors, setAuditors] = useState<string[]>(["", "", "", ""]);
  const [responses, setResponses] = useState<Record<string, string>>({});
  const [comments, setComments] = useState<Record<string, string>>({});
  const [statsResponses, setStatsResponses] = useState<AuditResponseRecord[]>([]);  
  const [activeTab, setActiveTab] = useState<"stats" | "audit">("audit");
  const [statsFilter, setStatsFilter] = useState("all");
  const [isSaving, setIsSaving] = useState(false);
  const [canManageAudits, setCanManageAudits] = useState(false);
  const [auditHistory, setAuditHistory] = useState<AuditSessionSummary[]>([]);
  const [deletingSessionId, setDeletingSessionId] = useState<string | null>(null);

  const buildQuestionKey = (
    sectionId: string,
    groupTitle: string,
    question: string
  ) => `${sectionId}::${groupTitle}::${question}`;

  const loadLatestAudit = async (currentUser: User) => {
    const { data: sessionData, error } = await supabase
      .from("internal_audit_sessions")
      .select("*")
      .eq("user_id", currentUser.id)
      .order("audit_date", { ascending: false })
      .limit(1)
      .maybeSingle();

    if (error) {
      toast.error("No pudimos cargar la última auditoría.");
      return;
    }

    if (!sessionData) {
      setAuditors(["", "", "", ""]);
      setResponses({});
      setComments({});
      return;
    }

    const { data: responseData, error: responsesError } = await supabase
      .from("internal_audit_responses")
      .select("*")
      .eq("session_id", sessionData.id);

    if (responsesError) {
      toast.error("No pudimos cargar las respuestas de la auditoría.");
      return;
    }

    const nextResponses: Record<string, string> = {};
    const nextComments: Record<string, string> = {};

    (responseData ?? []).forEach((response) => {
      const questionKey = buildQuestionKey(
        response.section_id,
        response.group_title,
        response.question
      );
      nextResponses[questionKey] = response.response;
      if (response.comment) {
        nextComments[questionKey] = response.comment;
      }
    });

    setResponses(nextResponses);
    setComments(nextComments);
    setAuditors([
      ...(sessionData.auditors ?? []),
      "",
      "",
      "",
      "",
    ].slice(0, 4));
  };

  const loadStatsAuditData = async () => {
    const { data: sessionData, error } = await supabase
      .from("internal_audit_sessions")
      .select("audit_date")
      .order("audit_date", { ascending: false })
      .limit(1)
      .maybeSingle();

    if (error) {
      toast.error("No pudimos cargar el historial de auditorías.");
      return;
    }

    setStatsLastAuditDate(sessionData?.audit_date ? new Date(sessionData.audit_date) : null);

    const { data: responseData, error: responsesError } = await supabase
      .from("internal_audit_responses")
      .select("section_id, group_title, question, response");

    if (responsesError) {
      toast.error("No pudimos cargar las estadísticas globales.");
      return;
    }

    setStatsResponses(responseData ?? []);
  };

  const loadAuditHistory = async () => {
    const { data, error } = await supabase
      .from("internal_audit_sessions")
      .select("id, audit_date, auditors")
      .order("audit_date", { ascending: false });

    if (error) {
      toast.error("No pudimos cargar el historial de auditorías.");
      return;
    }

    setAuditHistory(data ?? []);
  };

  const resolveAuditPermissions = useCallback(async (currentUser: User | null) => {
    if (!currentUser) {
      setCanManageAudits(false);
      return false;
    }
    if (allowedManagers.has(currentUser.email ?? "")) {
      setCanManageAudits(true);
      return true;
    }

    const { data: pageData, error: pageError } = await supabase
      .from("sgi_pages")
      .select("id")
      .eq("path", pagePath)
      .maybeSingle();

    if (pageError) {
      console.error(pageError);
      setCanManageAudits(false);
      return false;
    }

    if (!pageData?.id) {
      setCanManageAudits(true);
      return true;
    }

    const { data: permissionData, error: permissionError } = await supabase
      .from("sgi_admin_permissions")
      .select("id")
      .eq("page_id", pageData.id)
      .eq("user_id", currentUser.id)
      .maybeSingle();

    if (permissionError) {
      console.error(permissionError);
      setCanManageAudits(false);
      return false;
    }

    const canManage = Boolean(permissionData?.id);
    setCanManageAudits(canManage);
    return canManage;
  }, []);

  useEffect(() => {
    const interval = setInterval(() => setNow(new Date()), 60000);
    return () => clearInterval(interval);
  }, []);

  useEffect(() => {
    const loadSession = async () => {
      const { data } = await supabase.auth.getSession();
      const currentUser = data.session?.user ?? null;
      setUser(currentUser);
      if (currentUser) {
        await loadLatestAudit(currentUser);
        const canManage = await resolveAuditPermissions(currentUser);
        if (canManage) {
          await loadAuditHistory();
        } else {
          setAuditHistory([]);
        }
      } else {
        setCanManageAudits(false);
        setAuditHistory([]);
      }
      await loadStatsAuditData();
    };

    loadSession();

    const {
      data: { subscription },
    } = supabase.auth.onAuthStateChange((_event, session) => {
      const nextUser = session?.user ?? null;
      setUser(nextUser);
      if (nextUser) {
        loadLatestAudit(nextUser);
        resolveAuditPermissions(nextUser).then((canManage) => {
          if (canManage) {
            loadAuditHistory();
          } else {
            setAuditHistory([]);
          }
        });
      }
      loadStatsAuditData();
      if (!nextUser) {
        setCanManageAudits(false);
        setAuditHistory([]);
        setAuditors(["", "", "", ""]);
        setResponses({});
        setComments({});
      }
    });

    return () => subscription.unsubscribe();
  }, [resolveAuditPermissions]);

  const formattedNow = useMemo(
    () =>
      new Intl.DateTimeFormat("es-MX", {
        dateStyle: "full",
        timeStyle: "short",
      }).format(now),
    [now]
  );
  const formattedStatsAuditDate = useMemo(
    () =>
      new Intl.DateTimeFormat("es-MX", {
        dateStyle: "short",
      }).format(statsLastAuditDate ?? now),
    [statsLastAuditDate, now]
  );
  const formatAuditDate = (auditDate: string) =>
    new Intl.DateTimeFormat("es-MX", {
      dateStyle: "medium",
      timeStyle: "short",
    }).format(new Date(auditDate));

  const handleContinue = () => {
    const element = document.getElementById(selectedSection);
    element?.scrollIntoView({ behavior: "smooth", block: "start" });
  };

  const activeSection = sections.find((section) => section.id === selectedSection) ?? sections[0];

  const statsBySection = useMemo(() => {
    return sections.map((section) => {
      const sectionResponses = statsResponses.filter(
        (response) => response.section_id === section.id
      );
      const counts = Object.fromEntries(
        classificationDetailLabels.map((item) => [item.key, 0])
      ) as Record<string, number>;
      let hasNonCompliance = false;
      const totalResponses = sectionResponses.length;

      sectionResponses.forEach((response) => {
        if (!response.response) return;
        if (nonComplianceResponses.has(response.response)) {
          hasNonCompliance = true;
        }
        const classification = classificationByResponse[response.response];
        if (classification) {
          counts[classification] = (counts[classification] ?? 0) + 1;
        }
      });

      const status =
        totalResponses === 0 ? "Sin datos" : hasNonCompliance ? "No cumple" : "Cumple";

      return {
        id: section.id,
        title: section.title,
        status,
        totalResponses,
        counts,
      };
    });
  }, [statsResponses]);

  const globalSummary = useMemo(() => {
    const selectedSections =
      statsFilter === "all"
        ? statsBySection
        : statsBySection.filter((section) => section.id === statsFilter);

    const counts = Object.fromEntries(
      classificationDetailLabels.map((item) => [item.key, 0])
    ) as Record<string, number>;
    let hasNonCompliance = false;
    let totalResponses = 0;

    selectedSections.forEach((section) => {
      totalResponses += section.totalResponses;
      if (section.status === "No cumple") {
        hasNonCompliance = true;
      }
      classificationDetailLabels.forEach((item) => {
        counts[item.key] += section.counts[item.key] ?? 0;
      });
    });

    const status =
      totalResponses === 0 ? "Sin datos" : hasNonCompliance ? "No cumple" : "Cumple";

    return {
      status,
      counts,
      totalResponses,
    };
  }, [statsBySection, statsFilter]);

  const globalSummaryExtended = useMemo(() => {
    const nonCompliantCount =
      (globalSummary.counts["No conformidad menor"] ?? 0) +
      (globalSummary.counts["No conformidad mayor"] ?? 0);
    const observationCount = globalSummary.counts["Observación"] ?? 0;
    const compliantCount = Math.max(globalSummary.totalResponses - nonCompliantCount, 0);
    const auditedSections = statsBySection.filter((section) => section.totalResponses > 0).length;
    const complianceRate =
      globalSummary.totalResponses > 0
        ? Math.round((compliantCount / globalSummary.totalResponses) * 100)
        : 0;

    return {
      ...globalSummary,
      auditedSections,
      complianceRate,
      compliantCount,
      nonCompliantCount,
      observationCount,
    };
  }, [globalSummary, statsBySection]);

  const donutMetrics = useMemo(() => {
    if (globalSummaryExtended.totalResponses === 0) {
      return {
        background: "conic-gradient(#e2e8f0 0 100%)",
        showObservation: false,
      };
    }

    const observationCount = globalSummaryExtended.observationCount;
    const nonCompliantCount = globalSummaryExtended.nonCompliantCount;
    const compliantCount = Math.max(
      globalSummaryExtended.compliantCount - observationCount,
      0
    );
    const total = globalSummaryExtended.totalResponses;
    const compliantPercent = (compliantCount / total) * 100;
    const observationPercent = (observationCount / total) * 100;
    const observationStop = compliantPercent + observationPercent;

    return {
      background: `conic-gradient(#22c55e 0 ${compliantPercent}%, #facc15 ${compliantPercent}% ${observationStop}%, #ef4444 ${observationStop}% 100%)`,
      showObservation: observationPercent > 0,
    };
  }, [globalSummaryExtended]);

  const handleSaveAudit = async () => {
    if (!user) {
      toast.error("Necesitas iniciar sesión para guardar la auditoría.");
      return;
    }

    const trimmedAuditors = auditors.map((auditor) => auditor.trim()).filter(Boolean);
    const responseEntries = Object.entries(responses).filter(([, value]) => value);

    if (responseEntries.length === 0) {
      toast.error("Agrega al menos una respuesta antes de guardar.");
      return;
    }

    setIsSaving(true);
    const auditDate = new Date();

    const { data: sessionData, error: sessionError } = await supabase
      .from("internal_audit_sessions")
      .insert({
        user_id: user.id,
        audit_date: auditDate.toISOString(),
        auditors: trimmedAuditors,
      })
      .select()
      .single();

    if (sessionError || !sessionData) {
      toast.error("No pudimos guardar la auditoría.");
      setIsSaving(false);
      return;
    }

    const payload = responseEntries.map(([key, response]) => {
      const [sectionId = "", groupTitle = "", ...questionParts] = key.split("::");
      const question = questionParts.join("::");

      return {
        session_id: sessionData.id,
        section_id: sectionId,
        group_title: groupTitle,
        question,
        response,
        classification: classificationByResponse[response] ?? "",
        comment: comments[key] ?? null,
      };
    });

    if (payload.length > 0) {
      const { error: responsesError } = await supabase
        .from("internal_audit_responses")
        .insert(payload);

      if (responsesError) {
        toast.error("No pudimos guardar las respuestas.");
        setIsSaving(false);
        return;
      }
    }

    toast.success("Auditoría guardada correctamente.");
    await loadStatsAuditData();    
    setIsSaving(false);
  };

  const handleDeleteSession = async (sessionId: string) => {
    if (!canManageAudits) return;
    const confirmed = window.confirm(
      "¿Deseas eliminar este registro de auditoría? Esta acción no se puede deshacer."
    );
    if (!confirmed) return;

    setDeletingSessionId(sessionId);
    const { error: responsesError } = await supabase
      .from("internal_audit_responses")
      .delete()
      .eq("session_id", sessionId);

    if (responsesError) {
      toast.error("No pudimos eliminar las respuestas de la auditoría.");
      setDeletingSessionId(null);
      return;
    }

    const { error: sessionError } = await supabase
      .from("internal_audit_sessions")
      .delete()
      .eq("id", sessionId);

    if (sessionError) {
      toast.error("No pudimos eliminar el registro de auditoría.");
      setDeletingSessionId(null);
      return;
    }

    toast.success("Registro de auditoría eliminado.");
    await loadStatsAuditData();
    await loadAuditHistory();
    setDeletingSessionId(null);
  };
  
  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-10 bg-gradient-to-br from-slate-50 via-white to-emerald-50 pb-20">
        <section className="relative overflow-hidden rounded-3xl border border-emerald-100 bg-white px-6 py-10 shadow-sm sm:px-10">
          <div className="absolute inset-0 bg-gradient-to-r from-emerald-500/10 to-lime-400/10" />
          <div className="relative space-y-4">
            <p className="text-xs font-semibold uppercase tracking-[0.5em] text-emerald-600">
              Sistema de Gestión Integral
            </p>
            <h1 className="text-3xl font-bold text-gray-900 sm:text-4xl">
              Auditoría Interna
            </h1>
            <p className="max-w-4xl text-base text-gray-600 sm:text-lg">
              Formulario de auditoría interna con criterios ISO 14001 para evaluar el desempeño,
              cumplimiento y mejora del SGA. Completa cada sección con la respuesta y clasificación
              correspondiente.
            </p>
          </div>
        </section>

        <section className="rounded-3xl border border-emerald-100 bg-white p-2 shadow-sm">
          <div className="flex flex-wrap gap-2">
            {[
              { id: "stats", label: "Estadísticas" },
              { id: "audit", label: "Auditoría interna" },
            ].map((tab) => (
              <button
                key={tab.id}
                type="button"
                onClick={() => setActiveTab(tab.id as "stats" | "audit")}
                className={`rounded-2xl px-4 py-2 text-sm font-semibold transition-all ${
                  activeTab === tab.id
                    ? "bg-emerald-600 text-white shadow-sm"
                    : "text-emerald-700 hover:bg-emerald-50"
                }`}
              >
                {tab.label}
              </button>
            ))}
          </div>
        </section>

        {activeTab === "stats" ? (
          <section className="space-y-6">
            <Card className="border-emerald-100">
              <CardHeader className="flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
                <div className="space-y-1">
                  <CardTitle className="text-xl">Resultado global</CardTitle>
                  <p className="text-sm text-gray-600">
                    Consolidado de todas las auditorías registradas según el filtro seleccionado.
                  </p>
                </div>
                <div className="w-full sm:w-64">
                  <Select value={statsFilter} onValueChange={setStatsFilter}>
                    <SelectTrigger>
                      <SelectValue placeholder="Filtrar por sección" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">Todas las secciones</SelectItem>
                      {sections.map((section) => (
                        <SelectItem key={section.id} value={section.id}>
                          {section.title}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="flex flex-wrap items-center gap-3">
                  <span
                    className={`rounded-full px-4 py-1 text-xs font-semibold uppercase tracking-wide ${
                      globalSummary.status === "No cumple"
                        ? "bg-rose-100 text-rose-700"
                        : globalSummary.status === "Cumple"
                          ? "bg-emerald-100 text-emerald-700"
                          : "bg-slate-100 text-slate-600"
                    }`}
                  >
                    {globalSummaryExtended.status}
                  </span>
                  <span className="text-sm text-gray-500">
                    {globalSummaryExtended.totalResponses} respuestas registradas
                  </span>
                </div>
                <div className="grid gap-6 lg:grid-cols-[320px_1fr]">
                  <div className="rounded-2xl border border-slate-100 bg-slate-50 p-4">
                    <p className="text-xs font-semibold uppercase text-slate-500">
                      Cumple vs No cumple
                    </p>
                    <div className="mt-4 flex flex-col items-center gap-3">
                      <div
                        className="relative flex h-40 w-40 items-center justify-center rounded-full"
                        style={{ background: donutMetrics.background }}
                      >
                        <div className="flex h-24 w-24 flex-col items-center justify-center rounded-full bg-white text-center shadow-sm">
                          <span className="text-xl font-semibold text-emerald-700">
                            {globalSummaryExtended.complianceRate}%
                          </span>
                          <span className="text-xs text-slate-500">Cumplimiento</span>
                        </div>
                      </div>
                      <div className="grid gap-2 text-xs text-slate-600">
                        <div className="flex items-center gap-2">
                          <span className="h-2 w-2 rounded-full bg-emerald-500" />
                          Cumple
                        </div>
                        {donutMetrics.showObservation ? (
                          <div className="flex items-center gap-2">
                            <span className="h-2 w-2 rounded-full bg-yellow-400" />
                            Observaciones
                          </div>
                        ) : null}
                        <div className="flex items-center gap-2">
                          <span className="h-2 w-2 rounded-full bg-rose-500" />
                          No cumple
                        </div>
                      </div>
                    </div>
                  </div>
                  <div className="grid gap-4 sm:grid-cols-2">
                    <div className="rounded-2xl border border-slate-100 bg-white p-4 shadow-sm">
                      <div className="flex items-center gap-3">
                        <span className="flex h-10 w-10 items-center justify-center rounded-full bg-emerald-100 text-emerald-600">
                          <PieChart className="h-5 w-5" />
                        </span>
                        <div>
                          <p className="text-xs font-semibold uppercase text-slate-500">
                            Cumplimiento global
                          </p>
                          <p className="text-2xl font-semibold text-gray-900">
                            {globalSummaryExtended.complianceRate}%
                          </p>
                        </div>
                      </div>
                      <p className="mt-3 text-xs text-gray-500">
                        Última auditoría:{" "}
                        {globalSummaryExtended.totalResponses > 0
                          ? formattedStatsAuditDate
                          : "Sin auditoría"}
                      </p>
                    </div>
                    <div className="rounded-2xl border border-emerald-100 bg-emerald-50 p-4 shadow-sm">
                      <div className="flex items-center gap-3">
                        <span className="flex h-10 w-10 items-center justify-center rounded-full bg-emerald-100 text-emerald-700">
                          <CheckCircle2 className="h-5 w-5" />
                        </span>
                        <div>
                          <p className="text-xs font-semibold uppercase text-emerald-700">
                            Cumple
                          </p>
                          <p className="text-2xl font-semibold text-emerald-900">
                            {globalSummaryExtended.compliantCount}
                          </p>
                        </div>
                      </div>
                      <p className="mt-3 text-xs text-emerald-700">Requisitos que cumplen</p>
                    </div>
                    <div className="rounded-2xl border border-rose-100 bg-rose-50 p-4 shadow-sm">
                      <div className="flex items-center gap-3">
                        <span className="flex h-10 w-10 items-center justify-center rounded-full bg-rose-100 text-rose-600">
                          <XCircle className="h-5 w-5" />
                        </span>
                        <div>
                          <p className="text-xs font-semibold uppercase text-rose-700">
                            No cumple
                          </p>
                          <p className="text-2xl font-semibold text-rose-900">
                            {globalSummaryExtended.nonCompliantCount}
                          </p>
                        </div>
                      </div>
                      <p className="mt-3 text-xs text-rose-700">
                        Requisitos que no cumplen
                      </p>
                    </div>
                    <div className="rounded-2xl border border-slate-100 bg-white p-4 shadow-sm">
                      <div className="flex items-center gap-3">
                        <span className="flex h-10 w-10 items-center justify-center rounded-full bg-slate-100 text-slate-600">
                          <Puzzle className="h-5 w-5" />
                        </span>
                        <div>
                          <p className="text-xs font-semibold uppercase text-slate-500">
                            Secciones auditadas
                          </p>
                          <p className="text-2xl font-semibold text-gray-900">
                            {globalSummaryExtended.auditedSections} / {statsBySection.length}
                          </p>
                        </div>
                      </div>
                      <p className="mt-3 text-xs text-gray-500">
                        Última auditoría:{" "}
                        {globalSummaryExtended.totalResponses > 0
                          ? formattedStatsAuditDate
                          : "Sin auditoría"}
                      </p>
                    </div>
                  </div>
                </div>
                <div className="grid gap-3 sm:grid-cols-2 lg:grid-cols-3">
                  {classificationDetailLabels.map((item) => (
                    <div
                      key={item.key}
                      className="rounded-2xl border border-slate-100 bg-slate-50 p-3"
                    >
                      <p className="text-xs font-semibold uppercase text-slate-500">
                        {item.label}
                      </p>
                      <p className="mt-1 text-lg font-semibold text-gray-900">
                        {globalSummary.counts[item.key] ?? 0}
                      </p>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>

            <Card className="border-emerald-100">
              <CardHeader>
                <CardTitle className="text-xl">Resultado por sección</CardTitle>
                <p className="text-sm text-gray-600">
                  Consolidado de auditorías por categoría del sistema de gestión ambiental.
                </p>
              </CardHeader>
              <CardContent className="grid gap-4 lg:grid-cols-2">
                {statsBySection.map((section) => (
                  <div
                    key={section.id}
                    className="rounded-2xl border border-slate-100 bg-white p-4 shadow-sm"
                  >
                    <div className="flex flex-wrap items-center justify-between gap-3">
                      <h3 className="text-sm font-semibold text-gray-800">{section.title}</h3>
                      <span
                        className={`rounded-full px-3 py-1 text-xs font-semibold uppercase ${
                          section.status === "No cumple"
                            ? "bg-rose-100 text-rose-700"
                            : section.status === "Cumple"
                              ? "bg-emerald-100 text-emerald-700"
                              : "bg-slate-100 text-slate-600"
                        }`}
                      >
                        {section.status}
                      </span>
                    </div>
                    <p className="mt-2 text-xs text-gray-500">
                      {section.totalResponses} respuestas registradas
                    </p>
                    <p className="mt-1 text-xs text-gray-400">
                      Última auditoría:{" "}
                      {section.totalResponses > 0 ? formattedStatsAuditDate : "Sin auditoría"}
                    </p>
                    <div className="mt-4 grid gap-2 sm:grid-cols-2">
                      {classificationDetailLabels.map((item) => (
                        <div
                          key={item.key}
                          className="rounded-xl border border-slate-100 bg-slate-50 px-3 py-2"
                        >
                          <p className="text-[11px] font-semibold uppercase text-slate-500">
                            {item.label}
                          </p>
                          <p className="text-base font-semibold text-gray-900">
                            {section.counts[item.key] ?? 0}
                          </p>
                        </div>
                      ))}
                    </div>
                  </div>
                ))}
              </CardContent>
            </Card>
            {canManageAudits ? (
              <Card className="border-emerald-100">
                <CardHeader>
                  <CardTitle className="text-xl">Historial de auditorías</CardTitle>
                  <p className="text-sm text-gray-600">
                    Elimina registros individuales cuando sea necesario.
                  </p>
                </CardHeader>
                <CardContent className="space-y-4">
                  {auditHistory.length === 0 ? (
                    <p className="text-sm text-gray-500">
                      No hay auditorías registradas todavía.
                    </p>
                  ) : (
                    <div className="space-y-3">
                      {auditHistory.map((session) => (
                        <div
                          key={session.id}
                          className="flex flex-col gap-3 rounded-2xl border border-slate-100 bg-white p-4 shadow-sm sm:flex-row sm:items-center sm:justify-between"
                        >
                          <div className="space-y-1">
                            <p className="text-sm font-semibold text-gray-900">
                              {formatAuditDate(session.audit_date)}
                            </p>
                            <p className="text-xs text-gray-500">
                              Auditores:{" "}
                              {session.auditors?.filter(Boolean).join(", ") || "Sin registrar"}
                            </p>
                          </div>
                          <Button
                            type="button"
                            variant="destructive"
                            size="sm"
                            onClick={() => handleDeleteSession(session.id)}
                            disabled={deletingSessionId === session.id}
                          >
                            {deletingSessionId === session.id ? "Eliminando..." : "Eliminar"}
                          </Button>
                        </div>
                      ))}
                    </div>
                  )}
                </CardContent>
              </Card>
            ) : null}
          </section>
        ) : (
          <>
            <section
              id="audit-section-selector"
              className="grid gap-6 lg:grid-cols-[1.1fr_1fr]"
            >
              <Card className="border-emerald-100">
                <CardHeader className="flex flex-row items-center gap-3">
                  <div className="flex h-10 w-10 items-center justify-center rounded-full bg-emerald-100 text-emerald-600">
                    <ClipboardList className="h-5 w-5" />
                  </div>
                  <div>
                    <CardTitle className="text-xl">Descripción del formulario</CardTitle>
                    <p className="text-sm text-gray-600">
                      Información general previa a la auditoría interna.
                    </p>
                  </div>
                </CardHeader>
                <CardContent className="space-y-6">
                  <div className="space-y-2">
                    <Label htmlFor="audit-datetime">Fecha y hora automáticas</Label>
                    <div className="relative">
                      <CalendarDays className="absolute left-3 top-3 h-4 w-4 text-gray-400" />
                      <Input
                        id="audit-datetime"
                        readOnly
                        value={formattedNow}
                        className="pl-9"
                      />
                    </div>
                  </div>

                  <div className="space-y-4">
                    <div className="flex items-center gap-2">
                      <UserCircle2 className="h-4 w-4 text-emerald-600" />
                      <Label>Nombre del Auditor (máximo 4)</Label>
                    </div>
                    <div className="grid gap-3 sm:grid-cols-2">
                      {Array.from({ length: 4 }).map((_, index) => (
                        <Input
                          key={index}
                          placeholder={`Auditor ${index + 1}`}
                          value={auditors[index] ?? ""}
                          onChange={(event) =>
                            setAuditors((prev) => {
                              const next = [...prev];
                              next[index] = event.target.value;
                              return next;
                            })
                          }
                        />
                      ))}
                    </div>                 
                  </div>
                </CardContent>
              </Card>

              <Card className="border-emerald-100">
                <CardHeader className="flex flex-row items-center gap-3">
                  <div className="flex h-10 w-10 items-center justify-center rounded-full bg-emerald-100 text-emerald-600">
                    <ListChecks className="h-5 w-5" />
                  </div>
                  <div>
                    <CardTitle className="text-xl">Sección por auditar</CardTitle>
                    <p className="text-sm text-gray-600">
                      Selecciona la sección y continúa para ir directamente al bloque correspondiente.
                    </p>
                  </div>
                </CardHeader>
                <CardContent className="space-y-5">

                  <div className="grid gap-3 sm:grid-cols-2">
                    {sections.map((section) => (
                      <button
                        key={section.id}
                        type="button"
                        onClick={() => setSelectedSection(section.id)}
                        className={`rounded-2xl border px-4 py-3 text-left text-sm font-semibold transition-all duration-200 ${
                          selectedSection === section.id
                            ? "border-emerald-500 bg-emerald-50 text-emerald-700"
                            : "border-gray-200 bg-white text-gray-600 hover:border-emerald-200"
                        }`}
                      >
                        {section.title}
                      </button>
                    ))}
                  </div>

                  <Button type="button" className="w-full" onClick={handleContinue}>
                    Continuar
                  </Button>
                </CardContent>
              </Card>
            </section>

            <section className="rounded-3xl border border-emerald-100 bg-white p-6 shadow-sm">
              <h2 className="text-xl font-semibold text-gray-900">
                Criterios de evaluación de cumplimiento
              </h2>
              <p className="mt-2 text-sm text-gray-600">
                Cada pregunta debe responderse con el nivel de cumplimiento y su clasificación
                asociada.
              </p>
              <div className="mt-4 grid gap-4 md:grid-cols-2">
                <div className="rounded-2xl border border-emerald-100 bg-emerald-50 p-4">
                  <p className="text-xs font-semibold uppercase text-emerald-700">Cumple</p>
                  <ul className="mt-2 space-y-2 text-sm text-emerald-900">
                    <li>• Cumple correctamente → Conformidad</li>
                    <li>• Cumple, pero puede mejorar → Oportunidad de mejora</li>
                    <li>• Cumple, pero hay riesgo futuro → Observación</li>
                    <li>• Cumple de forma destacada → Buena práctica</li>
                  </ul>
                </div>
                <div className="rounded-2xl border border-rose-100 bg-rose-50 p-4">
                  <p className="text-xs font-semibold uppercase text-rose-700">No cumple</p>
                  <ul className="mt-2 space-y-2 text-sm text-rose-900">
                    <li>• No cumple parcialmente → No conformidad menor</li>
                    <li>• No cumple totalmente o hay riesgo grave → No conformidad mayor</li>
                  </ul>
                </div>
              </div>
            </section>

            <div className="space-y-10">
              <section
                key={activeSection.id}
                id={activeSection.id}
                className="space-y-6 rounded-3xl border border-slate-200 bg-white p-6 shadow-sm"
              >
                <header className="space-y-2">
                  <p className="text-xs font-semibold uppercase tracking-[0.3em] text-emerald-600">
                    {activeSection.title}
                  </p>
                  <h2 className="text-2xl font-semibold text-gray-900">
                    {activeSection.description}
                  </h2>
                </header>

                <div className="space-y-6">
                  {activeSection.groups.map((group) => (
                    <Card key={group.title} className="border-emerald-100 shadow-none">
                      <CardHeader>
                        <CardTitle className="text-lg text-gray-900">{group.title}</CardTitle>
                      </CardHeader>
                      <CardContent className="space-y-5">
                        {group.questions?.map((question) => {
                          const questionKey = buildQuestionKey(
                            activeSection.id,
                            group.title,
                            question
                          );
                          const response = responses[questionKey] ?? "";
                          const classification = response ? classificationByResponse[response] : "";
                          const isNonCompliance = response && nonComplianceResponses.has(response);

                          return (
                            <div
                              key={question}
                              className="rounded-2xl border border-gray-100 bg-slate-50 p-4"
                            >
                              <p className="text-sm font-medium text-gray-800">{question}</p>
                              <div className="mt-4 grid gap-4 md:grid-cols-2">
                                <div className="space-y-2">
                                  <Label>Respuesta</Label>
                                  <Select
                                    value={response}
                                    onValueChange={(value) =>
                                      setResponses((prev) => ({
                                        ...prev,
                                        [questionKey]: value,
                                      }))
                                    }
                                  >
                                    <SelectTrigger>
                                      <SelectValue placeholder="Selecciona la respuesta" />
                                    </SelectTrigger>
                                    <SelectContent>
                                      {responseOptions.map((option) => (
                                        <SelectItem key={option} value={option}>
                                          {option}
                                        </SelectItem>
                                      ))}
                                    </SelectContent>
                                  </Select>
                                </div>
                                <div className="space-y-2">
                                  <Label>Clasificación (automática)</Label>
                                  <Input
                                    readOnly
                                    value={classification}
                                    placeholder="Se asigna automáticamente"
                                  />
                                </div>
                              </div>
                              {isNonCompliance ? (
                                <div className="mt-4 space-y-2">
                                  <Label>Comentario requerido</Label>
                                  <Textarea
                                    value={comments[questionKey] ?? ""}
                                    onChange={(event) =>
                                      setComments((prev) => ({
                                        ...prev,
                                        [questionKey]: event.target.value,
                                      }))
                                    }
                                    placeholder="Describe la no conformidad detectada y las acciones inmediatas."
                                  />
                                </div>
                              ) : null}
                            </div>
                          );
                        })}

                        {group.subsections?.map((subsection) => (
                          <div key={subsection.title} className="space-y-4">
                            <h3 className="text-base font-semibold text-emerald-700">
                              {subsection.title}
                            </h3>
                            {subsection.questions.map((question) => {
                              const questionKey = buildQuestionKey(
                                activeSection.id,
                                subsection.title,
                                question
                              );
                              const response = responses[questionKey] ?? "";
                              const classification = response
                                ? classificationByResponse[response]
                                : "";
                              const isNonCompliance =
                                response && nonComplianceResponses.has(response);

                              return (
                                <div
                                  key={question}
                                  className="rounded-2xl border border-gray-100 bg-slate-50 p-4"
                                >
                                  <p className="text-sm font-medium text-gray-800">{question}</p>
                                  <div className="mt-4 grid gap-4 md:grid-cols-2">
                                    <div className="space-y-2">
                                      <Label>Respuesta</Label>
                                      <Select
                                        value={response}
                                        onValueChange={(value) =>
                                          setResponses((prev) => ({
                                            ...prev,
                                            [questionKey]: value,
                                          }))
                                        }
                                      >
                                        <SelectTrigger>
                                          <SelectValue placeholder="Selecciona la respuesta" />
                                        </SelectTrigger>
                                        <SelectContent>
                                          {responseOptions.map((option) => (
                                            <SelectItem key={option} value={option}>
                                              {option}
                                            </SelectItem>
                                          ))}
                                        </SelectContent>
                                      </Select>
                                    </div>
                                    <div className="space-y-2">
                                      <Label>Clasificación (automática)</Label>
                                      <Input
                                        readOnly
                                        value={classification}
                                        placeholder="Se asigna automáticamente"
                                      />
                                    </div>
                                  </div>
                                  {isNonCompliance ? (
                                    <div className="mt-4 space-y-2">
                                      <Label>Comentario requerido</Label>
                                      <Textarea
                                        value={comments[questionKey] ?? ""}
                                        onChange={(event) =>
                                          setComments((prev) => ({
                                            ...prev,
                                            [questionKey]: event.target.value,
                                          }))
                                        }
                                        placeholder="Describe la no conformidad detectada y las acciones inmediatas."
                                      />
                                    </div>
                                  ) : null}
                                </div>
                              );
                            })}
                          </div>
                        ))}
                      </CardContent>
                    </Card>
                  ))}
                </div>

                <div className="flex flex-col gap-3 border-t border-dashed border-gray-200 pt-6 sm:flex-row sm:items-center sm:justify-between">
                  <Button
                    type="button"
                    variant="outline"
                    onClick={() =>
                      document.getElementById("audit-section-selector")?.scrollIntoView({
                        behavior: "smooth",
                      })
                    }
                  >
                    Elegir otra sección
                  </Button>
                  <Button
                    type="button"
                    onClick={handleSaveAudit}
                    disabled={isSaving}
                    className="sm:w-auto"
                  >
                    {isSaving ? "Guardando..." : "Guardar auditoría"}
                  </Button>
                </div>
              </section>
            </div>
          </>
        )}
      </div>
    </SGIDashboardLayout>
  );
}