"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import Link from "next/link";
import type { User } from "@supabase/supabase-js";
import {
  CalendarCheck,
  CalendarDays,
  FileDown,
  Flag,
  Layers,
  ListChecks,
  NotebookPen,
  Plus,
  ShieldCheck,
} from "lucide-react";
import { toast } from "sonner";

import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Dialog,
  DialogContent,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Textarea } from "@/components/ui/textarea";
import { supabase } from "@/lib/supabase";

const allowedManagers = new Set([
  "ehs.mexico@meiwa.com.mx",
  "auxiliar.ehs@meiwa.com.mx",
]);
const actionSourceLabel = "Acciones y seguimiento";

const buildActionPlanHref = (action: { decision_text?: string | null; responsible?: string | null; due_date?: string | null; title?: string | null }) => {
  const params = new URLSearchParams({
    source: actionSourceLabel,
    description: action.decision_text ?? "",
    comments: action.title ?? "",
    owner: action.responsible ?? "",
    targetDate: action.due_date ?? "",
  });
  return `/meiwa-core/meiwa-sgi/planning/action-plan?${params.toString()}`;
};

const sessionStatusOptions = [
  { value: "draft", label: "Borrador" },
  { value: "closed", label: "Cerrada" },
];

const outputStatusOptions = [
  { value: "open", label: "Abierta" },
  { value: "in_progress", label: "En proceso" },
  { value: "closed", label: "Cerrada" },
];

const outputStatusLabelMap = outputStatusOptions.reduce<Record<string, string>>((acc, option) => {
  acc[option.value] = option.label;
  return acc;
}, {});

const inputSections = [
  {
    key: "a",
    title: "Estado de acciones de revisiones previas",
    description:
      "Seguimiento a compromisos anteriores, incluyendo acciones pendientes, retrasadas y cerradas.",
    links: [{ label: "Plan de acción", href: "/meiwa-core/meiwa-sgi/planning/action-plan" }],
  },
  {
    key: "b1",
    title: "Cambios en requisitos legales y partes interesadas",
    description:
      "Actualizaciones legales, necesidades de partes interesadas y cambios en requisitos aplicables.",
    links: [{ label: "Matriz legal", href: "/meiwa-core/meiwa-sgi/planning/legal-requirements" }],
  },
  {
    key: "b2",
    title: "Cuestiones internas y externas pertinentes",
    description:
      "Contexto estratégico, comunicación de partes interesadas y cambios en condiciones internas/externas.",
    links: [{ label: "Comunicación", href: "/meiwa-core/meiwa-sgi/support/comunication" }],
  },
  {
    key: "b3",
    title: "Aspectos ambientales significativos",
    description:
      "Aspectos significativos vigentes, cambios de estado y nuevos aspectos identificados.",
    links: [{ label: "Aspectos ambientales", href: "/meiwa-core/meiwa-sgi/planning/aspects-impacts" }],
  },
  {
    key: "b4",
    title: "Riesgos y oportunidades",
    description:
      "Riesgos críticos, residuales no aceptables y cambios respecto a la revisión anterior.",
    links: [{ label: "Riesgos y oportunidades", href: "/meiwa-core/meiwa-sgi/planning/risk-and-opportunities" }],
  },
  {
    key: "c",
    title: "Cumplimiento de objetivos ambientales",
    description:
      "Resultados de objetivos, contramedidas activas y metas en riesgo.",
    links: [{ label: "Objetivos ambientales", href: "/meiwa-core/meiwa-sgi/planning/environmental-objectives" }],
  },
  {
    key: "d",
    title: "Desempeño ambiental",
    description:
      "No conformidades, desempeño en mediciones, cumplimiento legal y resultados de auditoría.",
    links: [{ label: "Seguimiento y medición", href: "/meiwa-core/meiwa-sgi/evaluation/monitoring-measurement" }],
  },
  {
    key: "e",
    title: "Adecuación de recursos",
    description:
      "Disponibilidad de recursos críticos y cobertura de procesos.",
    links: [{ label: "Recursos", href: "/meiwa-core/meiwa-sgi/support/resources" }],
  },
  {
    key: "f",
    title: "Comunicaciones de partes interesadas",
    description:
      "Comunicados relevantes, quejas y tendencias registradas.",
    links: [{ label: "Comunicaciones", href: "/meiwa-core/meiwa-sgi/support/comunication" }],
  },
  {
    key: "g",
    title: "Oportunidades de mejora continua",
    description:
      "Mejoras registradas, cerradas y nuevas oportunidades detectadas.",
    links: [{ label: "Mejora continua", href: "/meiwa-core/meiwa-sgi/improvement/continuous-improvement" }],
  },
];

const outputSections = [
  {
    key: "conclusions",
    title: "Conclusiones sobre adecuación, conveniencia y eficacia",
  },
  {
    key: "improvement_decisions",
    title: "Decisiones sobre oportunidades de mejora continua",
  },
  {
    key: "system_changes",
    title: "Necesidades de cambio al SGA y recursos",
  },
  {
    key: "objectives_actions",
    title: "Acciones cuando se hayan logrado objetivos ambientales",
  },
  {
    key: "business_integration",
    title: "Mejoras para integrar el SGA en procesos de negocio",
  },
  {
    key: "strategic_implications",
    title: "Implicaciones para la dirección estratégica",
  },
];

type ManagementReviewSession = {
  id: string;
  session_code: string | null;
  review_date: string | null;
  period_start: string | null;
  period_end: string | null;
  participants: string | null;
  prepared_by: string | null;
  status: string | null;
  general_conclusions: string | null;
  strategic_implications: string | null;
  created_at: string | null;
  updated_at: string | null;
};

type ManagementReviewInput = {
  id: string;
  session_id: string;
  input_key: string;
  summary_snapshot: Record<string, unknown> | null;
  management_comment: string | null;
  evidence_link: string | null;
  created_at: string | null;
  updated_at?: string | null;
};

type ManagementReviewOutput = {
  id: string;
  session_id: string;
  output_key: string;
  decision_text: string | null;
  responsible: string | null;
  due_date: string | null;
  action_link: string | null;
  status: string | null;
  evidence_link: string | null;
  created_at: string | null;
  updated_at?: string | null;
};

type InputDraft = {
  comment: string;
  evidenceFile: File | null;
};

type OutputDraft = {
  output_key: string;
  decision_text: string;
  responsible: string;
  due_date: string;
  action_link: string;
  status: string;
};

const emptySessionForm = {
  session_code: "",
  review_date: "",
  period_start: "",
  period_end: "",
  participants: "",
  prepared_by: "",
  status: "draft",
  general_conclusions: "",
  strategic_implications: "",
};

const emptyOutputDraft = (): OutputDraft => ({
  output_key: "conclusions",
  decision_text: "",
  responsible: "",
  due_date: "",
  action_link: "",
  status: "open",
});

const formatDate = (value?: string | null) => {
  if (!value) return "Sin fecha";
  const date = new Date(value);
  if (Number.isNaN(date.getTime())) return "Sin fecha";
  return date.toLocaleDateString("es-MX");
};

const formatPeriod = (start?: string | null, end?: string | null) => {
  if (!start && !end) return "Sin periodo";
  if (start && end) return `${formatDate(start)} - ${formatDate(end)}`;
  return start ? formatDate(start) : formatDate(end);
};

const getEvidenceUrl = async (path?: string | null) => {
  if (!path) return null;
  if (path.startsWith("http://") || path.startsWith("https://")) return path;
  const { data, error } = await supabase.storage
    .from("management-review-evidence")
    .createSignedUrl(path, 60 * 60);
  if (error) return null;
  return data.signedUrl;
};

const buildSnapshotItems = (summary: Record<string, string | number | null>) => {
  return Object.entries(summary).map(([label, value]) => ({
    label,
    value: value ?? "Sin dato",
  }));
};

export default function ManagementReviewPage() {
  const [user, setUser] = useState<User | null>(null);
  const [canManage, setCanManage] = useState(false);
  const [sessions, setSessions] = useState<ManagementReviewSession[]>([]);
  const [selectedSessionId, setSelectedSessionId] = useState<string>("");
  const [inputs, setInputs] = useState<ManagementReviewInput[]>([]);
  const [outputs, setOutputs] = useState<ManagementReviewOutput[]>([]);
  const [sessionMatrixDetails, setSessionMatrixDetails] = useState<
    Record<
      string,
      {
        inputs: Record<string, ManagementReviewInput>;
        outputs: ManagementReviewOutput[];
      }
    >
  >({});  
  const [inputDrafts, setInputDrafts] = useState<Record<string, InputDraft>>({});
  const [inputSummaries, setInputSummaries] = useState<Record<string, Record<string, string | number | null>>>(
    {}
  );
  const [loading, setLoading] = useState(true);
  const [sessionDialogOpen, setSessionDialogOpen] = useState(false);
  const [editingSession, setEditingSession] = useState<ManagementReviewSession | null>(null);
  const [sessionForm, setSessionForm] = useState(emptySessionForm);
  const [outputDialogOpen, setOutputDialogOpen] = useState(false);
  const [editingOutput, setEditingOutput] = useState<ManagementReviewOutput | null>(null);
  const [outputForm, setOutputForm] = useState<OutputDraft>(emptyOutputDraft);
  const [outputEvidenceFile, setOutputEvidenceFile] = useState<File | null>(null);
  const [savingInputs, setSavingInputs] = useState(false);
  const [deletingSessionId, setDeletingSessionId] = useState<string | null>(null);

  useEffect(() => {
    const loadSession = async () => {
      const { data } = await supabase.auth.getSession();
      const currentUser = data.session?.user ?? null;
      setUser(currentUser);
      setCanManage(allowedManagers.has(currentUser?.email ?? ""));
    };

    loadSession();

    const { data: subscription } = supabase.auth.onAuthStateChange((_event, session) => {
      setUser(session?.user ?? null);
      setCanManage(allowedManagers.has(session?.user?.email ?? ""));
    });

    return () => {
      subscription.subscription.unsubscribe();
    };
  }, []);

  const loadSessions = useCallback(async () => {
    setLoading(true);
    const { data, error } = await supabase
      .from("management_review_sessions")
      .select("*")
      .order("review_date", { ascending: false });

    if (error) {
      toast.error("No se pudieron cargar las sesiones de revisión.");
      setLoading(false);
      return;
    }

    setSessions(data ?? []);
    if (data?.length && !selectedSessionId) {
      setSelectedSessionId(data[0].id);
    }
    setLoading(false);
  }, [selectedSessionId]);

  const loadInputs = useCallback(async (sessionId: string) => {
    const { data, error } = await supabase
      .from("management_review_inputs")
      .select("*")
      .eq("session_id", sessionId)
      .order("input_key", { ascending: true });

    if (error) {
      toast.error("No se pudieron cargar las entradas de la revisión.");
      return [] as ManagementReviewInput[];
    }

    return (data ?? []) as ManagementReviewInput[];
  }, []);

  const loadOutputs = useCallback(async (sessionId: string) => {
    const { data, error } = await supabase
      .from("management_review_outputs")
      .select("*")
      .eq("session_id", sessionId)
      .order("created_at", { ascending: false });

    if (error) {
      toast.error("No se pudieron cargar las salidas de la revisión.");
      return [] as ManagementReviewOutput[];
    }

    return (data ?? []) as ManagementReviewOutput[];
  }, []);

  const loadSessionMatrixDetails = useCallback(async () => {
    const [{ data: inputsData, error: inputsError }, { data: outputsData, error: outputsError }] =
      await Promise.all([
        supabase
          .from("management_review_inputs")
          .select("id, session_id, input_key, management_comment, created_at, updated_at"),
        supabase
          .from("management_review_outputs")
          .select("id, session_id, output_key, decision_text, created_at, updated_at")
          .order("created_at", { ascending: false }),
      ]);

    if (inputsError || outputsError) {
      toast.error("No se pudo cargar la matriz completa de sesiones.");
      return;
    }

    const matrix: Record<
      string,
      {
        inputs: Record<string, ManagementReviewInput>;
        outputs: ManagementReviewOutput[];
      }
    > = {};

    (inputsData ?? []).forEach((item) => {
      if (!matrix[item.session_id]) {
        matrix[item.session_id] = { inputs: {}, outputs: [] };
      }
      matrix[item.session_id].inputs[item.input_key] = item as ManagementReviewInput;
    });

    (outputsData ?? []).forEach((item) => {
      if (!matrix[item.session_id]) {
        matrix[item.session_id] = { inputs: {}, outputs: [] };
      }
      matrix[item.session_id].outputs.push(item as ManagementReviewOutput);
    });

    setSessionMatrixDetails(matrix);
  }, []);

  const buildInputSummaries = useCallback(
    async (session: ManagementReviewSession | undefined) => {
      const currentIndex = sessions.findIndex((entry) => entry.id === session?.id);
      const previousSession = currentIndex >= 0 ? sessions[currentIndex + 1] : undefined;

      const [
        prevOutputsResponse,
        legalResponse,
        aspectsResponse,
        risksResponse,
        objectivesResponse,
        ncResponse,
        complianceResponse,
        auditSessionsResponse,
        auditResponsesResponse,
        resourcesResponse,
        processesResponse,
        communicationResponse,
        incidentsResponse,
        improvementResponse,
        actionPlanResponse,
      ] = await Promise.all([
        previousSession
          ? supabase
              .from("management_review_outputs")
              .select("status")
              .eq("session_id", previousSession.id)
          : Promise.resolve({ data: [], error: null }),
        supabase
          .from("legal_requirements")
          .select("compliance_status,action_status,updated_at"),
        supabase
          .from("environmental_aspects")
          .select("expected_value_ve,is_significant,updated_at"),
        supabase
          .from("env_risk_opportunities")
          .select("p_residual,i_residual,updated_at"),
        supabase
          .from("objectives")
          .select("status,countermeasures"),
        supabase.from("nc_corrective_actions").select("status,classification"),
        supabase.from("legal_requirements").select("compliance_status,action_status"),
        supabase.from("internal_audit_sessions").select("id"),
        supabase.from("internal_audit_responses").select("response"),
        supabase.from("sgm_resources").select("status,is_critical,linked_to_legal"),
        supabase.from("sgm_processes").select("is_active"),
        supabase.from("communications_posts").select("category,status,publish_date"),
        supabase.from("communication_incidents").select("status,category"),
        supabase.from("continuous_improvement_register").select("status,source_type"),
        supabase.from("sga_action_plan").select("due_date,end_date"),
      ]);

      if (
        legalResponse.error ||
        aspectsResponse.error ||
        risksResponse.error ||
        objectivesResponse.error ||
        ncResponse.error ||
        complianceResponse.error ||
        auditSessionsResponse.error ||
        auditResponsesResponse.error ||
        resourcesResponse.error ||
        processesResponse.error ||
        communicationResponse.error ||
        incidentsResponse.error ||
        improvementResponse.error ||
        actionPlanResponse.error
      ) {
        toast.error("No se pudieron cargar los KPIs automáticos de entradas.");
      }

      const prevOutputs = (prevOutputsResponse.data ?? []) as { status: string | null }[];
      const prevClosed = prevOutputs.filter((item) => item.status === "closed").length;
      const prevOpen = prevOutputs.filter((item) => item.status !== "closed").length;
      const prevTotal = prevOutputs.length;

      const actionPlanItems = (actionPlanResponse.data ?? []) as {
        due_date: string | null;
        end_date: string | null;
      }[];
      const today = new Date();
      const actionDelayed = actionPlanItems.filter(
        (item) =>
          item.due_date &&
          !item.end_date &&
          new Date(item.due_date) < today
      ).length;

      const legalItems = (legalResponse.data ?? []) as {
        compliance_status: string | null;
        action_status: string | null;
        updated_at?: string | null;
      }[];
      const legalOpen = legalItems.filter((item) =>
        ["open", "in_progress"].includes(item.action_status ?? "")
      ).length;

      const aspectItems = (aspectsResponse.data ?? []) as {
        expected_value_ve: number | null;
        is_significant: boolean | null;
        updated_at?: string | null;
      }[];
      const significantAspects = aspectItems.filter(
        (item) => (item.expected_value_ve ?? 0) >= 10 || Boolean(item.is_significant)
      ).length;

      const riskItems = (risksResponse.data ?? []) as {
        p_residual: number | null;
        i_residual: number | null;
        updated_at?: string | null;
      }[];
      const residualScores = riskItems.map(
        (item) => (item.p_residual ?? 0) * (item.i_residual ?? 0)
      );
      const nonAcceptableRisks = residualScores.filter((score) => score >= 13).length;
      const criticalRisks = residualScores.filter((score) => score >= 20).length;

      const objectiveItems = (objectivesResponse.data ?? []) as {
        status: string | null;
        countermeasures: string | null;
      }[];
      const objectivesMet = objectiveItems.filter((item) => item.status === "met").length;
      const objectivesRisk = objectiveItems.filter((item) => item.status === "at_risk").length;
      const objectivesNotMet = objectiveItems.filter((item) => item.status === "not_met").length;
      const objectivesWithCountermeasures = objectiveItems.filter((item) =>
        Boolean(item.countermeasures)
      ).length;

      const ncItems = (ncResponse.data ?? []) as {
        status: string | null;
        classification: string | null;
      }[];
      const ncOpen = ncItems.filter((item) => item.status !== "closed").length;
      const ncClosed = ncItems.filter((item) => item.status === "closed").length;

      const complianceItems = (complianceResponse.data ?? []) as {
        compliance_status: string | null;
        action_status: string | null;
      }[];
      const complianceOk = complianceItems.filter(
        (item) => item.compliance_status === "complies"
      ).length;
      const compliancePending = complianceItems.filter(
        (item) => item.compliance_status === "not_evaluated"
      ).length;
      const complianceNon = complianceItems.filter(
        (item) => item.compliance_status === "non_complies"
      ).length;
      const complianceOpenActions = complianceItems.filter((item) =>
        ["open", "in_progress"].includes(item.action_status ?? "")
      ).length;

      const auditSessions = (auditSessionsResponse.data ?? []) as { id: string }[];
      const auditResponses = (auditResponsesResponse.data ?? []) as {
        response: string | null;
      }[];
      const auditFindings = auditResponses.filter(
        (item) => item.response && item.response !== "conforme"
      ).length;

      const resourceItems = (resourcesResponse.data ?? []) as {
        status: string | null;
        is_critical: boolean | null;
        linked_to_legal: boolean | null;
      }[];
      const resourcesCritical = resourceItems.filter((item) => item.is_critical).length;
      const resourcesOutdated = resourceItems.filter((item) => item.status === "to_update").length;

      const processItems = (processesResponse.data ?? []) as { is_active: boolean | null }[];
      const activeProcesses = processItems.filter((item) => item.is_active).length;

      const communicationPosts = (communicationResponse.data ?? []) as {
        category: string | null;
        status: string | null;
        publish_date: string | null;
      }[];
      const communicationRelevant = communicationPosts.filter(
        (item) => item.status === "published"
      ).length;

      const incidentItems = (incidentsResponse.data ?? []) as { status: string | null }[];
      const incidentOpen = incidentItems.filter((item) => item.status !== "closed").length;

      const improvementItems = (improvementResponse.data ?? []) as {
        status: string | null;
        source_type: string | null;
      }[];
      const improvementsClosed = improvementItems.filter((item) => item.status === "closed").length;

      setInputSummaries({
        a: {
          "Acciones previas cerradas": prevClosed,
          "Acciones previas abiertas": prevOpen,
          "% cierre previas": prevTotal ? Math.round((prevClosed / prevTotal) * 100) : 0,
          "Acciones retrasadas (plan)": actionDelayed,
        },
        b1: {
          "Requisitos legales registrados": legalItems.length,
          "Acciones legales abiertas": legalOpen,
        },
        b2: {
          "Comunicados relevantes": communicationRelevant,
          "Quejas/incident. abiertas": incidentOpen,
        },
        b3: {
          "Aspectos significativos": significantAspects,
          "Aspectos totales": aspectItems.length,
        },
        b4: {
          "Riesgos no aceptables": nonAcceptableRisks,
          "Riesgos críticos": criticalRisks,
          "Riesgos evaluados": riskItems.length,
        },
        c: {
          "Objetivos cumplidos": objectivesMet,
          "Objetivos en riesgo": objectivesRisk,
          "Objetivos no cumplidos": objectivesNotMet,
          "Con contramedidas": objectivesWithCountermeasures,
        },
        d: {
          "NC abiertas": ncOpen,
          "NC cerradas": ncClosed,
          "Cumplimiento legal OK": complianceOk,
          "Cumplimiento no evaluado": compliancePending,
          "Cumplimiento no cumple": complianceNon,
          "Acciones legales abiertas": complianceOpenActions,
          "Auditorías realizadas": auditSessions.length,
          "Hallazgos auditoría": auditFindings,
        },
        e: {
          "Recursos críticos": resourcesCritical,
          "Recursos por actualizar": resourcesOutdated,
          "Procesos activos": activeProcesses,
        },
        f: {
          "Comunicados publicados": communicationRelevant,
          "Quejas/incident. abiertas": incidentOpen,
        },
        g: {
          "Mejoras registradas": improvementItems.length,
          "Mejoras cerradas": improvementsClosed,
        },
      });
    },
    [sessions]
  );

  useEffect(() => {
    loadSessions();
  }, [loadSessions]);

  useEffect(() => {
    loadSessionMatrixDetails();
  }, [loadSessionMatrixDetails]);

  useEffect(() => {
    const loadSessionDetails = async () => {
      if (!selectedSessionId) return;
      const session = sessions.find((item) => item.id === selectedSessionId);
      const [inputsData, outputsData] = await Promise.all([
        loadInputs(selectedSessionId),
        loadOutputs(selectedSessionId),
      ]);

      setInputs(inputsData);
      setOutputs(outputsData);

      const draftState: Record<string, InputDraft> = {};
      inputSections.forEach((section) => {
        const existing = inputsData.find((item) => item.input_key === section.key);
        draftState[section.key] = {
          comment: existing?.management_comment ?? "",
          evidenceFile: null,
        };
      });
      setInputDrafts(draftState);
      await buildInputSummaries(session);
    };

    loadSessionDetails();
  }, [
    selectedSessionId,
    sessions,
    loadInputs,
    loadOutputs,
    buildInputSummaries,
  ]);

  const selectedSession = useMemo(
    () => sessions.find((session) => session.id === selectedSessionId),
    [sessions, selectedSessionId]
  );

  const sessionSummary = useMemo(() => {
    return {
      inputs: inputs.length,
      outputs: outputs.length,
      openActions: outputs.filter((output) => output.status !== "closed").length,
      closedActions: outputs.filter((output) => output.status === "closed").length,
    };
  }, [inputs, outputs]);

  const handleOpenSessionDialog = (session?: ManagementReviewSession) => {
    if (session) {
      setEditingSession(session);
      setSessionForm({
        session_code: session.session_code ?? "",
        review_date: session.review_date ?? "",
        period_start: session.period_start ?? "",
        period_end: session.period_end ?? "",
        participants: session.participants ?? "",
        prepared_by: session.prepared_by ?? "",
        status: session.status ?? "draft",
        general_conclusions: session.general_conclusions ?? "",
        strategic_implications: session.strategic_implications ?? "",
      });
    } else {
      setEditingSession(null);
      setSessionForm(emptySessionForm);
    }
    setSessionDialogOpen(true);
  };

  const handleSaveSession = async () => {
    if (!user) return;
    if (!sessionForm.review_date) {
      toast.error("Indica la fecha de revisión.");
      return;
    }

    const payload = {
      session_code: sessionForm.session_code || null,
      review_date: sessionForm.review_date,
      period_start: sessionForm.period_start || null,
      period_end: sessionForm.period_end || null,
      participants: sessionForm.participants || null,
      prepared_by: sessionForm.prepared_by || null,
      status: sessionForm.status || "draft",
      general_conclusions: sessionForm.general_conclusions || null,
      strategic_implications: sessionForm.strategic_implications || null,
      updated_at: new Date().toISOString(),
    };

    const response = editingSession
      ? await supabase
          .from("management_review_sessions")
          .update(payload)
          .eq("id", editingSession.id)
      : await supabase.from("management_review_sessions").insert({
          ...payload,
          created_at: new Date().toISOString(),
        });

    if (response.error) {
      toast.error("No se pudo guardar la sesión de revisión.");
      return;
    }

    toast.success("Sesión de revisión guardada.");
    setSessionDialogOpen(false);
    setEditingSession(null);
    await loadSessions();
  };

  const handleDeleteSession = async (session: ManagementReviewSession) => {
    if (!canManage) return;
    const confirmed = window.confirm(
      "¿Deseas eliminar esta sesión de revisión? Esta acción no se puede deshacer."
    );
    if (!confirmed) return;

    setDeletingSessionId(session.id);
    const { error: inputsError } = await supabase
      .from("management_review_inputs")
      .delete()
      .eq("session_id", session.id);

    if (inputsError) {
      toast.error("No se pudieron eliminar las entradas de la sesión.");
      setDeletingSessionId(null);
      return;
    }

    const { error: outputsError } = await supabase
      .from("management_review_outputs")
      .delete()
      .eq("session_id", session.id);

    if (outputsError) {
      toast.error("No se pudieron eliminar las salidas de la sesión.");
      setDeletingSessionId(null);
      return;
    }

    const { error: sessionError } = await supabase
      .from("management_review_sessions")
      .delete()
      .eq("id", session.id);

    if (sessionError) {
      toast.error("No se pudo eliminar la sesión.");
      setDeletingSessionId(null);
      return;
    }

    toast.success("Sesión eliminada.");
    if (selectedSessionId === session.id) {
      setSelectedSessionId("");
    }
    await loadSessions();
    await loadSessionMatrixDetails();
    setDeletingSessionId(null);
  };
    
  const handleSaveAllInputs = async () => {
    if (!selectedSessionId) return;
    setSavingInputs(true);
    let hasErrors = false;

    for (const section of inputSections) {
      const draft = inputDrafts[section.key];
      if (!draft) continue;

      const existing = inputs.find((item) => item.input_key === section.key);
      let evidenceLink = existing?.evidence_link ?? null;

      if (draft.evidenceFile && user) {
        const path = `sessions/${selectedSessionId}/${section.key}-${Date.now()}-${draft.evidenceFile.name}`;
        const { error } = await supabase.storage
          .from("management-review-evidence")
          .upload(path, draft.evidenceFile, { upsert: true });

        if (error) {
          toast.error(`No se pudo subir la evidencia de la entrada ${section.title}.`);
          hasErrors = true;
          continue;
        }

        evidenceLink = path;
      }

      const payload = {
        session_id: selectedSessionId,
        input_key: section.key,
        summary_snapshot: inputSummaries[section.key] ?? {},
        management_comment: draft.comment || null,
        evidence_link: evidenceLink,
        updated_at: new Date().toISOString(),
      };

      const response = existing
        ? await supabase
            .from("management_review_inputs")
            .update(payload)
            .eq("id", existing.id)
        : await supabase.from("management_review_inputs").insert({
            ...payload,
            created_at: new Date().toISOString(),
          });

      if (response.error) {
        toast.error(`No se pudo guardar la entrada ${section.title}.`);
        hasErrors = true;
      }
    }

    if (!hasErrors) {
      toast.success("Entradas guardadas.");
    }

    const updatedInputs = await loadInputs(selectedSessionId);
    setInputs(updatedInputs);
    await loadSessionMatrixDetails();
    setInputDrafts((prev) => {
      const updatedDrafts = { ...prev };
      inputSections.forEach((section) => {
        if (updatedDrafts[section.key]) {
          updatedDrafts[section.key] = {
            ...updatedDrafts[section.key],
            evidenceFile: null,
          };
        }
      });
      return updatedDrafts;
    });
    setSavingInputs(false);
  };

  const handleOpenOutputDialog = (output?: ManagementReviewOutput) => {
    if (output) {
      setEditingOutput(output);
      setOutputForm({
        output_key: output.output_key ?? "conclusions",
        decision_text: output.decision_text ?? "",
        responsible: output.responsible ?? "",
        due_date: output.due_date ?? "",
        action_link: output.action_link ?? "",
        status: output.status ?? "open",
      });
    } else {
      setEditingOutput(null);
      setOutputForm(emptyOutputDraft());
    }
    setOutputEvidenceFile(null);
    setOutputDialogOpen(true);
  };

  const handleSaveOutput = async () => {
    if (!selectedSessionId) return;
    if (!outputForm.decision_text.trim()) {
      toast.error("Describe la decisión o conclusión.");
      return;
    }

    let evidenceLink = editingOutput?.evidence_link ?? null;

    if (outputEvidenceFile) {
      const path = `sessions/${selectedSessionId}/outputs-${Date.now()}-${outputEvidenceFile.name}`;
      const { error } = await supabase.storage
        .from("management-review-evidence")
        .upload(path, outputEvidenceFile, { upsert: true });

      if (error) {
        toast.error("No se pudo subir la evidencia de salida.");
        return;
      }

      evidenceLink = path;
    }

    const payload = {
      session_id: selectedSessionId,
      output_key: outputForm.output_key,
      decision_text: outputForm.decision_text,
      responsible: outputForm.responsible || null,
      due_date: outputForm.due_date || null,
      action_link: outputForm.action_link || null,
      status: outputForm.status || "open",
      evidence_link: evidenceLink,
      updated_at: new Date().toISOString(),
    };

    const response = editingOutput
      ? await supabase
          .from("management_review_outputs")
          .update(payload)
          .eq("id", editingOutput.id)
      : await supabase.from("management_review_outputs").insert({
          ...payload,
          created_at: new Date().toISOString(),
        });

    if (response.error) {
      toast.error("No se pudo guardar la salida.");
      return;
    }

    toast.success("Salida registrada.");
    setOutputDialogOpen(false);
    setEditingOutput(null);
    const updatedOutputs = await loadOutputs(selectedSessionId);
    setOutputs(updatedOutputs);
    await loadSessionMatrixDetails();    
  };

  const actionsTable = useMemo(() => {
    return outputs.map((output) => {
      const section = outputSections.find((entry) => entry.key === output.output_key);
      return {
        ...output,
        title: section?.title ?? output.output_key,
      };
    });
  }, [outputs]);

  const escapeHtml = (value: string) =>
    value
      .replace(/&/g, "&amp;")
      .replace(/</g, "&lt;")
      .replace(/>/g, "&gt;")
      .replace(/"/g, "&quot;")
      .replace(/'/g, "&#039;");

  const handlePrint = () => {
    if (typeof window === "undefined") return;
    if (!selectedSession) {
      toast.error("Selecciona una sesión para imprimir el acta.");
      return;
    }

    const printWindow = window.open("", "print");
    if (!printWindow) return;

    const inputsRows = inputSections
      .map((section) => {
        const entry = inputs.find((item) => item.input_key === section.key);
        const comment = entry?.management_comment?.trim() || "Sin comentario";
        return `
          <tr>
            <td>${escapeHtml(section.title)}</td>
            <td>${escapeHtml(comment)}</td>
          </tr>
        `;
      })
      .join("");

    const outputsRows = outputs.length
      ? outputs
          .map((output) => {
            const section = outputSections.find((entry) => entry.key === output.output_key);
            return `
              <tr>
                <td>${escapeHtml(section?.title ?? output.output_key)}</td>
                <td>${escapeHtml(output.decision_text ?? "")}</td>
              </tr>
            `;
          })
          .join("")
      : `
          <tr>
            <td colspan="2">Sin salidas registradas.</td>
          </tr>
        `;

    printWindow.document.write(`
      <html>
        <head>
          <title>Acta de revisión - ${escapeHtml(
            selectedSession.session_code || "Sesión"
          )}</title>
          <style>
            body { font-family: Arial, sans-serif; padding: 24px; color: #111827; }
            h1 { font-size: 20px; margin-bottom: 4px; }
            h2 { font-size: 16px; margin-top: 24px; }
            p { margin: 0 0 8px; font-size: 12px; }
            table { width: 100%; border-collapse: collapse; margin-top: 8px; }
            th, td { border: 1px solid #e5e7eb; padding: 8px; font-size: 12px; text-align: left; }
            th { background: #f9fafb; }
          </style>
        </head>
        <body>
          <h1>Acta de revisión por la dirección</h1>
          <p><strong>Sesión:</strong> ${escapeHtml(selectedSession.session_code || "Sesión")}</p>
          <p><strong>Fecha:</strong> ${escapeHtml(formatDate(selectedSession.review_date))}</p>
          <p><strong>Periodo:</strong> ${escapeHtml(
            formatPeriod(selectedSession.period_start, selectedSession.period_end)
          )}</p>
          <p><strong>Responsable:</strong> ${escapeHtml(
            selectedSession.prepared_by || "Sin asignar"
          )}</p>

          <h2>Matriz de entradas</h2>
          <table>
            <thead>
              <tr>
                <th>Entrada</th>
                <th>Comentario de la dirección</th>
              </tr>
            </thead>
            <tbody>
              ${inputsRows}
            </tbody>
          </table>

          <h2>Matriz de salidas</h2>
          <table>
            <thead>
              <tr>
                <th>Salida</th>
                <th>Descripción</th>
              </tr>
            </thead>
            <tbody>
              ${outputsRows}
            </tbody>
          </table>
        </body>
      </html>
    `);
    printWindow.document.close();
    printWindow.focus();
    printWindow.print();
    printWindow.close();    
  };

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-8 bg-gradient-to-br from-slate-50 via-white to-emerald-50 pb-20">
        <section className="rounded-3xl border border-emerald-100 bg-white px-6 py-8 shadow-sm sm:px-10">
          <div className="flex flex-col gap-6 lg:flex-row lg:items-center lg:justify-between">
            <div className="space-y-3">
              <p className="text-xs font-semibold uppercase tracking-[0.4em] text-emerald-600">
                Evaluación
              </p>
              <h1 className="text-3xl font-bold text-gray-900 sm:text-4xl">
                Revision por la Dirección
              </h1>
              <p className="max-w-3xl text-sm text-gray-600 sm:text-base">
                Genera evidencia documentada de la revisión por la dirección, integrando
                automáticamente los indicadores del SGI y registrando decisiones, acciones y
                seguimiento.
              </p>
            </div>
            <div className="flex flex-wrap gap-3">
              <Button variant="outline" onClick={handlePrint}>
                <FileDown className="mr-2 h-4 w-4" />
                Imprimir acta
              </Button>
              {canManage && (
                <Button onClick={() => handleOpenSessionDialog()}>
                  <Plus className="mr-2 h-4 w-4" />
                  Nueva sesión
                </Button>
              )}
            </div>
          </div>
        </section>

        <section className="grid gap-4 lg:grid-cols-[2fr,1fr]">
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
              <div>
                <CardTitle className="text-base font-semibold text-gray-900">
                  Sesión de revisión
                </CardTitle>
                <p className="text-sm text-gray-500">
                  Selecciona una sesión para capturar entradas, salidas y acciones derivadas.
                </p>
              </div>
              {canManage && selectedSession && (
                <Button variant="outline" onClick={() => handleOpenSessionDialog(selectedSession)}>
                  Editar sesión
                </Button>
              )}
            </CardHeader>
            <CardContent className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <Label>Sesiones disponibles</Label>
                <Select
                  value={selectedSessionId}
                  onValueChange={(value) => setSelectedSessionId(value)}
                  disabled={loading || sessions.length === 0}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona una sesión" />
                  </SelectTrigger>
                  <SelectContent>
                    {sessions.map((session) => (
                      <SelectItem key={session.id} value={session.id}>
                        {session.session_code || "Sesión"} · {formatDate(session.review_date)}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              {selectedSession && (
                <div className="space-y-2 rounded-2xl border border-emerald-100 bg-emerald-50/40 p-4">
                  <div className="flex items-center gap-2 text-sm text-gray-700">
                    <CalendarDays className="h-4 w-4 text-emerald-600" />
                    <span>Fecha: {formatDate(selectedSession.review_date)}</span>
                  </div>
                  <div className="flex items-center gap-2 text-sm text-gray-700">
                    <CalendarCheck className="h-4 w-4 text-emerald-600" />
                    <span>Periodo: {formatPeriod(selectedSession.period_start, selectedSession.period_end)}</span>
                  </div>
                  <div className="flex items-center gap-2 text-sm text-gray-700">
                    <NotebookPen className="h-4 w-4 text-emerald-600" />
                    <span>Responsable: {selectedSession.prepared_by || "Sin asignar"}</span>
                  </div>
                  <Badge className="w-fit bg-emerald-100 text-emerald-700">
                    {selectedSession.status === "closed" ? "Cerrada" : "Borrador"}
                  </Badge>
                </div>
              )}
            </CardContent>
          </Card>

          <Card className="border-emerald-100">
            <CardHeader>
              <CardTitle className="text-base font-semibold text-gray-900">Resumen ejecutivo</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center justify-between text-sm">
                <span className="text-gray-500">Entradas registradas</span>
                <span className="font-semibold text-gray-900">{sessionSummary.inputs}</span>
              </div>
              <div className="flex items-center justify-between text-sm">
                <span className="text-gray-500">Salidas registradas</span>
                <span className="font-semibold text-gray-900">{sessionSummary.outputs}</span>
              </div>
              <div className="flex items-center justify-between text-sm">
                <span className="text-gray-500">Acciones abiertas</span>
                <span className="font-semibold text-gray-900">{sessionSummary.openActions}</span>
              </div>
              <div className="flex items-center justify-between text-sm">
                <span className="text-gray-500">Acciones cerradas</span>
                <span className="font-semibold text-gray-900">{sessionSummary.closedActions}</span>
              </div>
            </CardContent>
          </Card>
        </section>
        
        <Tabs defaultValue="inputs" className="space-y-6">
          <TabsList className="grid w-full grid-cols-2 gap-2 md:w-[1100px] md:grid-cols-5">
            <TabsTrigger value="inputs">Entradas</TabsTrigger>
            <TabsTrigger value="outputs">Salidas</TabsTrigger>
            <TabsTrigger value="actions">Acciones y seguimiento</TabsTrigger>
            <TabsTrigger value="session-details">Comentarios y salidas</TabsTrigger>
            <TabsTrigger value="session-matrix">Matriz de sesiones</TabsTrigger>      
          </TabsList>

          <TabsContent value="inputs" className="space-y-6">
            <section className="flex flex-wrap items-center justify-between gap-4">
              <div>
                <h2 className="text-lg font-semibold text-gray-900">Entradas de la revisión</h2>
                <p className="text-sm text-gray-500">
                  Captura comentarios y evidencia para todas las entradas antes de guardar.
                </p>
              </div>
              {canManage && (
                <Button onClick={handleSaveAllInputs} disabled={savingInputs}>
                  {savingInputs ? "Guardando..." : "Guardar todas las entradas"}
                </Button>
              )}
            </section>            
            {inputSections.map((section) => {
              const summary = inputSummaries[section.key] ?? {};
              const snapshotItems = buildSnapshotItems(summary);
              const draft = inputDrafts[section.key];
              const existing = inputs.find((item) => item.input_key === section.key);

              return (
                <Card key={section.key} className="border-emerald-100">
                  <CardHeader className="space-y-2">
                    <div className="flex flex-wrap items-center justify-between gap-3">
                      <div>
                        <CardTitle className="text-base font-semibold text-gray-900">
                          {section.title}
                        </CardTitle>
                        <p className="text-sm text-gray-500">{section.description}</p>
                      </div>
                      <div className="flex flex-wrap gap-2">
                        {section.links?.map((link) => (
                          <Button key={link.href} variant="outline" size="sm" asChild>
                            <Link href={link.href}>{link.label}</Link>
                          </Button>
                        ))}
                      </div>
                    </div>
                  </CardHeader>
                  <CardContent className="space-y-6">
                    <div className="grid gap-3 md:grid-cols-2 lg:grid-cols-4">
                      {snapshotItems.map((item) => (
                        <div
                          key={item.label}
                          className="rounded-2xl border border-emerald-100 bg-emerald-50/40 p-3"
                        >
                          <p className="text-xs font-semibold uppercase text-gray-500">
                            {item.label}
                          </p>
                          <p className="text-lg font-semibold text-gray-900">{item.value}</p>
                        </div>
                      ))}
                    </div>

                    <div className="space-y-2">
                      <Label>Comentario / Observación de la dirección</Label>
                      <Textarea
                        value={draft?.comment ?? ""}
                        onChange={(event) =>
                          setInputDrafts((prev) => ({
                            ...prev,
                            [section.key]: {
                              ...prev[section.key],
                              comment: event.target.value,
                            },
                          }))
                        }
                        placeholder="Captura comentarios, conclusiones o decisiones sobre esta entrada."
                      />
                    </div>

                    <div className="grid gap-4 md:grid-cols-2">
                      <div className="space-y-2">
                        <Label>Adjuntar evidencia (opcional)</Label>
                        <Input
                          type="file"
                          onChange={(event) =>
                            setInputDrafts((prev) => ({
                              ...prev,
                              [section.key]: {
                                ...prev[section.key],
                                evidenceFile: event.target.files?.[0] ?? null,
                              },
                            }))
                          }
                        />
                      </div>
                      <div className="space-y-2">
                        <Label>Evidencia vinculada</Label>
                        {existing?.evidence_link ? (
                          <Button
                            variant="outline"
                            onClick={async () => {
                              const url = await getEvidenceUrl(existing.evidence_link);
                              if (url) window.open(url, "_blank");
                            }}
                          >
                            Ver evidencia
                          </Button>
                        ) : (
                          <p className="text-sm text-gray-500">Sin evidencia adjunta.</p>
                        )}
                      </div>
                    </div>

                    <div className="flex flex-wrap items-center justify-between gap-3">
                      <p className="text-xs text-gray-500">
                        Última actualización: {formatDate(existing?.updated_at ?? existing?.created_at)}
                      </p>
                    </div>
                  </CardContent>
                </Card>
              );
            })}
          </TabsContent>

          <TabsContent value="outputs" className="space-y-6">
            <section className="flex flex-wrap items-center justify-between gap-4">
              <div>
                <h2 className="text-lg font-semibold text-gray-900">Salidas de la revisión</h2>
                <p className="text-sm text-gray-500">
                  Registra decisiones, responsables y fechas objetivo para dar seguimiento.
                </p>
              </div>
              {canManage && (
                <Button onClick={() => handleOpenOutputDialog()}>
                  <Plus className="mr-2 h-4 w-4" />
                  Nueva salida
                </Button>
              )}
            </section>

            <Card className="border-emerald-100">
              <CardHeader>
                <CardTitle className="text-base font-semibold text-gray-900">Listado de salidas</CardTitle>
              </CardHeader>
              <CardContent className="overflow-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Tipo</TableHead>
                      <TableHead>Decisión</TableHead>
                      <TableHead>Responsable</TableHead>
                      <TableHead>Fecha objetivo</TableHead>
                      <TableHead>Estatus</TableHead>
                      <TableHead>Evidencia</TableHead>
                      {canManage && <TableHead>Acciones</TableHead>}
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {outputs.length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={canManage ? 7 : 6} className="text-center text-sm text-gray-500">
                          Sin salidas registradas.
                        </TableCell>
                      </TableRow>
                    ) : (
                      outputs.map((output) => {
                        const section = outputSections.find(
                          (entry) => entry.key === output.output_key
                        );
                        return (
                          <TableRow key={output.id}>
                            <TableCell className="text-sm font-medium text-gray-900">
                              {section?.title ?? output.output_key}
                            </TableCell>
                            <TableCell className="text-sm text-gray-600">
                              {output.decision_text}
                            </TableCell>
                            <TableCell className="text-sm text-gray-600">
                              {output.responsible || "Sin asignar"}
                            </TableCell>
                            <TableCell className="text-sm text-gray-600">
                              {formatDate(output.due_date)}
                            </TableCell>
                            <TableCell>
                              <Badge className="bg-emerald-100 text-emerald-700">
                                {outputStatusLabelMap[output.status ?? "open"] ?? "Abierta"}
                              </Badge>
                            </TableCell>
                            <TableCell>
                              {output.evidence_link ? (
                                <Button
                                  variant="outline"
                                  size="sm"
                                  onClick={async () => {
                                    const url = await getEvidenceUrl(output.evidence_link);
                                    if (url) window.open(url, "_blank");
                                  }}
                                >
                                  Ver
                                </Button>
                              ) : (
                                <span className="text-xs text-gray-500">Sin evidencia</span>
                              )}
                            </TableCell>
                            {canManage && (
                              <TableCell>
                                <Button
                                  variant="outline"
                                  size="sm"
                                  onClick={() => handleOpenOutputDialog(output)}
                                >
                                  Editar
                                </Button>
                              </TableCell>
                            )}
                          </TableRow>
                        );
                      })
                    )}
                  </TableBody>
                </Table>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="actions" className="space-y-6">
            <section className="grid gap-4 lg:grid-cols-3">
              <Card className="border-emerald-100">
                <CardHeader className="flex flex-row items-center justify-between space-y-0">
                  <CardTitle className="text-sm font-semibold text-gray-600">Acciones abiertas</CardTitle>
                  <Flag className="h-4 w-4 text-emerald-500" />
                </CardHeader>
                <CardContent>
                  <div className="text-3xl font-bold text-gray-900">{sessionSummary.openActions}</div>
                  <p className="text-xs text-gray-500">Salidas no cerradas</p>
                </CardContent>
              </Card>
              <Card className="border-emerald-100">
                <CardHeader className="flex flex-row items-center justify-between space-y-0">
                  <CardTitle className="text-sm font-semibold text-gray-600">Acciones cerradas</CardTitle>
                  <ShieldCheck className="h-4 w-4 text-emerald-500" />
                </CardHeader>
                <CardContent>
                  <div className="text-3xl font-bold text-gray-900">{sessionSummary.closedActions}</div>
                  <p className="text-xs text-gray-500">Confirmadas en la sesión</p>
                </CardContent>
              </Card>
              <Card className="border-emerald-100">
                <CardHeader className="flex flex-row items-center justify-between space-y-0">
                  <CardTitle className="text-sm font-semibold text-gray-600">Total de salidas</CardTitle>
                  <Layers className="h-4 w-4 text-emerald-500" />
                </CardHeader>
                <CardContent>
                  <div className="text-3xl font-bold text-gray-900">{sessionSummary.outputs}</div>
                  <p className="text-xs text-gray-500">Decisiones capturadas</p>
                </CardContent>
              </Card>
            </section>

            <Card className="border-emerald-100">
              <CardHeader className="flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
                <div>
                  <CardTitle className="text-base font-semibold text-gray-900">Tabla de seguimiento</CardTitle>
                  <p className="text-sm text-gray-500">
                    Da seguimiento a las decisiones convertidas en acciones y vincula planes
                  </p>
                </div>
              </CardHeader>
              <CardContent className="overflow-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Acción</TableHead>
                      <TableHead>Fuente</TableHead>
                      <TableHead>Responsable</TableHead>
                      <TableHead>Fecha compromiso</TableHead>
                      <TableHead>Estatus</TableHead>
                      <TableHead>Evidencia</TableHead>
                      <TableHead>Vínculo</TableHead>
                      <TableHead>Crear acción</TableHead>                      
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {actionsTable.length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={8} className="text-center text-sm text-gray-500">
                          Sin acciones derivadas registradas.
                        </TableCell>
                      </TableRow>
                    ) : (
                      actionsTable.map((action) => (
                        <TableRow key={action.id}>
                          <TableCell className="text-sm font-medium text-gray-900">
                            {action.decision_text}
                          </TableCell>
                          <TableCell className="text-sm text-gray-600">{action.title}</TableCell>
                          <TableCell className="text-sm text-gray-600">
                            {action.responsible || "Sin asignar"}
                          </TableCell>
                          <TableCell className="text-sm text-gray-600">
                            {formatDate(action.due_date)}
                          </TableCell>
                          <TableCell>
                            <Badge className="bg-emerald-100 text-emerald-700">
                              {outputStatusLabelMap[action.status ?? "open"] ?? "Abierta"}
                            </Badge>
                          </TableCell>
                          <TableCell>
                            {action.evidence_link ? (
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={async () => {
                                  const url = await getEvidenceUrl(action.evidence_link);
                                  if (url) window.open(url, "_blank");
                                }}
                              >
                                Ver
                              </Button>
                            ) : (
                              <span className="text-xs text-gray-500">Sin evidencia</span>
                            )}
                          </TableCell>
                          <TableCell>
                            {action.action_link ? (
                              <Button asChild variant="outline" size="sm">
                                <a href={action.action_link} target="_blank" rel="noreferrer">
                                  Abrir
                                </a>
                              </Button>
                            ) : (
                              <span className="text-xs text-gray-500">Sin vínculo</span>
                            )}
                          </TableCell>
                          <TableCell>
                            <Button
                              asChild
                              size="sm"
                              className="bg-sky-500 text-white hover:bg-sky-600"
                            >
                              <Link href={buildActionPlanHref(action)}>
                                Crear acción
                              </Link>
                            </Button>
                          </TableCell>                          
                        </TableRow>
                      ))
                    )}
                  </TableBody>
                </Table>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="session-details" className="space-y-6">
            <section className="space-y-2">
              <div>
                <h2 className="text-lg font-semibold text-gray-900">
                  Matriz de sesiones con comentarios y salidas
                </h2>
                <p className="text-sm text-gray-500">
                  Consulta los comentarios de entradas y las descripciones de salidas por cada sesión
                  guardada.
                </p>
              </div>
            </section>
            <div className="space-y-4">
              {sessions.length === 0 ? (
                <Card className="border-emerald-100">
                  <CardContent className="py-10 text-center text-sm text-gray-500">
                    No hay sesiones registradas.
                  </CardContent>
                </Card>
              ) : (
                sessions.map((session) => {
                  const sessionInputs = sessionMatrixDetails[session.id]?.inputs ?? {};
                  const sessionOutputs = sessionMatrixDetails[session.id]?.outputs ?? [];

                  return (
                    <Card key={session.id} className="border-emerald-100">
                      <CardHeader className="space-y-2">
                        <CardTitle className="text-base font-semibold text-gray-900">
                          {session.session_code || "Sesión"} · {formatDate(session.review_date)}
                        </CardTitle>
                        <p className="text-sm text-gray-500">
                          Periodo: {formatPeriod(session.period_start, session.period_end)}
                        </p>
                      </CardHeader>
                      <CardContent className="grid gap-4 lg:grid-cols-2">
                        <div className="space-y-3">
                          <h3 className="text-sm font-semibold text-gray-700">Comentarios de entradas</h3>
                          <Table>
                            <TableHeader>
                              <TableRow>
                                <TableHead>Entrada</TableHead>
                                <TableHead>Comentario</TableHead>
                              </TableRow>
                            </TableHeader>
                            <TableBody>
                              {inputSections.map((section) => {
                                const entry = sessionInputs[section.key];
                                return (
                                  <TableRow key={`${session.id}-input-${section.key}`}>
                                    <TableCell className="text-sm font-medium text-gray-900">
                                      {section.title}
                                    </TableCell>
                                    <TableCell className="text-sm text-gray-600">
                                      {entry?.management_comment || "Sin comentario."}
                                    </TableCell>
                                  </TableRow>
                                );
                              })}
                            </TableBody>
                          </Table>
                        </div>

                        <div className="space-y-3">
                          <h3 className="text-sm font-semibold text-gray-700">Salidas registradas</h3>
                          <Table>
                            <TableHeader>
                              <TableRow>
                                <TableHead>Salida</TableHead>
                                <TableHead>Descripción</TableHead>
                              </TableRow>
                            </TableHeader>
                            <TableBody>
                              {sessionOutputs.length === 0 ? (
                                <TableRow>
                                  <TableCell colSpan={2} className="text-sm text-gray-500">
                                    Sin salidas registradas.
                                  </TableCell>
                                </TableRow>
                              ) : (
                                sessionOutputs.map((output) => {
                                  const outputSection = outputSections.find(
                                    (entry) => entry.key === output.output_key
                                  );
                                  return (
                                    <TableRow key={`${session.id}-output-${output.id}`}>
                                      <TableCell className="text-sm font-medium text-gray-900">
                                        {outputSection?.title ?? output.output_key}
                                      </TableCell>
                                      <TableCell className="text-sm text-gray-600">
                                        {output.decision_text || "Sin descripción."}
                                      </TableCell>
                                    </TableRow>
                                  );
                                })
                              )}
                            </TableBody>
                          </Table>
                        </div>
                      </CardContent>
                    </Card>
                  );
                })
              )}
            </div>
          </TabsContent>

          <TabsContent value="session-matrix" className="space-y-6">
            <section className="space-y-2">
              <div>
                <h2 className="text-lg font-semibold text-gray-900">Matriz de sesiones registradas</h2>
                <p className="text-sm text-gray-500">
                  Consulta todas las sesiones y administra los registros disponibles.
                </p>
              </div>
            </section>
            <Card className="border-emerald-100">
              <CardHeader>
                <CardTitle className="text-base font-semibold text-gray-900">Listado de sesiones</CardTitle>
              </CardHeader>
              <CardContent className="overflow-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Código</TableHead>
                      <TableHead>Fecha</TableHead>
                      <TableHead>Periodo</TableHead>
                      <TableHead>Responsable</TableHead>
                      <TableHead>Estatus</TableHead>
                      {canManage && <TableHead>Acciones</TableHead>}
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {sessions.length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={canManage ? 6 : 5} className="text-center text-sm text-gray-500">
                          No hay sesiones registradas.
                        </TableCell>
                      </TableRow>
                    ) : (
                      sessions.map((session) => (
                        <TableRow key={`matrix-${session.id}`}>
                          <TableCell className="text-sm font-medium text-gray-900">
                            {session.session_code || "Sesión"}
                          </TableCell>
                          <TableCell className="text-sm text-gray-600">
                            {formatDate(session.review_date)}
                          </TableCell>
                          <TableCell className="text-sm text-gray-600">
                            {formatPeriod(session.period_start, session.period_end)}
                          </TableCell>
                          <TableCell className="text-sm text-gray-600">
                            {session.prepared_by || "Sin asignar"}
                          </TableCell>
                          <TableCell>
                            <Badge className="bg-emerald-100 text-emerald-700">
                              {session.status === "closed" ? "Cerrada" : "Borrador"}
                            </Badge>
                          </TableCell>
                          {canManage && (
                            <TableCell>
                              <div className="flex flex-wrap gap-2">
                                <Button
                                  variant="outline"
                                  size="sm"
                                  onClick={() => handleOpenSessionDialog(session)}
                                >
                                  Editar
                                </Button>
                                <Button
                                  variant="destructive"
                                  size="sm"
                                  disabled={deletingSessionId === session.id}
                                  onClick={() => handleDeleteSession(session)}
                                >
                                  {deletingSessionId === session.id ? "Eliminando..." : "Eliminar"}
                                </Button>
                              </div>
                            </TableCell>
                          )}
                        </TableRow>
                      ))
                    )}
                  </TableBody>
                </Table>
              </CardContent>
            </Card>
          </TabsContent>               
        </Tabs>
      </div>

      <Dialog open={sessionDialogOpen} onOpenChange={setSessionDialogOpen}>
        <DialogContent className="max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>
              {editingSession ? "Editar sesión de revisión" : "Nueva sesión de revisión"}
            </DialogTitle>
          </DialogHeader>
          <div className="space-y-4">
            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <Label>Código / ID de sesión</Label>
                <Input
                  value={sessionForm.session_code}
                  onChange={(event) =>
                    setSessionForm((prev) => ({ ...prev, session_code: event.target.value }))
                  }
                  placeholder="MR-2026-Q1"
                />
              </div>
              <div className="space-y-2">
                <Label>Fecha de revisión</Label>
                <Input
                  type="date"
                  value={sessionForm.review_date}
                  onChange={(event) =>
                    setSessionForm((prev) => ({ ...prev, review_date: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label>Periodo inicio</Label>
                <Input
                  type="date"
                  value={sessionForm.period_start}
                  onChange={(event) =>
                    setSessionForm((prev) => ({ ...prev, period_start: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label>Periodo fin</Label>
                <Input
                  type="date"
                  value={sessionForm.period_end}
                  onChange={(event) =>
                    setSessionForm((prev) => ({ ...prev, period_end: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label>Responsable de elaboración</Label>
                <Input
                  value={sessionForm.prepared_by}
                  onChange={(event) =>
                    setSessionForm((prev) => ({ ...prev, prepared_by: event.target.value }))
                  }
                  placeholder="Nombre del responsable"
                />
              </div>
              <div className="space-y-2">
                <Label>Estado de la sesión</Label>
                <Select
                  value={sessionForm.status}
                  onValueChange={(value) => setSessionForm((prev) => ({ ...prev, status: value }))}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona estatus" />
                  </SelectTrigger>
                  <SelectContent>
                    {sessionStatusOptions.map((option) => (
                      <SelectItem key={option.value} value={option.value}>
                        {option.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div className="space-y-2">
              <Label>Participantes</Label>
              <Textarea
                value={sessionForm.participants}
                onChange={(event) =>
                  setSessionForm((prev) => ({ ...prev, participants: event.target.value }))
                }
                placeholder="Lista de participantes y roles"
              />
            </div>

            <div className="space-y-2">
              <Label>Minuta / Conclusiones generales</Label>
              <Textarea
                value={sessionForm.general_conclusions}
                onChange={(event) =>
                  setSessionForm((prev) => ({ ...prev, general_conclusions: event.target.value }))
                }
                placeholder="Resumen ejecutivo y conclusiones generales"
              />
            </div>

            <div className="space-y-2">
              <Label>Implicaciones estratégicas</Label>
              <Textarea
                value={sessionForm.strategic_implications}
                onChange={(event) =>
                  setSessionForm((prev) => ({ ...prev, strategic_implications: event.target.value }))
                }
                placeholder="Implicaciones para la dirección estratégica"
              />
            </div>
          </div>
          <DialogFooter className="gap-2 sm:justify-end">
            <Button variant="outline" onClick={() => setSessionDialogOpen(false)}>
              Cancelar
            </Button>
            <Button onClick={handleSaveSession}>Guardar sesión</Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <Dialog open={outputDialogOpen} onOpenChange={setOutputDialogOpen}>
        <DialogContent className="max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>{editingOutput ? "Editar salida" : "Nueva salida"}</DialogTitle>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <Label>Tipo de salida</Label>
              <Select
                value={outputForm.output_key}
                onValueChange={(value) => setOutputForm((prev) => ({ ...prev, output_key: value }))}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Selecciona el tipo" />
                </SelectTrigger>
                <SelectContent>
                  {outputSections.map((option) => (
                    <SelectItem key={option.key} value={option.key}>
                      {option.title}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label>Decisión / conclusión</Label>
              <Textarea
                value={outputForm.decision_text}
                onChange={(event) =>
                  setOutputForm((prev) => ({ ...prev, decision_text: event.target.value }))
                }
                placeholder="Describe la decisión tomada por la dirección"
              />
            </div>

            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <Label>Responsable</Label>
                <Input
                  value={outputForm.responsible}
                  onChange={(event) =>
                    setOutputForm((prev) => ({ ...prev, responsible: event.target.value }))
                  }
                  placeholder="Nombre / área"
                />
              </div>
              <div className="space-y-2">
                <Label>Fecha objetivo</Label>
                <Input
                  type="date"
                  value={outputForm.due_date}
                  onChange={(event) =>
                    setOutputForm((prev) => ({ ...prev, due_date: event.target.value }))
                  }
                />
              </div>
            </div>

            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <Label>Estatus</Label>
                <Select
                  value={outputForm.status}
                  onValueChange={(value) => setOutputForm((prev) => ({ ...prev, status: value }))}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona estatus" />
                  </SelectTrigger>
                  <SelectContent>
                    {outputStatusOptions.map((option) => (
                      <SelectItem key={option.value} value={option.value}>
                        {option.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Vínculo a acción / mejora</Label>
                <Input
                  value={outputForm.action_link}
                  onChange={(event) =>
                    setOutputForm((prev) => ({ ...prev, action_link: event.target.value }))
                  }
                  placeholder="URL o referencia"
                />
              </div>
            </div>

            <div className="space-y-2">
              <Label>Evidencia</Label>
              <Input type="file" onChange={(event) => setOutputEvidenceFile(event.target.files?.[0] ?? null)} />
            </div>
          </div>
          <DialogFooter className="gap-2 sm:justify-end">
            <Button variant="outline" onClick={() => setOutputDialogOpen(false)}>
              Cancelar
            </Button>
            <Button onClick={handleSaveOutput}>Guardar salida</Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </SGIDashboardLayout>
  );
}