"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import Link from "next/link";
import type { User } from "@supabase/supabase-js";
import {
  ArrowUpRight,
  CalendarDays,
  ClipboardList,
  FileSearch,
  Filter,
  LineChart,
  ListChecks,
  Plus,
  RefreshCcw,
  ShieldCheck,
  Target,
} from "lucide-react";
import { toast } from "sonner";

import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Dialog,
  DialogContent,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Drawer,
  DrawerClose,
  DrawerContent,
  DrawerFooter,
  DrawerHeader,
  DrawerTitle,
} from "@/components/ui/drawer";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Textarea } from "@/components/ui/textarea";
import { supabase } from "@/lib/supabase";

const allowedManagers = new Set([
  "ehs.mexico@meiwa.com.mx",
  "auxiliar.ehs@meiwa.com.mx",
]);

const sourceOptions = [
  { value: "corrective_action", label: "Acciones correctivas (10.2)" },
  { value: "audit", label: "Auditoría (interna/externa)" },
  { value: "indicator", label: "Indicador / Objetivo / Meta" },
  { value: "incident", label: "Incidente / Inspección" },
  { value: "system_change", label: "Cambio al sistema / Control operacional" },
  { value: "document_control", label: "Control documental" },
  { value: "management_review", label: "Revisión por la Dirección (9.3)" },
  { value: "manual", label: "Manual / Otro" },
];

const statusOptions = [
  { value: "planned", label: "Planeada" },
  { value: "in_progress", label: "En proceso" },
  { value: "implemented", label: "Implementada" },
  { value: "verified", label: "Verificada" },
  { value: "closed", label: "Cerrada" },
];

const statusBadgeStyles: Record<string, string> = {
  planned: "bg-sky-100 text-sky-700",
  in_progress: "bg-amber-100 text-amber-700",
  implemented: "bg-indigo-100 text-indigo-700",
  verified: "bg-emerald-100 text-emerald-700",
  closed: "bg-emerald-200 text-emerald-800",
};

const statusLabelMap = statusOptions.reduce<Record<string, string>>((acc, option) => {
  acc[option.value] = option.label;
  return acc;
}, {});

const sourceLabelMap = sourceOptions.reduce<Record<string, string>>((acc, option) => {
  acc[option.value] = option.label;
  return acc;
}, {});

const sourceCards = [
  {
    id: "corrective_action",
    title: "Acciones correctivas (10.2)",
    description:
      "Hallazgos y no conformidades que se convierten en mejoras del SGA con trazabilidad.",
    href: "/meiwa-core/meiwa-sgi/improvement/nonconformity-corrective-action",
    icon: ClipboardList,
    sourceType: "corrective_action",
  },
  {
    id: "audit",
    title: "Auditorías internas/externas",
    description: "Oportunidades, observaciones y no conformidades capturadas en auditorías.",
    href: "/meiwa-core/meiwa-sgi/evaluation/internal-audit",
    icon: FileSearch,
    sourceType: "audit",
  },
  {
    id: "indicator",
    title: "Objetivos e indicadores",
    description: "Seguimiento de metas, contramedidas y ajustes a indicadores ambientales.",
    href: "/meiwa-core/meiwa-sgi/planning/environmental-indicators",
    icon: LineChart,
    sourceType: "indicator",
  },
  {
    id: "document_control",
    title: "Control documental",
    description: "Cambios en procedimientos y controles operacionales del SGA.",
    href: "/meiwa-core/meiwa-sgi/support/document-control",
    icon: ListChecks,
    sourceType: "document_control",
  },
  {
    id: "management_review",
    title: "Revisión por la Dirección (9.3)",
    description: "Decisiones, recursos y ajustes estratégicos derivados de la revisión gerencial.",
    href: "/meiwa-core/meiwa-sgi/leadership",
    icon: ShieldCheck,
    sourceType: "management_review",
  },
  {
    id: "manual",
    title: "Mejoras manuales",
    description: "Registro directo de mejoras adicionales o cambios emergentes en el SGA.",
    href: null,
    icon: RefreshCcw,
    sourceType: "manual",
  },
];

type ImprovementRecord = {
  id: string;
  date: string | null;
  source_type: string | null;
  description: string | null;
  action_implemented: string | null;
  responsible: string | null;
  target_date: string | null;
  close_date: string | null;
  evidence_url: string | null;
  expected_result: string | null;
  achieved_result: string | null;
  status: string | null;
  source_table: string | null;
  source_id: string | null;
  created_at?: string | null;
  updated_at?: string | null;
  created_by?: string | null;
  updated_by?: string | null;
};

type FormState = {
  date: string;
  source_type: string;
  description: string;
  action_implemented: string;
  responsible: string;
  target_date: string;
  close_date: string;
  evidence_url: string;
  expected_result: string;
  achieved_result: string;
  status: string;
  source_table: string;
  source_id: string;
};

const buildEmptyFormState = (): FormState => ({
  date: "",
  source_type: "",
  description: "",
  action_implemented: "",
  responsible: "",
  target_date: "",
  close_date: "",
  evidence_url: "",
  expected_result: "",
  achieved_result: "",
  status: "planned",
  source_table: "",
  source_id: "",
});

const formatDate = (value?: string | null) => {
  if (!value) return "Sin fecha";
  const parsed = new Date(value);
  if (Number.isNaN(parsed.getTime())) return "Sin fecha";
  return parsed.toLocaleDateString("es-MX");
};

const formatDateInput = (value?: string | null) => {
  if (!value) return "";
  return new Date(value).toISOString().split("T")[0] ?? "";
};

const isHttpUrl = (value?: string | null) => {
  if (!value) return false;
  return value.startsWith("http://") || value.startsWith("https://");
};

const getSourceLabel = (value?: string | null) => {
  if (!value) return "Sin especificar";
  return sourceLabelMap[value] ?? value;
};

const getStatusLabel = (value?: string | null) => {
  if (!value) return "Sin estado";
  return statusLabelMap[value] ?? value;
};

export default function ContinuousImprovementPage() {
  const [records, setRecords] = useState<ImprovementRecord[]>([]);
  const [loading, setLoading] = useState(true);
  const [user, setUser] = useState<User | null>(null);
  const [canManage, setCanManage] = useState(false);
  const [filtersOpen, setFiltersOpen] = useState(false);
  const [sourceFilter, setSourceFilter] = useState("all");
  const [statusFilter, setStatusFilter] = useState("all");
  const [responsibleFilter, setResponsibleFilter] = useState("");
  const [searchFilter, setSearchFilter] = useState("");
  const [dateFrom, setDateFrom] = useState("");
  const [dateTo, setDateTo] = useState("");
  const [dialogOpen, setDialogOpen] = useState(false);
  const [formState, setFormState] = useState<FormState>(buildEmptyFormState);
  const [editingRecord, setEditingRecord] = useState<ImprovementRecord | null>(null);
  const [evidenceFile, setEvidenceFile] = useState<File | null>(null);
  const [drawerOpen, setDrawerOpen] = useState(false);
  const [activeRecord, setActiveRecord] = useState<ImprovementRecord | null>(null);
  const [deleteTarget, setDeleteTarget] = useState<ImprovementRecord | null>(null);
  const [evidencePreviewUrl, setEvidencePreviewUrl] = useState<string | null>(null);

  useEffect(() => {
    const loadSession = async () => {
      const { data } = await supabase.auth.getSession();
      const currentUser = data.session?.user ?? null;
      setUser(currentUser);
      setCanManage(allowedManagers.has(currentUser?.email ?? ""));
    };

    loadSession();

    const { data: subscription } = supabase.auth.onAuthStateChange((_event, session) => {
      setUser(session?.user ?? null);
      setCanManage(allowedManagers.has(session?.user?.email ?? ""));
    });

    return () => {
      subscription.subscription.unsubscribe();
    };
  }, []);

  const fetchRecords = useCallback(async () => {
    setLoading(true);
    const { data, error } = await supabase
      .from("continuous_improvement_register")
      .select("*")
      .order("date", { ascending: false });

    if (error) {
      toast.error("No pudimos cargar el registro de mejora continua.");
      setLoading(false);
      return;
    }

    setRecords(data ?? []);
    setLoading(false);
  }, []);

  useEffect(() => {
    if (user) {
      fetchRecords();
    }
  }, [user, fetchRecords]);

  useEffect(() => {
    const loadEvidencePreview = async () => {
      if (!activeRecord?.evidence_url) {
        setEvidencePreviewUrl(null);
        return;
      }

      if (isHttpUrl(activeRecord.evidence_url)) {
        setEvidencePreviewUrl(activeRecord.evidence_url);
        return;
      }

      const { data, error } = await supabase.storage
        .from("continuous-improvement-evidence")
        .createSignedUrl(activeRecord.evidence_url, 60 * 60);

      if (error) {
        setEvidencePreviewUrl(null);
        return;
      }

      setEvidencePreviewUrl(data.signedUrl);
    };

    if (drawerOpen) {
      loadEvidencePreview();
    }
  }, [activeRecord, drawerOpen]);

  const handleOpenDialog = (record?: ImprovementRecord) => {
    if (record) {
      setEditingRecord(record);
      setFormState({
        date: formatDateInput(record.date),
        source_type: record.source_type ?? "",
        description: record.description ?? "",
        action_implemented: record.action_implemented ?? "",
        responsible: record.responsible ?? "",
        target_date: formatDateInput(record.target_date),
        close_date: formatDateInput(record.close_date),
        evidence_url: record.evidence_url ?? "",
        expected_result: record.expected_result ?? "",
        achieved_result: record.achieved_result ?? "",
        status: record.status ?? "planned",
        source_table: record.source_table ?? "",
        source_id: record.source_id ?? "",
      });
    } else {
      setEditingRecord(null);
      setFormState(buildEmptyFormState());
    }

    setEvidenceFile(null);
    setDialogOpen(true);
  };

  const handleSubmit = async () => {
    if (!user) return;

    if (!formState.source_type) {
      toast.error("Selecciona una fuente de mejora.");
      return;
    }

    if (!formState.description.trim()) {
      toast.error("Describe la mejora registrada.");
      return;
    }

    let evidenceValue = formState.evidence_url.trim();

    if (evidenceFile) {
      const filePath = `continuous-improvement/${user.id}/${Date.now()}-${evidenceFile.name}`;
      const { error } = await supabase.storage
        .from("continuous-improvement-evidence")
        .upload(filePath, evidenceFile, { upsert: true });

      if (error) {
        toast.error("No pudimos cargar la evidencia en Supabase Storage.");
        return;
      }

      evidenceValue = filePath;
    }

    const payload = {
      date: formState.date || null,
      source_type: formState.source_type,
      description: formState.description,
      action_implemented: formState.action_implemented || null,
      responsible: formState.responsible || null,
      target_date: formState.target_date || null,
      close_date: formState.close_date || null,
      evidence_url: evidenceValue || null,
      expected_result: formState.expected_result || null,
      achieved_result: formState.achieved_result || null,
      status: formState.status || "planned",
      source_table: formState.source_table || null,
      source_id: formState.source_id || null,
      updated_at: new Date().toISOString(),
      updated_by: user.email ?? user.id,
    };

    if (editingRecord) {
      const { error } = await supabase
        .from("continuous_improvement_register")
        .update(payload)
        .eq("id", editingRecord.id);

      if (error) {
        toast.error("No pudimos actualizar la mejora.");
        return;
      }

      toast.success("Mejora continua actualizada.");
    } else {
      const { error } = await supabase.from("continuous_improvement_register").insert({
        ...payload,
        created_at: new Date().toISOString(),
        created_by: user.email ?? user.id,
      });

      if (error) {
        toast.error("No pudimos registrar la mejora.");
        return;
      }

      toast.success("Mejora continua registrada.");
    }

    setDialogOpen(false);
    setEditingRecord(null);
    setEvidenceFile(null);
    await fetchRecords();
  };

  const handleDelete = async () => {
    if (!deleteTarget) return;

    const { error } = await supabase
      .from("continuous_improvement_register")
      .delete()
      .eq("id", deleteTarget.id);

    if (error) {
      toast.error("No pudimos eliminar la mejora.");
      return;
    }

    toast.success("Mejora eliminada del registro.");
    setDeleteTarget(null);
    setDrawerOpen(false);
    await fetchRecords();
  };

  const filteredRecords = useMemo(() => {
    return records.filter((record) => {
      if (sourceFilter !== "all" && record.source_type !== sourceFilter) {
        return false;
      }
      if (statusFilter !== "all" && record.status !== statusFilter) {
        return false;
      }
      if (
        responsibleFilter &&
        !record.responsible?.toLowerCase().includes(responsibleFilter.toLowerCase())
      ) {
        return false;
      }
      if (searchFilter) {
        const haystack = [
          record.description,
          record.action_implemented,
          record.expected_result,
          record.achieved_result,
        ]
          .filter(Boolean)
          .join(" ")
          .toLowerCase();
        if (!haystack.includes(searchFilter.toLowerCase())) {
          return false;
        }
      }
      if (dateFrom) {
        const recordDate = record.date ?? record.created_at;
        if (recordDate && new Date(recordDate) < new Date(dateFrom)) {
          return false;
        }
      }
      if (dateTo) {
        const recordDate = record.date ?? record.created_at;
        if (recordDate && new Date(recordDate) > new Date(dateTo)) {
          return false;
        }
      }
      return true;
    });
  }, [records, sourceFilter, statusFilter, responsibleFilter, searchFilter, dateFrom, dateTo]);

  const kpiMetrics = useMemo(() => {
    const total = records.length;
    const statusCounts = records.reduce<Record<string, number>>((acc, record) => {
      const key = record.status ?? "unknown";
      acc[key] = (acc[key] ?? 0) + 1;
      return acc;
    }, {});

    const closedCount = statusCounts.closed ?? 0;
    const inProgressCount = statusCounts.in_progress ?? 0;
    const plannedCount = statusCounts.planned ?? 0;
    const implementedCount = statusCounts.implemented ?? 0;

    const now = new Date();
    const periodStart = new Date();
    periodStart.setDate(now.getDate() - 30);

    const periodCount = records.filter((record) => {
      const dateValue = record.date ?? record.created_at;
      if (!dateValue) return false;
      return new Date(dateValue) >= periodStart;
    }).length;

    const closeDurations = records
      .filter((record) => record.close_date && record.date)
      .map((record) => {
        const start = new Date(record.date ?? "");
        const end = new Date(record.close_date ?? "");
        if (Number.isNaN(start.getTime()) || Number.isNaN(end.getTime())) return null;
        return Math.max(0, end.getTime() - start.getTime());
      })
      .filter((duration): duration is number => duration !== null);

    const averageCloseDays = closeDurations.length
      ? Math.round(
          closeDurations.reduce((acc, duration) => acc + duration, 0) /
            closeDurations.length /
            (1000 * 60 * 60 * 24)
        )
      : null;

    const sourceCounts = records.reduce<Record<string, number>>((acc, record) => {
      const key = record.source_type ?? "unknown";
      acc[key] = (acc[key] ?? 0) + 1;
      return acc;
    }, {});

    const topSources = Object.entries(sourceCounts)
      .sort((a, b) => b[1] - a[1])
      .slice(0, 3)
      .map(([key, value]) => ({ key, value }));

    return {
      total,
      periodCount,
      closedCount,
      inProgressCount,
      plannedCount,
      implementedCount,
      averageCloseDays,
      topSources,
    };
  }, [records]);

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-10 bg-gradient-to-br from-slate-50 via-white to-emerald-50 pb-20">
        <section className="relative overflow-hidden rounded-3xl border border-emerald-100 bg-white px-6 py-10 shadow-sm sm:px-10">
          <div className="absolute inset-0 bg-gradient-to-r from-emerald-500/10 to-lime-400/10" />
          <div className="relative space-y-4">
            <p className="text-xs font-semibold uppercase tracking-[0.5em] text-emerald-600">
              Registro integrador ISO 14001: 10.3
            </p>
            <h1 className="text-3xl font-bold text-gray-900 sm:text-4xl">Continuous Improvement</h1>
            <p className="max-w-4xl text-base text-gray-600 sm:text-lg">
              Consolida evidencia de mejora continua del SGA, trazando el origen, la acción, la
              evidencia y los resultados esperados y logrados.
            </p>
          </div>
        </section>

        <section className="grid gap-6 lg:grid-cols-5">
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between space-y-0">
              <CardTitle className="text-sm font-semibold text-gray-600">Mejoras registradas</CardTitle>
              <RefreshCcw className="h-4 w-4 text-emerald-500" />
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-gray-900">{kpiMetrics.total}</div>
              <p className="text-xs text-gray-500">Últimos 30 días: {kpiMetrics.periodCount}</p>
            </CardContent>
          </Card>
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between space-y-0">
              <CardTitle className="text-sm font-semibold text-gray-600">Cerradas</CardTitle>
              <ShieldCheck className="h-4 w-4 text-emerald-500" />
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-gray-900">{kpiMetrics.closedCount}</div>
              <p className="text-xs text-gray-500">Implementadas: {kpiMetrics.implementedCount}</p>
            </CardContent>
          </Card>
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between space-y-0">
              <CardTitle className="text-sm font-semibold text-gray-600">En proceso</CardTitle>
              <ListChecks className="h-4 w-4 text-emerald-500" />
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-gray-900">{kpiMetrics.inProgressCount}</div>
              <p className="text-xs text-gray-500">Planeadas: {kpiMetrics.plannedCount}</p>
            </CardContent>
          </Card>
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between space-y-0">
              <CardTitle className="text-sm font-semibold text-gray-600">Tiempo promedio</CardTitle>
              <CalendarDays className="h-4 w-4 text-emerald-500" />
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-gray-900">
                {kpiMetrics.averageCloseDays !== null ? `${kpiMetrics.averageCloseDays} días` : "N/A"}
              </div>
              <p className="text-xs text-gray-500">Promedio de cierre</p>
            </CardContent>
          </Card>
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between space-y-0">
              <CardTitle className="text-sm font-semibold text-gray-600">Top fuentes</CardTitle>
              <Target className="h-4 w-4 text-emerald-500" />
            </CardHeader>
            <CardContent className="space-y-2 text-sm text-gray-600">
              {kpiMetrics.topSources.length === 0 ? (
                <p className="text-xs text-gray-500">Sin fuentes registradas.</p>
              ) : (
                kpiMetrics.topSources.map((source) => (
                  <div key={source.key} className="flex items-center justify-between">
                    <span>{getSourceLabel(source.key)}</span>
                    <span className="font-semibold text-gray-900">{source.value}</span>
                  </div>
                ))
              )}
            </CardContent>
          </Card>
        </section>

        <Tabs defaultValue="register" className="space-y-6">
          <TabsList>
            <TabsTrigger value="register">Registro integrador</TabsTrigger>
            <TabsTrigger value="sources">Fuentes de evidencia</TabsTrigger>
          </TabsList>

          <TabsContent value="register" className="space-y-6">
            <section className="flex flex-col gap-4 rounded-3xl border border-emerald-100 bg-white p-6 shadow-sm">
              <div className="flex flex-wrap items-center justify-between gap-4">
                <div>
                  <h2 className="text-lg font-semibold text-gray-900">
                    Registro de Mejora Continua del SGA
                  </h2>
                  <p className="text-sm text-gray-600">
                    Tabla integradora para demostrar mejora continua, vinculando origen, acción,
                    evidencia y resultados.
                  </p>
                </div>
                <div className="flex flex-wrap items-center gap-3">
                  <Button
                    variant="outline"
                    className="border-emerald-100"
                    onClick={() => setFiltersOpen((prev) => !prev)}
                  >
                    <Filter className="mr-2 h-4 w-4" />
                    Filtros
                  </Button>
                  <Button variant="outline" className="border-emerald-100" onClick={fetchRecords}>
                    <RefreshCcw className="mr-2 h-4 w-4" />
                    Actualizar
                  </Button>
                  {canManage && (
                    <Button onClick={() => handleOpenDialog()}>
                      <Plus className="mr-2 h-4 w-4" />
                      Agregar mejora
                    </Button>
                  )}
                </div>
              </div>

              {filtersOpen && (
                <div className="grid gap-4 rounded-2xl border border-emerald-100 bg-emerald-50/40 p-4 md:grid-cols-2 lg:grid-cols-4">
                  <div className="space-y-2">
                    <Label>Fuente</Label>
                    <Select value={sourceFilter} onValueChange={setSourceFilter}>
                      <SelectTrigger>
                        <SelectValue placeholder="Todas" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="all">Todas</SelectItem>
                        {sourceOptions.map((option) => (
                          <SelectItem key={option.value} value={option.value}>
                            {option.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="space-y-2">
                    <Label>Estatus</Label>
                    <Select value={statusFilter} onValueChange={setStatusFilter}>
                      <SelectTrigger>
                        <SelectValue placeholder="Todos" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="all">Todos</SelectItem>
                        {statusOptions.map((option) => (
                          <SelectItem key={option.value} value={option.value}>
                            {option.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="space-y-2">
                    <Label>Responsable</Label>
                    <Input
                      value={responsibleFilter}
                      onChange={(event) => setResponsibleFilter(event.target.value)}
                      placeholder="Buscar responsable"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label>Búsqueda</Label>
                    <Input
                      value={searchFilter}
                      onChange={(event) => setSearchFilter(event.target.value)}
                      placeholder="Descripción, acción o resultado"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label>Desde</Label>
                    <Input
                      type="date"
                      value={dateFrom}
                      onChange={(event) => setDateFrom(event.target.value)}
                    />
                  </div>
                  <div className="space-y-2">
                    <Label>Hasta</Label>
                    <Input
                      type="date"
                      value={dateTo}
                      onChange={(event) => setDateTo(event.target.value)}
                    />
                  </div>
                </div>
              )}

              <div className="overflow-x-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Fecha</TableHead>
                      <TableHead>Fuente</TableHead>
                      <TableHead>Descripción</TableHead>
                      <TableHead>Acción implementada</TableHead>
                      <TableHead>Responsable</TableHead>
                      <TableHead>Fecha objetivo</TableHead>
                      <TableHead>Cierre</TableHead>
                      <TableHead>Evidencia</TableHead>
                      <TableHead>Resultado</TableHead>
                      <TableHead>Estatus</TableHead>
                      <TableHead></TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {loading ? (
                      <TableRow>
                        <TableCell colSpan={11} className="text-center text-sm text-gray-500">
                          Cargando registro...
                        </TableCell>
                      </TableRow>
                    ) : filteredRecords.length === 0 ? (
                      <TableRow>
                        <TableCell colSpan={11} className="text-center text-sm text-gray-500">
                          Sin mejoras registradas con los filtros seleccionados.
                        </TableCell>
                      </TableRow>
                    ) : (
                      filteredRecords.map((record) => (
                        <TableRow key={record.id} className="align-top">
                          <TableCell className="text-sm text-gray-600">
                            {formatDate(record.date ?? record.created_at)}
                          </TableCell>
                          <TableCell className="text-sm text-gray-600">
                            {getSourceLabel(record.source_type)}
                          </TableCell>
                          <TableCell className="min-w-[220px] text-sm text-gray-700">
                            {record.description}
                          </TableCell>
                          <TableCell className="min-w-[220px] text-sm text-gray-600">
                            {record.action_implemented || "Sin definir"}
                          </TableCell>
                          <TableCell className="text-sm text-gray-600">
                            {record.responsible || "Sin asignar"}
                          </TableCell>
                          <TableCell className="text-sm text-gray-600">
                            {formatDate(record.target_date)}
                          </TableCell>
                          <TableCell className="text-sm text-gray-600">
                            {formatDate(record.close_date)}
                          </TableCell>
                          <TableCell className="text-sm text-gray-600">
                            {record.evidence_url ? (
                              <Badge variant="outline" className="border-emerald-200">
                                {isHttpUrl(record.evidence_url) ? "Link" : "Archivo"}
                              </Badge>
                            ) : (
                              "-"
                            )}
                          </TableCell>
                          <TableCell className="min-w-[180px] text-sm text-gray-600">
                            {record.achieved_result || record.expected_result || "Sin resultado"}
                          </TableCell>
                          <TableCell>
                            <Badge
                              className={
                                statusBadgeStyles[record.status ?? ""] ??
                                "bg-slate-100 text-slate-700"
                              }
                            >
                              {getStatusLabel(record.status)}
                            </Badge>
                          </TableCell>
                          <TableCell>
                            <Button
                              variant="ghost"
                              size="sm"
                              onClick={() => {
                                setActiveRecord(record);
                                setDrawerOpen(true);
                              }}
                            >
                              Ver detalle
                            </Button>
                          </TableCell>
                        </TableRow>
                      ))
                    )}
                  </TableBody>
                </Table>
              </div>
            </section>
          </TabsContent>

          <TabsContent value="sources" className="space-y-6">
            <section className="grid gap-6 lg:grid-cols-2">
              {sourceCards.map((source) => {
                const Icon = source.icon;
                const sourceRecords = records
                  .filter((record) => record.source_type === source.sourceType)
                  .slice(0, 3);

                return (
                  <Card key={source.id} className="border-emerald-100">
                    <CardHeader className="space-y-2">
                      <div className="flex items-start justify-between gap-4">
                        <div className="flex items-center gap-3">
                          <span className="rounded-2xl bg-emerald-50 p-2">
                            <Icon className="h-5 w-5 text-emerald-600" />
                          </span>
                          <div>
                            <CardTitle className="text-base font-semibold text-gray-900">
                              {source.title}
                            </CardTitle>
                            <p className="text-sm text-gray-600">{source.description}</p>
                          </div>
                        </div>
                        <Badge variant="outline" className="border-emerald-200">
                          {records.filter((record) => record.source_type === source.sourceType).length} registros
                        </Badge>
                      </div>
                      {source.href && (
                        <Link
                          href={source.href}
                          className="inline-flex items-center gap-2 text-sm font-semibold text-emerald-600"
                        >
                          Ir al módulo
                          <ArrowUpRight className="h-4 w-4" />
                        </Link>
                      )}
                    </CardHeader>
                    <CardContent className="space-y-3">
                      {sourceRecords.length === 0 ? (
                        <p className="text-sm text-gray-500">Sin mejoras vinculadas aún.</p>
                      ) : (
                        sourceRecords.map((record) => (
                          <div
                            key={record.id}
                            className="rounded-2xl border border-emerald-100 bg-emerald-50/40 p-3"
                          >
                            <p className="text-sm font-semibold text-gray-900">
                              {record.description}
                            </p>
                            <div className="mt-2 flex flex-wrap items-center gap-2 text-xs text-gray-600">
                              <Badge
                                variant="outline"
                                className={
                                  statusBadgeStyles[record.status ?? ""] ??
                                  "bg-slate-100 text-slate-700"
                                }
                              >
                                {getStatusLabel(record.status)}
                              </Badge>
                              <span>Responsable: {record.responsible || "Sin asignar"}</span>
                              <span>Fecha: {formatDate(record.date ?? record.created_at)}</span>
                            </div>
                          </div>
                        ))
                      )}
                    </CardContent>
                  </Card>
                );
              })}
            </section>
          </TabsContent>
        </Tabs>
      </div>

      <Drawer open={drawerOpen} onOpenChange={setDrawerOpen}>
        <DrawerContent className="max-h-[90vh] overflow-y-auto px-4 pb-6 sm:px-8">
          <DrawerHeader>
            <DrawerTitle>Detalle de mejora continua</DrawerTitle>
          </DrawerHeader>
          {activeRecord && (
            <div className="space-y-6">
              <Card className="border-emerald-100">
                <CardHeader>
                  <CardTitle className="text-base">Información general</CardTitle>
                </CardHeader>
                <CardContent className="grid gap-4 md:grid-cols-2">
                  <div>
                    <p className="text-xs font-semibold uppercase text-gray-500">Fuente</p>
                    <p className="text-sm text-gray-900">{getSourceLabel(activeRecord.source_type)}</p>
                  </div>
                  <div>
                    <p className="text-xs font-semibold uppercase text-gray-500">Estatus</p>
                    <Badge
                      className={
                        statusBadgeStyles[activeRecord.status ?? ""] ??
                        "bg-slate-100 text-slate-700"
                      }
                    >
                      {getStatusLabel(activeRecord.status)}
                    </Badge>
                  </div>
                  <div>
                    <p className="text-xs font-semibold uppercase text-gray-500">Fecha de detección</p>
                    <p className="text-sm text-gray-900">
                      {formatDate(activeRecord.date ?? activeRecord.created_at)}
                    </p>
                  </div>
                  <div>
                    <p className="text-xs font-semibold uppercase text-gray-500">Responsable</p>
                    <p className="text-sm text-gray-900">
                      {activeRecord.responsible || "Sin asignar"}
                    </p>
                  </div>
                  <div>
                    <p className="text-xs font-semibold uppercase text-gray-500">Fecha objetivo</p>
                    <p className="text-sm text-gray-900">{formatDate(activeRecord.target_date)}</p>
                  </div>
                  <div>
                    <p className="text-xs font-semibold uppercase text-gray-500">Fecha cierre</p>
                    <p className="text-sm text-gray-900">{formatDate(activeRecord.close_date)}</p>
                  </div>
                </CardContent>
              </Card>

              <Card className="border-emerald-100">
                <CardHeader>
                  <CardTitle className="text-base">Descripción y acción</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <p className="text-xs font-semibold uppercase text-gray-500">Descripción</p>
                    <p className="text-sm text-gray-900">
                      {activeRecord.description || "Sin descripción"}
                    </p>
                  </div>
                  <div>
                    <p className="text-xs font-semibold uppercase text-gray-500">Acción implementada</p>
                    <p className="text-sm text-gray-900">
                      {activeRecord.action_implemented || "Sin acción registrada"}
                    </p>
                  </div>
                  <div className="grid gap-4 md:grid-cols-2">
                    <div>
                      <p className="text-xs font-semibold uppercase text-gray-500">
                        Resultado esperado
                      </p>
                      <p className="text-sm text-gray-900">
                        {activeRecord.expected_result || "Sin resultado esperado"}
                      </p>
                    </div>
                    <div>
                      <p className="text-xs font-semibold uppercase text-gray-500">
                        Resultado logrado
                      </p>
                      <p className="text-sm text-gray-900">
                        {activeRecord.achieved_result || "Sin resultado logrado"}
                      </p>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card className="border-emerald-100">
                <CardHeader>
                  <CardTitle className="text-base">Vínculos y evidencia</CardTitle>
                </CardHeader>
                <CardContent className="space-y-3">
                  <div className="grid gap-4 md:grid-cols-2">
                    <div>
                      <p className="text-xs font-semibold uppercase text-gray-500">Tabla origen</p>
                      <p className="text-sm text-gray-900">
                        {activeRecord.source_table || "Sin referencia"}
                      </p>
                    </div>
                    <div>
                      <p className="text-xs font-semibold uppercase text-gray-500">ID origen</p>
                      <p className="text-sm text-gray-900">
                        {activeRecord.source_id || "Sin referencia"}
                      </p>
                    </div>
                  </div>
                  <div>
                    <p className="text-xs font-semibold uppercase text-gray-500">Evidencia</p>
                    {evidencePreviewUrl ? (
                      <Button asChild variant="outline" className="mt-2 border-emerald-100">
                        <a href={evidencePreviewUrl} target="_blank" rel="noreferrer">
                          Ver evidencia
                        </a>
                      </Button>
                    ) : (
                      <p className="text-sm text-gray-500">Sin evidencia vinculada.</p>
                    )}
                  </div>
                </CardContent>
              </Card>
            </div>
          )}
          <DrawerFooter className="mt-6 flex flex-col gap-2 sm:flex-row sm:justify-end">
            {canManage && activeRecord && (
              <>
                <Button variant="outline" onClick={() => handleOpenDialog(activeRecord)}>
                  Editar mejora
                </Button>
                <Button
                  variant="destructive"
                  onClick={() => setDeleteTarget(activeRecord)}
                >
                  Eliminar
                </Button>
              </>
            )}
            <DrawerClose asChild>
              <Button variant="outline">Cerrar</Button>
            </DrawerClose>
          </DrawerFooter>
        </DrawerContent>
      </Drawer>

      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>
              {editingRecord ? "Editar mejora continua" : "Registrar mejora continua"}
            </DialogTitle>
          </DialogHeader>
          <div className="space-y-4">
            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <Label>Fecha de detección</Label>
                <Input
                  type="date"
                  value={formState.date}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, date: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label>Fuente de la mejora</Label>
                <Select
                  value={formState.source_type}
                  onValueChange={(value) =>
                    setFormState((prev) => ({ ...prev, source_type: value }))
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona una fuente" />
                  </SelectTrigger>
                  <SelectContent>
                    {sourceOptions.map((option) => (
                      <SelectItem key={option.value} value={option.value}>
                        {option.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Responsable</Label>
                <Input
                  value={formState.responsible}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, responsible: event.target.value }))
                  }
                  placeholder="Nombre o área responsable"
                />
              </div>
              <div className="space-y-2">
                <Label>Estatus</Label>
                <Select
                  value={formState.status}
                  onValueChange={(value) =>
                    setFormState((prev) => ({ ...prev, status: value }))
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona estatus" />
                  </SelectTrigger>
                  <SelectContent>
                    {statusOptions.map((option) => (
                      <SelectItem key={option.value} value={option.value}>
                        {option.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Fecha objetivo</Label>
                <Input
                  type="date"
                  value={formState.target_date}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, target_date: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label>Fecha cierre</Label>
                <Input
                  type="date"
                  value={formState.close_date}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, close_date: event.target.value }))
                  }
                />
              </div>
            </div>

            <div className="space-y-2">
              <Label>Descripción de la mejora</Label>
              <Textarea
                value={formState.description}
                onChange={(event) =>
                  setFormState((prev) => ({ ...prev, description: event.target.value }))
                }
                placeholder="Describe qué se mejora y su impacto esperado."
              />
            </div>
            <div className="space-y-2">
              <Label>Acción implementada / plan</Label>
              <Textarea
                value={formState.action_implemented}
                onChange={(event) =>
                  setFormState((prev) => ({ ...prev, action_implemented: event.target.value }))
                }
                placeholder="Indica la acción ejecutada o planificada."
              />
            </div>
            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <Label>Resultado esperado</Label>
                <Textarea
                  value={formState.expected_result}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, expected_result: event.target.value }))
                  }
                  placeholder="Define el resultado esperado."
                />
              </div>
              <div className="space-y-2">
                <Label>Resultado logrado</Label>
                <Textarea
                  value={formState.achieved_result}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, achieved_result: event.target.value }))
                  }
                  placeholder="Verificación o impacto logrado."
                />
              </div>
            </div>

            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <Label>Evidencia (URL)</Label>
                <Input
                  value={formState.evidence_url}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, evidence_url: event.target.value }))
                  }
                  placeholder="https://..."
                />
              </div>
              <div className="space-y-2">
                <Label>Subir evidencia (opcional)</Label>
                <Input
                  type="file"
                  onChange={(event) => setEvidenceFile(event.target.files?.[0] ?? null)}
                />
              </div>
            </div>

            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <Label>Tabla origen (opcional)</Label>
                <Input
                  value={formState.source_table}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, source_table: event.target.value }))
                  }
                  placeholder="nc_corrective_actions"
                />
              </div>
              <div className="space-y-2">
                <Label>ID origen (opcional)</Label>
                <Input
                  value={formState.source_id}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, source_id: event.target.value }))
                  }
                  placeholder="UUID / folio"
                />
              </div>
            </div>
          </div>
          <DialogFooter className="gap-2 sm:justify-end">
            <Button variant="outline" onClick={() => setDialogOpen(false)}>
              Cancelar
            </Button>
            <Button onClick={handleSubmit}>
              {editingRecord ? "Guardar cambios" : "Registrar mejora"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <AlertDialog open={!!deleteTarget} onOpenChange={(open) => !open && setDeleteTarget(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Eliminar mejora</AlertDialogTitle>
            <AlertDialogDescription>
              Esta acción eliminará el registro de mejora continua y su trazabilidad. ¿Deseas
              continuar?
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancelar</AlertDialogCancel>
            <AlertDialogAction onClick={handleDelete}>Eliminar</AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </SGIDashboardLayout>
  );
}