"use client";

import { useEffect, useMemo, useState, type FormEvent } from "react";
import Link from "next/link";
import { useRouter } from "next/navigation";
import Image from "next/image";
import { Award, ClipboardList, Leaf, Minus, Plus, ShieldCheck } from "lucide-react";
import { User } from "@supabase/supabase-js";
import { supabase, type Database } from "@/lib/supabase";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { toast } from "sonner";

type LeadershipMember = Database["public"]["Tables"]["leadership_org"]["Row"];

const leadershipSections = [
  {
    id: "general",
    label: "General",
    description: "Liderazgo corporativo y coordinación global.",
    icon: ClipboardList,
  },
  {
    id: "medio-ambiente",
    label: "Medio Ambiente",
    description: "Gestión ambiental y control de impactos.",
    icon: Leaf,
  },
  {
    id: "calidad",
    label: "Calidad",
    description: "Aseguramiento de calidad y mejora continua.",
    icon: Award,
  },
  {
    id: "seguridad",
    label: "Seguridad",
    description: "Prevención y control de riesgos laborales.",
    icon: ShieldCheck,
  },
] as const;

type LeadershipSectionId = (typeof leadershipSections)[number]["id"];

const DEFAULT_SECTION: LeadershipSectionId = "general";

const emptyForm = {
  full_name: "",
  position: "",
  role: "",
  responsibilities: "",
  photo_url: "",
  commitment_letter_url: "",
  parent_id: "",
  sort_order: "0",
  category: DEFAULT_SECTION as LeadershipSectionId,  
};

type FormState = typeof emptyForm;

const levelHeaderClasses = [
  "bg-slate-900 text-white",
  "bg-emerald-100 text-emerald-900",
  "bg-sky-100 text-sky-900",
  "bg-amber-100 text-amber-900",
  "bg-violet-100 text-violet-900",
];

type LeadershipTree = {
  map: Map<string, LeadershipMember[]>;
  rootKey: string;
};

const normalizeCategory = (category: string | null): LeadershipSectionId => {
  if (leadershipSections.some((section) => section.id === category)) {
    return category as LeadershipSectionId;
  }
  return DEFAULT_SECTION;
};

const buildTree = (members: LeadershipMember[]): LeadershipTree => {
  const map = new Map<string, LeadershipMember[]>();
  const rootKey = "root";

  members.forEach((member) => {
    const key = member.parent_id ?? rootKey;
    if (!map.has(key)) {
      map.set(key, []);
    }
    map.get(key)?.push(member);
  });

  return { map, rootKey };
};

const photoOptions = [
  "Adrian Esparza.jpg",
  "Daniel Collazo.jpg",
  "David Roque.png",
  "Fernando Duron.jpg",
  "Hector Sanroman.jpg",
  "Henri Hendrick.png",
  "Jonathan Chavez.jpg",
  "Judith Vazquez.jpg",
  "Lorena Reyes.jpg",
  "Miguel Jimenez.jpg",
  "Morimichi Oyachi.png",
  "Nelly Cordero.jpg",
  "Priyono.png",
  "Ruben Gallegos.jpg",
  "Shinya Sasaki.jpg",
  "Tsutsumi Katsutoshi.jpg",
  "Wendy Martinez.jpg",
].map((file) => ({
  label: file.replace(/\.[^.]+$/, ""),
  value: encodeURI(`/other resources/${file}`),
}));
const allowedManagers = new Set(["ehs.mexico@meiwa.com.mx", "recursoshumanos@meiwa.com.mx"]);

export default function LeadershipPage() {
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [members, setMembers] = useState<LeadershipMember[]>([]);
  const [formState, setFormState] = useState<FormState>(emptyForm);
  const [editingId, setEditingId] = useState<string | null>(null);
  const [saving, setSaving] = useState(false);
  const [printSectionId, setPrintSectionId] = useState<LeadershipSectionId | null>(null);
  const [sectionZoom, setSectionZoom] = useState<Record<LeadershipSectionId, number>>(() =>
    leadershipSections.reduce(
      (acc, section) => ({ ...acc, [section.id]: 100 }),
      {} as Record<LeadershipSectionId, number>,
    ),
  );
  const router = useRouter();
  const canManageOrg = allowedManagers.has(user?.email ?? "");
  const selectedPhotoOptions =
    formState.photo_url && !photoOptions.some((option) => option.value === formState.photo_url)
      ? [{ label: `Actual (${formState.photo_url})`, value: formState.photo_url }]
      : [];

  const loadMembers = async () => {
    const { data, error } = await supabase
      .from("leadership_org")
      .select("*")
      .order("sort_order", { ascending: true })
      .order("created_at", { ascending: true });

    if (error) {
      toast.error("No pudimos cargar el organigrama.");
      return;
    }

    setMembers(data ?? []);
  };

  useEffect(() => {
    const getSession = async () => {
      const { data: { session } } = await supabase.auth.getSession();

      if (!session?.user) {
        router.push("/meiwa-core/meiwa-sgi");
        return;
      }

      setUser(session.user);
      setLoading(false);
    };

    getSession();

    const { data: { subscription } } = supabase.auth.onAuthStateChange((_event, session) => {
      if (!session?.user) {
        router.push("/meiwa-core/meiwa-sgi");
      } else {
        setUser(session.user);
      }
    });

    return () => subscription.unsubscribe();
  }, [router]);

  useEffect(() => {
    if (user) {
      loadMembers();
    }
  }, [user]);

  useEffect(() => {
    const handleAfterPrint = () => setPrintSectionId(null);
    window.addEventListener("afterprint", handleAfterPrint);
    return () => {
      window.removeEventListener("afterprint", handleAfterPrint);
    };
  }, []);

  const membersBySection = useMemo(() => {
    const grouped = new Map<LeadershipSectionId, LeadershipMember[]>();
    leadershipSections.forEach((section) => grouped.set(section.id, []));

    members.forEach((member) => {
      const category = normalizeCategory(member.category ?? null);
      grouped.get(category)?.push(member);
    });

    return grouped;
  }, [members]);

  const treesBySection = useMemo(() => {
    const groupedTrees = new Map<LeadershipSectionId, LeadershipTree>();
    leadershipSections.forEach((section) => {
      groupedTrees.set(section.id, buildTree(membersBySection.get(section.id) ?? []));
    });
    return groupedTrees;
  }, [membersBySection]);

  const resetForm = () => {
    setFormState(emptyForm);
    setEditingId(null);
  };

  const handleSubmit = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!canManageOrg) {
      toast.error("No tienes permisos para gestionar el organigrama.");
      return;
    }
    setSaving(true);

    const payload = {
      full_name: formState.full_name.trim(),
      position: formState.position.trim(),
      role: formState.role.trim(),
      responsibilities: formState.responsibilities.trim() || null,
      photo_url: formState.photo_url.trim() || null,
      commitment_letter_url: formState.commitment_letter_url.trim() || null,
      parent_id: formState.parent_id || null,
      sort_order: Number.parseInt(formState.sort_order, 10) || 0,
      category: formState.category,      
    };

    if (!payload.full_name || !payload.position || !payload.role) {
      toast.error("Nombre, puesto y rol son obligatorios.");
      setSaving(false);
      return;
    }

    const { error } = editingId
      ? await supabase.from("leadership_org").update(payload).eq("id", editingId)
      : await supabase.from("leadership_org").insert(payload);

    if (error) {
      toast.error("No pudimos guardar los cambios.");
      setSaving(false);
      return;
    }

    toast.success(editingId ? "Integrante actualizado." : "Integrante agregado.");
    resetForm();
    await loadMembers();
    setSaving(false);
  };

  const handleEdit = (member: LeadershipMember) => {
    if (!canManageOrg) {
      toast.error("No tienes permisos para gestionar el organigrama.");
      return;
    }
    setEditingId(member.id);
    setFormState({
      full_name: member.full_name,
      position: member.position,
      role: member.role,
      responsibilities: member.responsibilities ?? "",
      photo_url: member.photo_url ?? "",
      commitment_letter_url: member.commitment_letter_url ?? "",
      parent_id: member.parent_id ?? "",
      sort_order: member.sort_order.toString(),
      category: normalizeCategory(member.category ?? null),      
    });
  };

  const handleDelete = async (member: LeadershipMember) => {
    if (!canManageOrg) {
      toast.error("No tienes permisos para gestionar el organigrama.");
      return;
    }
    const confirmed = window.confirm(`¿Eliminar a ${member.full_name} del organigrama?`);
    if (!confirmed) {
      return;
    }

    const { error } = await supabase.from("leadership_org").delete().eq("id", member.id);
    if (error) {
      toast.error("No pudimos eliminar a esta persona.");
      return;
    }

    toast.success("Integrante eliminado.");
    await loadMembers();
  };

  const renderMember = (member: LeadershipMember, tree: LeadershipTree, level = 0) => {
    const children = tree.map.get(member.id) ?? [];
    const headerClass = levelHeaderClasses[level % levelHeaderClasses.length];

    return (
      <div key={member.id} className="flex flex-col items-center">
        <Card className="w-[280px] overflow-hidden border border-slate-200 shadow-sm transition hover:-translate-y-1 hover:shadow-lg">
          <CardContent className="overflow-hidden p-0">
            <div className={`${headerClass} px-4 py-4`}>
              <div className="flex items-center gap-3">
                <div className="h-[72px] w-[72px] shrink-0 overflow-hidden rounded-full border-2 border-white/80 bg-white/90 shadow-sm">
                  {member.photo_url ? (
                    <Image
                      src={member.photo_url}
                      alt={member.full_name}
                      width={70}
                      height={70}
                      className="h-full w-full object-cover"
                    />
                  ) : (
                    <div className="flex h-full w-full items-center justify-center text-lg font-bold text-slate-700">
                      {member.full_name
                        .split(" ")
                        .map((word) => word[0])
                        .join("")
                        .slice(0, 2)
                        .toUpperCase()}
                    </div>
                  )}
                </div>
                <div className="space-y-1">
                  <p className="text-xs font-semibold uppercase tracking-[0.2em] opacity-80">
                    Nivel {level + 1}
                  </p>
                  <h3 className="text-sm font-bold uppercase tracking-wide">
                    {member.full_name}
                  </h3>
                </div>
              </div>
            </div>
            <div className="space-y-3 px-4 py-4">
              <p className="text-center text-xs font-semibold uppercase tracking-wide text-slate-800">
                {member.position}
              </p>
              <div className="rounded-full border border-slate-200 bg-slate-100 px-3 py-1 text-[0.6rem] font-semibold uppercase tracking-wide text-slate-700 text-center">
                {member.role}
              </div>
              {member.responsibilities && (
                <p className="text-xs text-slate-600">{member.responsibilities}</p>
              )}
              {member.commitment_letter_url && (
                <a
                  href={member.commitment_letter_url}
                  target="_blank"
                  rel="noreferrer"
                  className="text-xs font-semibold text-blue-600 underline underline-offset-4"
                >
                  Ver carta compromiso
                </a>
              )}
              {canManageOrg && (
                <div className="flex flex-wrap justify-center gap-2 pt-2">
                  <Button size="sm" variant="outline" onClick={() => handleEdit(member)}>
                    Editar
                  </Button>
                  <Button size="sm" variant="destructive" onClick={() => handleDelete(member)}>
                    Eliminar
                  </Button>
                </div>
              )}
            </div>
          </CardContent>
        </Card>
        {children.length > 0 && (
          <>
            <div className="h-6 w-[2px] bg-slate-500" />
            <div className="relative flex w-full flex-col items-center">
              <div className="absolute top-0 h-[2px] w-full max-w-5xl bg-slate-500" />
              <div className="flex w-max flex-nowrap justify-center gap-6 pt-6">
                {children.map((child) => (
                  <div key={child.id} className="flex flex-col items-center">
                    <div className="h-6 w-[2px] bg-slate-500" />
                    {renderMember(child, tree, level + 1)}
                  </div>
                ))}
              </div>
            </div>
          </>
        )}
      </div>
    );
  };

  const handleExportPdf = (sectionId: LeadershipSectionId) => {
    setPrintSectionId(sectionId);
    requestAnimationFrame(() => {
      window.print();
    });
  };

  const handleZoomChange = (sectionId: LeadershipSectionId, delta: number) => {
    setSectionZoom((prev) => {
      const nextValue = Math.min(140, Math.max(60, prev[sectionId] + delta));
      return {
        ...prev,
        [sectionId]: nextValue,
      };
    });
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-emerald-600"></div>
      </div>
    );
  }

  if (!user) {
    return null;
  }

  return (
    <SGIDashboardLayout>
      <style jsx global>{`
        @media print {
          body * {
            visibility: hidden;
          }
          .print-section,
          .print-section * {
            visibility: visible;
          }
          .print-section {
            position: absolute;
            inset: 0;
            width: 100%;
          }
          .no-print {
            display: none !important;
          }
        }
      `}</style>      
      <div className="space-y-8">
        <div className="rounded-2xl border border-slate-200 bg-white/90 p-6 shadow-sm">
          <div className="flex flex-col gap-6 lg:flex-row lg:items-center lg:justify-between">
            <div className="space-y-2">
              <p className="text-sm font-semibold uppercase tracking-[0.2em] text-emerald-600">
                Sistema de gestión
              </p>
              <h1 className="text-3xl font-bold text-gray-900 md:text-4xl">Liderazgo</h1>
              <p className="max-w-2xl text-sm text-gray-600 md:text-base">
                Consulta el organigrama de cada sistema de gestión y revisa los roles clave dentro
                de la organización.
              </p>
            </div>
            <div className="flex flex-wrap items-center gap-3">
              <div className="rounded-full border border-emerald-200 bg-emerald-50 px-4 py-2 text-sm font-semibold text-emerald-800">
                {members.length} integrantes registrados
              </div>
              <Button asChild variant="outline">
                <Link href="/meiwa-core/meiwa-sgi/policies">Ver políticas</Link>
              </Button>
            </div>
          </div>
        </div>
      </div>

      <div className="flex flex-wrap gap-3">
        {leadershipSections.map((section) => {
          const Icon = section.icon;

          return (
            <a
              key={section.id}
              href={`#${section.id}`}
              className="flex items-center gap-2 rounded-full border border-gray-200 bg-white px-4 py-2 text-sm font-semibold text-gray-600 transition-all duration-200 hover:border-emerald-200 hover:text-emerald-700"
            >
              <Icon className="h-4 w-4 text-emerald-500" />
              {section.label}
            </a>
          );
        })}
      </div>

      <div
        className={`grid gap-8 ${canManageOrg ? "lg:grid-cols-[380px_1fr]" : "lg:grid-cols-1"}`}
      >
        {canManageOrg && (
          <Card className="border border-slate-200 shadow-sm">
            <CardHeader className="border-b border-slate-100 bg-slate-50/80">
              <CardTitle className="text-base font-semibold">
                {editingId ? "Editar integrante" : "Agregar integrante"}
              </CardTitle>
            </CardHeader>
            <CardContent>
              <form className="space-y-4" onSubmit={handleSubmit}>
                <div className="space-y-2">
                  <Label htmlFor="full_name">Nombre</Label>
                  <Input
                    id="full_name"
                    value={formState.full_name}
                    onChange={(event) =>
                      setFormState((prev) => ({ ...prev, full_name: event.target.value }))
                    }
                    placeholder="Nombre completo"
                    required
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="position">Puesto</Label>
                  <Input
                    id="position"
                    value={formState.position}
                    onChange={(event) =>
                      setFormState((prev) => ({ ...prev, position: event.target.value }))
                    }
                    placeholder="Dirección, gerencia, etc."
                    required
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="role">Rol</Label>
                  <Input
                    id="role"
                    value={formState.role}
                    onChange={(event) =>
                      setFormState((prev) => ({ ...prev, role: event.target.value }))
                    }
                    placeholder="Responsable, líder, etc."
                    required
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="responsibilities">Funciones</Label>
                  <Textarea
                    id="responsibilities"
                    value={formState.responsibilities}
                    onChange={(event) =>
                      setFormState((prev) => ({ ...prev, responsibilities: event.target.value }))
                    }
                    placeholder="Describe las funciones principales"
                    rows={4}
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="category">Sección</Label>
                  <select
                    id="category"
                    value={formState.category}
                    onChange={(event) =>
                      setFormState((prev) => ({
                        ...prev,
                        category: normalizeCategory(event.target.value),
                        parent_id: "",
                      }))
                    }
                    className="flex h-12 w-full rounded-md border border-input bg-background px-3 py-2 text-sm"
                  >
                    {leadershipSections.map((section) => (
                      <option key={section.id} value={section.id}>
                        {section.label}
                      </option>
                    ))}
                  </select>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="photo_url">Foto (carpeta public/other resources)</Label>
                  <select
                    id="photo_url"
                    value={formState.photo_url}
                    onChange={(event) =>
                      setFormState((prev) => ({ ...prev, photo_url: event.target.value }))
                    }
                    className="flex h-14 w-full rounded-md border border-input bg-background px-3 py-2 text-sm"
                  >
                    <option value="">Sin foto</option>
                    {selectedPhotoOptions.map((option) => (
                      <option key={option.value} value={option.value}>
                        {option.label}
                      </option>
                    ))}
                    {photoOptions.map((option) => (
                      <option key={option.value} value={option.value}>
                        {option.label}
                      </option>
                    ))}
                  </select>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="commitment_letter_url">Carta compromiso (URL)</Label>
                  <Input
                    id="commitment_letter_url"
                    value={formState.commitment_letter_url}
                    onChange={(event) =>
                      setFormState((prev) => ({ ...prev, commitment_letter_url: event.target.value }))
                    }
                    placeholder="https://"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="parent_id">Reporta a</Label>
                  <select
                    id="parent_id"
                    value={formState.parent_id}
                    onChange={(event) =>
                      setFormState((prev) => ({ ...prev, parent_id: event.target.value }))
                    }
                    className="flex h-10 w-full rounded-md border border-input bg-background px-3 py-2 text-sm"
                  >
                    <option value="">Sin jefe directo</option>
                    {members
                      .filter(
                        (member) =>
                          member.id !== editingId &&
                          normalizeCategory(member.category ?? null) === formState.category,
                      )
                      .map((member) => (
                        <option key={member.id} value={member.id}>
                          {member.full_name} · {member.position}
                        </option>
                      ))}
                  </select>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="sort_order">Orden</Label>
                  <Input
                    id="sort_order"
                    type="number"
                    value={formState.sort_order}
                    onChange={(event) =>
                      setFormState((prev) => ({ ...prev, sort_order: event.target.value }))
                    }
                    min={0}
                  />
                </div>
                <div className="flex flex-col gap-2 pt-2">
                  <Button type="submit" disabled={saving}>
                    {saving
                      ? "Guardando..."
                      : editingId
                      ? "Actualizar integrante"
                      : "Agregar integrante"}
                  </Button>
                  {editingId && (
                    <Button type="button" variant="ghost" onClick={resetForm}>
                      Cancelar edición
                    </Button>
                  )}
                </div>
              </form>
            </CardContent>
          </Card>
        )}

        <div className="space-y-8">
          {leadershipSections.map((section) => {
            const sectionMembers = membersBySection.get(section.id) ?? [];
            const sectionTree = treesBySection.get(section.id);
            const Icon = section.icon;
            const rootMembers = sectionTree?.map.get(sectionTree.rootKey) ?? [];

            return (
              <Card
                key={section.id}
                id={section.id}
                className={`scroll-mt-24 border border-slate-200 shadow-sm ${
                  printSectionId === section.id ? "print-section" : ""
                }`}
              >
                <CardHeader className="border-b border-slate-100 bg-slate-50/80">
                  <div className="flex flex-wrap items-center justify-between gap-4">
                    <div className="flex items-start gap-3">
                      <div className="flex h-10 w-10 items-center justify-center rounded-2xl bg-emerald-50">
                        <Icon className="h-5 w-5 text-emerald-600" />
                      </div>
                      <div className="space-y-1">
                        <CardTitle className="text-base font-semibold">
                          {section.label}
                        </CardTitle>
                        <p className="text-xs text-slate-500">{section.description}</p>
                      </div>
                    </div>
                    <div className="flex flex-wrap items-center gap-3">
                      <div className="flex items-center gap-2 rounded-full border border-slate-200 bg-white px-3 py-1 text-xs font-semibold text-slate-700">
                        <Button
                          type="button"
                          variant="outline"
                          size="icon"
                          className="h-7 w-7"
                          onClick={() => handleZoomChange(section.id, -10)}
                          aria-label={`Reducir zoom en ${section.label}`}
                        >
                          <Minus className="h-4 w-4" />
                        </Button>
                        <span className="min-w-[56px] text-center text-xs font-semibold text-slate-700">
                          {sectionZoom[section.id]}%
                        </span>
                        <Button
                          type="button"
                          variant="outline"
                          size="icon"
                          className="h-7 w-7"
                          onClick={() => handleZoomChange(section.id, 10)}
                          aria-label={`Aumentar zoom en ${section.label}`}
                        >
                          <Plus className="h-4 w-4" />
                        </Button>
                      </div>
                      <div className="rounded-full border border-emerald-200 bg-emerald-50 px-3 py-1 text-xs font-semibold text-emerald-800">
                        {sectionMembers.length} integrantes
                      </div>
                      <Button
                        type="button"
                        variant="outline"
                        size="sm"
                        className="no-print"
                        onClick={() => handleExportPdf(section.id)}
                      >
                        Imprimir sección
                      </Button>
                    </div>                
                  </div>
                </CardHeader>
                <CardContent className="space-y-6 overflow-x-auto pb-10">
                  {rootMembers.length ? (
                    <div
                      className="min-w-max"
                      style={{
                        transform: `scale(${sectionZoom[section.id] / 100})`,
                        transformOrigin: "top left",
                      }}
                    >
                      <div className="flex flex-col items-center gap-10">
                        {rootMembers.map((member) =>
                          sectionTree ? renderMember(member, sectionTree) : null,
                        )}
                      </div>
                    </div>
                  ) : (
                    <div className="rounded-xl border border-dashed border-gray-200 bg-slate-50 p-8 text-center text-sm text-gray-500">
                      Aún no hay integrantes registrados en esta sección.
                    </div>
                  )}
                </CardContent>
              </Card>
            );
          })}          
        </div>
      </div>
    </SGIDashboardLayout>
  );
}