"use client";

import { type FormEvent, useEffect, useMemo, useState } from "react";
import { type User } from "@supabase/supabase-js";
import { toast } from "sonner";
import {
  BatteryCharging,
  ClipboardList,
  Factory,
  Gauge,
  LineChart,
  Percent,
  PlugZap,
  Plus,
  Trash2,
} from "lucide-react";

import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { supabase } from "@/lib/supabase";

const allowedManagers = new Set([
  "ehs.mexico@meiwa.com.mx",
  "admin.mexico@meiwa.com.mx",
  "admin2.mexico@meiwa.com.mx",
]);

const spanishMonths = [
  "Enero",
  "Febrero",
  "Marzo",
  "Abril",
  "Mayo",
  "Junio",
  "Julio",
  "Agosto",
  "Septiembre",
  "Octubre",
  "Noviembre",
  "Diciembre",
];

type EnergyRecord = {
  id: string;
  date: string;
  month: string;
  year: number;
  generationKwh: number;
  consumptionKwh: number;
  energyAcquiredCfeKwh: number;
  energySoldCfeKwh: number;
  cfeBill: number;
  kiwapowerRate: number;
  replacementRate: number;
  cfeRate: number;
};

type EnergyFormState = {
  date: string;
  generationKwh: string;
  consumptionKwh: string;
  energyAcquiredCfeKwh: string;
  energySoldCfeKwh: string;
  cfeBill: string;
  kiwapowerRate: string;
  replacementRate: string;
  cfeRate: string;
};

const numberFormatter = new Intl.NumberFormat("es-MX", {
  minimumFractionDigits: 0,
  maximumFractionDigits: 2,
});

const percentFormatter = new Intl.NumberFormat("es-MX", {
  style: "percent",
  minimumFractionDigits: 1,
  maximumFractionDigits: 1,
});

const currencyFormatter = new Intl.NumberFormat("es-MX", {
  style: "currency",
  currency: "MXN",
  minimumFractionDigits: 2,
  maximumFractionDigits: 2,
});

const toSpanishMonth = (dateValue: string) => {
  const month = new Date(`${dateValue}T00:00:00`).toLocaleDateString("es-MX", {
    month: "long",
  });
  return month ? `${month[0].toUpperCase()}${month.slice(1)}` : "";
};

const formatDate = (value: string) =>
  new Date(`${value}T00:00:00`).toLocaleDateString("es-MX", {
    day: "2-digit",
    month: "short",
    year: "numeric",
  });

const safeDivide = (numerator: number, denominator: number) =>
  denominator ? numerator / denominator : 0;

const formatPercent = (numerator: number, denominator: number) =>
  denominator ? percentFormatter.format(numerator / denominator) : "N/A";

const mapRecord = (record: Record<string, unknown>): EnergyRecord => {
  const dateValue = String(record.date ?? "");
  const getNumber = (value: unknown) =>
    typeof value === "number" ? value : Number(value ?? 0) || 0;
  return {
    id: String(record.id ?? ""),
    date: dateValue,
    month: String(record.month ?? toSpanishMonth(dateValue)),
    year:
      typeof record.year === "number"
        ? record.year
        : new Date(`${dateValue}T00:00:00`).getFullYear(),
    generationKwh: getNumber(record.generation_kwh),
    consumptionKwh: getNumber(record.consumption_kwh),
    energyAcquiredCfeKwh: getNumber(record.energy_acquired_cfe_kwh),
    energySoldCfeKwh: getNumber(record.energy_sold_cfe_kwh),
    cfeBill: getNumber(record.cfe_bill),
    kiwapowerRate: getNumber(record.kiwapower_rate),
    replacementRate: getNumber(record.replacement_rate),
    cfeRate: getNumber(record.cfe_rate),
  };
};

const sortByDateDesc = (records: EnergyRecord[]) =>
  [...records].sort((first, second) => {
    const firstTime = new Date(`${first.date}T00:00:00`).getTime();
    const secondTime = new Date(`${second.date}T00:00:00`).getTime();
    if (Number.isNaN(firstTime) && Number.isNaN(secondTime)) return 0;
    if (Number.isNaN(firstTime)) return 1;
    if (Number.isNaN(secondTime)) return -1;
    return secondTime - firstTime;
  });

const buildRecordKpis = (record: EnergyRecord) => {
  const coverageGross = safeDivide(record.generationKwh, record.consumptionKwh);
  const coverageNet = safeDivide(
    record.generationKwh - record.energySoldCfeKwh,
    record.consumptionKwh,
  );
  const dependencyGross = safeDivide(
    record.energyAcquiredCfeKwh,
    record.consumptionKwh,
  );
  const dependencyNet = safeDivide(
    record.energyAcquiredCfeKwh - record.energySoldCfeKwh,
    record.consumptionKwh,
  );
  const exportShare = safeDivide(record.energySoldCfeKwh, record.generationKwh);
  const importShare = safeDivide(record.energyAcquiredCfeKwh, record.consumptionKwh);
  const selfSufficiency = coverageNet - dependencyNet;
  const importExportIndex = importShare - exportShare;

  return {
    coverageGross,
    coverageNet,
    dependencyGross,
    dependencyNet,
    exportShare,
    importShare,
    selfSufficiency,
    importExportIndex,
  };
};

const StatCard = ({
  title,
  value,
  helper,
  icon: Icon,
  accent,
}: {
  title: string;
  value: string;
  helper: string;
  icon: typeof BatteryCharging;
  accent: string;
}) => (
  <Card className="border-gray-200">
    <CardContent className="flex items-start justify-between gap-4 p-6">
      <div>
        <p className="text-xs font-semibold uppercase tracking-[0.2em] text-gray-500">
          {title}
        </p>
        <p className="mt-3 text-2xl font-semibold text-gray-900">{value}</p>
        <p className="mt-2 text-sm text-gray-500">{helper}</p>
      </div>
      <div className={`rounded-2xl p-3 ${accent}`}>
        <Icon className="h-5 w-5 text-white" />
      </div>
    </CardContent>
  </Card>
);

export default function EnergyPage() {
  const [records, setRecords] = useState<EnergyRecord[]>([]);
  const [formState, setFormState] = useState<EnergyFormState>({
    date: "",
    generationKwh: "",
    consumptionKwh: "",
    energyAcquiredCfeKwh: "",
    energySoldCfeKwh: "",
    cfeBill: "",
    kiwapowerRate: "",
    replacementRate: "",
    cfeRate: "",
  });
  const [user, setUser] = useState<User | null>(null);
  const [selectedYear, setSelectedYear] = useState<string>("Todo");
  const [selectedMonth, setSelectedMonth] = useState<string>("Todo");
  const [isLoading, setIsLoading] = useState<boolean>(true);
  const canManageEnergy = allowedManagers.has(user?.email ?? "");

  const availableYears = useMemo(
    () =>
      Array.from(new Set(records.map((record) => record.year)))
        .filter((year) => Number.isFinite(year))
        .sort((a, b) => b - a),
    [records],
  );

  const availableMonths = useMemo(() => {
    const monthSet = new Set(records.map((record) => record.month).filter(Boolean));
    return spanishMonths.filter((month) => monthSet.has(month));
  }, [records]);

  const filteredRecords = useMemo(
    () =>
      records.filter((record) => {
        const matchesYear = selectedYear === "Todo" || record.year === Number(selectedYear);
        const matchesMonth = selectedMonth === "Todo" || record.month === selectedMonth;
        return matchesYear && matchesMonth;
      }),
    [records, selectedMonth, selectedYear],
  );

  const aggregateStats = useMemo(() => {
    const totals = filteredRecords.reduce(
      (acc, record) => {
        acc.generation += record.generationKwh;
        acc.consumption += record.consumptionKwh;
        acc.acquired += record.energyAcquiredCfeKwh;
        acc.sold += record.energySoldCfeKwh;
        acc.cfeBill += record.cfeBill;
        return acc;
      },
      {
        generation: 0,
        consumption: 0,
        acquired: 0,
        sold: 0,
        cfeBill: 0,
      },
    );

    const coverageGross = safeDivide(totals.generation, totals.consumption);
    const coverageNet = safeDivide(totals.generation - totals.sold, totals.consumption);
    const dependencyGross = safeDivide(totals.acquired, totals.consumption);
    const dependencyNet = safeDivide(totals.acquired - totals.sold, totals.consumption);
    const exportShare = safeDivide(totals.sold, totals.generation);
    const importShare = safeDivide(totals.acquired, totals.consumption);
    const selfSufficiency = coverageNet - dependencyNet;
    const importExportIndex = importShare - exportShare;

    return {
      totals,
      coverageGross,
      coverageNet,
      dependencyGross,
      dependencyNet,
      exportShare,
      importShare,
      selfSufficiency,
      importExportIndex,
    };
  }, [filteredRecords]);

  useEffect(() => {
    const loadSession = async () => {
      const { data } = await supabase.auth.getSession();
      setUser(data.session?.user ?? null);
    };

    const loadRecords = async () => {
      setIsLoading(true);
      const { data, error } = await supabase.from("energy_records").select("*");

      if (error) {
        toast.error("No pudimos cargar los registros de energía.");
        setIsLoading(false);
        return;
      }

      const mapped = (data ?? []).map((record) => mapRecord(record));
      setRecords(sortByDateDesc(mapped));
      setIsLoading(false);
    };

    loadSession();
    loadRecords();

    const { data: { subscription } } = supabase.auth.onAuthStateChange((_event, session) => {
      setUser(session?.user ?? null);
    });

    return () => subscription.unsubscribe();
  }, []);

  const handleAddRecord = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!canManageEnergy) return;

    const parsedDate = new Date(`${formState.date}T00:00:00`);
    if (!formState.date || Number.isNaN(parsedDate.valueOf())) {
      toast.error("La fecha es obligatoria.");
      return;
    }

    const numericValues = {
      generationKwh: Number(formState.generationKwh),
      consumptionKwh: Number(formState.consumptionKwh),
      energyAcquiredCfeKwh: Number(formState.energyAcquiredCfeKwh),
      energySoldCfeKwh: Number(formState.energySoldCfeKwh),
      cfeBill: Number(formState.cfeBill),
      kiwapowerRate: Number(formState.kiwapowerRate),
      replacementRate: Number(formState.replacementRate),
      cfeRate: Number(formState.cfeRate),
    };

    const invalidEntry = Object.values(numericValues).some(
      (value) => !Number.isFinite(value) || value < 0,
    );

    if (invalidEntry) {
      toast.error("Verifica que todos los valores sean números válidos (>= 0).");
      return;
    }

    if (!user) {
      toast.error("Necesitamos una sesión activa para guardar el registro.");
      return;
    }

    const payload = {
      date: formState.date,
      month: toSpanishMonth(formState.date),
      year: parsedDate.getFullYear(),
      generation_kwh: numericValues.generationKwh,
      consumption_kwh: numericValues.consumptionKwh,
      energy_acquired_cfe_kwh: numericValues.energyAcquiredCfeKwh,
      energy_sold_cfe_kwh: numericValues.energySoldCfeKwh,
      cfe_bill: numericValues.cfeBill,
      kiwapower_rate: numericValues.kiwapowerRate,
      replacement_rate: numericValues.replacementRate,
      cfe_rate: numericValues.cfeRate,
      created_by: user.id,
    };

    const { data, error } = await supabase
      .from("energy_records")
      .insert(payload)
      .select()
      .single();

    if (error || !data) {
      toast.error("No pudimos guardar el registro.");
      return;
    }

    setRecords((prev) => [mapRecord(data), ...prev]);
    setFormState({
      date: "",
      generationKwh: "",
      consumptionKwh: "",
      energyAcquiredCfeKwh: "",
      energySoldCfeKwh: "",
      cfeBill: "",
      kiwapowerRate: "",
      replacementRate: "",
      cfeRate: "",
    });
    toast.success("Registro guardado.");
  };

  const promptNumber = (label: string, value: number) => {
    const input = window.prompt(label, String(value));
    if (input === null) return null;
    const parsed = Number(input);
    if (!Number.isFinite(parsed) || parsed < 0) {
      toast.error("El valor debe ser un número válido (>= 0).");
      return null;
    }
    return parsed;
  };

  const handleEditRecord = async (recordId: string) => {
    if (!canManageEnergy) return;
    const record = records.find((item) => item.id === recordId);
    if (!record) return;

    const date = window.prompt("Fecha de registro (YYYY-MM-DD)", record.date);
    if (!date) return;
    const parsedDate = new Date(`${date}T00:00:00`);
    if (Number.isNaN(parsedDate.valueOf())) {
      toast.error("La fecha no es válida.");
      return;
    }

    const generationKwh = promptNumber("Generación de energía (kWh)", record.generationKwh);
    if (generationKwh === null) return;
    const consumptionKwh = promptNumber("Consumo de energía (kWh)", record.consumptionKwh);
    if (consumptionKwh === null) return;
    const energyAcquiredCfeKwh = promptNumber(
      "Energía adquirida de CFE (kWh)",
      record.energyAcquiredCfeKwh,
    );
    if (energyAcquiredCfeKwh === null) return;
    const energySoldCfeKwh = promptNumber(
      "Energía vendida a CFE (kWh)",
      record.energySoldCfeKwh,
    );
    if (energySoldCfeKwh === null) return;
    const cfeBill = promptNumber("Factura actual de CFE (MXN)", record.cfeBill);
    if (cfeBill === null) return;
    const kiwapowerRate = promptNumber(
      "Tarifa de Kiwapower (sin IVA) MXN/kWh",
      record.kiwapowerRate,
    );
    if (kiwapowerRate === null) return;
    const replacementRate = promptNumber(
      "Tarifa de reemplazo (sin IVA) MXN/kWh",
      record.replacementRate,
    );
    if (replacementRate === null) return;
    const cfeRate = promptNumber(
      "Tarifa de energía de CFE (sin IVA) MXN/kWh",
      record.cfeRate,
    );
    if (cfeRate === null) return;

    const payload = {
      date,
      month: toSpanishMonth(date),
      year: parsedDate.getFullYear(),
      generation_kwh: generationKwh,
      consumption_kwh: consumptionKwh,
      energy_acquired_cfe_kwh: energyAcquiredCfeKwh,
      energy_sold_cfe_kwh: energySoldCfeKwh,
      cfe_bill: cfeBill,
      kiwapower_rate: kiwapowerRate,
      replacement_rate: replacementRate,
      cfe_rate: cfeRate,
    };

    const { data, error } = await supabase
      .from("energy_records")
      .update(payload)
      .eq("id", recordId)
      .select()
      .single();

    if (error || !data) {
      toast.error("No pudimos actualizar el registro.");
      return;
    }

    setRecords((prev) =>
      prev.map((item) => (item.id === recordId ? mapRecord(data) : item)),
    );
    toast.success("Registro actualizado.");
  };

  const handleDeleteRecord = async (recordId: string) => {
    if (!canManageEnergy) return;
    const confirmed = window.confirm("¿Deseas eliminar este registro?");
    if (!confirmed) return;

    const { error } = await supabase.from("energy_records").delete().eq("id", recordId);
    if (error) {
      toast.error("No pudimos eliminar el registro.");
      return;
    }

    setRecords((prev) => prev.filter((item) => item.id !== recordId));
    toast.success("Registro eliminado.");
  };

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-10 bg-gradient-to-br from-slate-50 via-white to-emerald-50">
        <section className="relative overflow-hidden rounded-3xl border border-emerald-100 bg-white px-6 py-10 shadow-sm sm:px-10">
          <div className="absolute inset-0 bg-gradient-to-r from-emerald-500/10 to-lime-500/10" />
          <div className="relative space-y-4">
            <p className="text-xs font-semibold uppercase tracking-[0.5em] text-emerald-600">
              Operación · Energía
            </p>
            <h1 className="text-3xl font-bold text-gray-900 sm:text-4xl">
              Gestión y balance energético
            </h1>
            <p className="max-w-8xl text-base text-gray-600 xs:text-lg">
              Aquí se enceuntran los registros relacionados con el consumo energetico y el calculo de indicadores por periodo.
            </p>
            <div className="mt-6 grid gap-4 rounded-2xl border border-emerald-100 bg-white/80 p-4 sm:grid-cols-3">
              <div className="space-y-2">
                <Label className="text-xs font-semibold uppercase tracking-[0.2em] text-gray-500">
                  Filtro de año
                </Label>
                <select
                  className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm text-slate-900 shadow-sm"
                  value={selectedYear}
                  onChange={(event) => setSelectedYear(event.target.value)}
                >
                  <option value="Todo">Todo</option>
                  {availableYears.map((year) => (
                    <option key={year} value={String(year)}>
                      {year}
                    </option>
                  ))}
                </select>
              </div>
              <div className="space-y-2">
                <Label className="text-xs font-semibold uppercase tracking-[0.2em] text-gray-500">
                  Filtro de mes
                </Label>
                <select
                  className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm text-slate-900 shadow-sm"
                  value={selectedMonth}
                  onChange={(event) => setSelectedMonth(event.target.value)}
                >
                  <option value="Todo">Todo</option>
                  {availableMonths.map((month) => (
                    <option key={month} value={month}>
                      {month}
                    </option>
                  ))}
                </select>
              </div>
              <div className="rounded-2xl border border-emerald-100 bg-emerald-50 px-4 py-5">
                <p className="text-xs font-semibold uppercase text-emerald-600">
                  Registros filtrados
                </p>
                <p className="mt-2 text-2xl font-semibold text-emerald-900">
                  {filteredRecords.length} registros
                </p>
                <p className="mt-1 text-xs text-emerald-700">
                  Actualiza los filtros para recalcular indicadores.
                </p>
              </div>
            </div>
          </div>
        </section>

        <section className="grid gap-4 md:grid-cols-2 xl:grid-cols-4">
          <StatCard
            title="Generación total"
            value={`${numberFormatter.format(aggregateStats.totals.generation)} kWh`}
            helper="Periodo filtrado"
            icon={BatteryCharging}
            accent="bg-emerald-500"
          />
          <StatCard
            title="Consumo total"
            value={`${numberFormatter.format(aggregateStats.totals.consumption)} kWh`}
            helper="Periodo filtrado"
            icon={Gauge}
            accent="bg-sky-500"
          />
          <StatCard
            title="CFE adquirida"
            value={`${numberFormatter.format(aggregateStats.totals.acquired)} kWh`}
            helper="Energía importada"
            icon={PlugZap}
            accent="bg-indigo-500"
          />
          <StatCard
            title="CFE vendida"
            value={`${numberFormatter.format(aggregateStats.totals.sold)} kWh`}
            helper="Energía exportada"
            icon={LineChart}
            accent="bg-emerald-400"
          />
        </section>

        <section className="grid gap-6 lg:grid-cols-2">
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between gap-3">
              <div>
                <CardTitle className="text-xl text-gray-900">Coberturas y dependencias</CardTitle>
                <p className="text-sm text-gray-500">
                  Indicadores ponderados por consumo en el periodo seleccionado.
                </p>
              </div>
              <div className="rounded-2xl bg-emerald-100 p-3 text-emerald-600">
                <BatteryCharging className="h-5 w-5" />
              </div>
            </CardHeader>
            <CardContent className="grid gap-4 sm:grid-cols-2">
              <StatCard
                title="Cobertura FV bruta"
                value={formatPercent(aggregateStats.totals.generation, aggregateStats.totals.consumption)}
                helper="Generación / Consumo"
                icon={Percent}
                accent="bg-emerald-500"
              />
              <StatCard
                title="Cobertura FV neta"
                value={formatPercent(
                  aggregateStats.totals.generation - aggregateStats.totals.sold,
                  aggregateStats.totals.consumption,
                )}
                helper="(Generación - Vendida) / Consumo"
                icon={Percent}
                accent="bg-emerald-400"
              />
              <StatCard
                title="Dependencia bruta CFE"
                value={formatPercent(aggregateStats.totals.acquired, aggregateStats.totals.consumption)}
                helper="Adquirida / Consumo"
                icon={PlugZap}
                accent="bg-sky-500"
              />
              <StatCard
                title="Dependencia neta CFE"
                value={formatPercent(
                  aggregateStats.totals.acquired - aggregateStats.totals.sold,
                  aggregateStats.totals.consumption,
                )}
                helper="(Adquirida - Vendida) / Consumo"
                icon={PlugZap}
                accent="bg-sky-400"
              />
            </CardContent>
          </Card>

          <Card className="border-sky-100">
            <CardHeader className="flex flex-row items-center justify-between gap-3">
              <div>
                <CardTitle className="text-xl text-gray-900">Índices energéticos</CardTitle>
                <p className="text-sm text-gray-500">
                  Autosuficiencia e importación/exportación del periodo.
                </p>
              </div>
              <div className="rounded-2xl bg-sky-100 p-3 text-sky-600">
                <ClipboardList className="h-5 w-5" />
              </div>
            </CardHeader>
            <CardContent className="grid gap-4 sm:grid-cols-2">
              <StatCard
                title="Autosuficiencia energética"
                value={percentFormatter.format(aggregateStats.selfSufficiency)}
                helper="Cobertura neta - Dependencia neta"
                icon={BatteryCharging}
                accent="bg-sky-500"
              />
              <StatCard
                title="% Exportación a CFE"
                value={formatPercent(aggregateStats.totals.sold, aggregateStats.totals.generation)}
                helper="Vendida / Generación"
                icon={LineChart}
                accent="bg-sky-400"
              />
              <StatCard
                title="% Importación a CFE"
                value={formatPercent(aggregateStats.totals.acquired, aggregateStats.totals.consumption)}
                helper="Adquirida / Consumo"
                icon={PlugZap}
                accent="bg-indigo-500"
              />
              <StatCard
                title="Índice Imp-Exp"
                value={percentFormatter.format(aggregateStats.importExportIndex)}
                helper="% Importación - % Exportación"
                icon={Factory}
                accent="bg-indigo-400"
              />
            </CardContent>
          </Card>
        </section>

        <section className="space-y-4">
          <div className="flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
            <div>
              <h2 className="text-2xl font-semibold text-gray-900">Registro histórico</h2>
              <p className="text-sm text-gray-500">
                Administra los registros energéticos con KPIs por fecha y periodo.
              </p>
            </div>
            {!canManageEnergy && (
              <span className="text-sm font-medium text-gray-500">
                Acceso de solo lectura.
              </span>
            )}
          </div>

          <Card className="border-emerald-100">
            {canManageEnergy && (
              <>
                <CardHeader>
                  <CardTitle className="text-lg text-gray-900">Nuevo registro de energía</CardTitle>
                  <p className="text-sm text-gray-500">
                    Completa el formulario para registrar el balance del periodo.
                  </p>
                </CardHeader>
                <CardContent>
                  <form
                    className="grid gap-4 rounded-2xl border border-emerald-100 bg-emerald-50 p-4 md:grid-cols-2"
                    onSubmit={handleAddRecord}
                  >
                    <div className="space-y-2">
                      <Label htmlFor="energy-date">Fecha de registro</Label>
                      <Input
                        id="energy-date"
                        type="date"
                        value={formState.date}
                        onChange={(event) =>
                          setFormState((prev) => ({ ...prev, date: event.target.value }))
                        }
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="energy-generation">Generación (kWh)</Label>
                      <Input
                        id="energy-generation"
                        type="number"
                        min="0"
                        step="0.01"
                        placeholder="Ej. 12500"
                        value={formState.generationKwh}
                        onChange={(event) =>
                          setFormState((prev) => ({
                            ...prev,
                            generationKwh: event.target.value,
                          }))
                        }
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="energy-consumption">Consumo (kWh)</Label>
                      <Input
                        id="energy-consumption"
                        type="number"
                        min="0"
                        step="0.01"
                        placeholder="Ej. 14800"
                        value={formState.consumptionKwh}
                        onChange={(event) =>
                          setFormState((prev) => ({
                            ...prev,
                            consumptionKwh: event.target.value,
                          }))
                        }
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="energy-acquired">Energía adquirida CFE (kWh)</Label>
                      <Input
                        id="energy-acquired"
                        type="number"
                        min="0"
                        step="0.01"
                        placeholder="Ej. 3200"
                        value={formState.energyAcquiredCfeKwh}
                        onChange={(event) =>
                          setFormState((prev) => ({
                            ...prev,
                            energyAcquiredCfeKwh: event.target.value,
                          }))
                        }
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="energy-sold">Energía vendida CFE (kWh)</Label>
                      <Input
                        id="energy-sold"
                        type="number"
                        min="0"
                        step="0.01"
                        placeholder="Ej. 950"
                        value={formState.energySoldCfeKwh}
                        onChange={(event) =>
                          setFormState((prev) => ({
                            ...prev,
                            energySoldCfeKwh: event.target.value,
                          }))
                        }
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="energy-cfe-bill">Factura actual de CFE (MXN)</Label>
                      <Input
                        id="energy-cfe-bill"
                        type="number"
                        min="0"
                        step="0.01"
                        placeholder="Ej. 42500"
                        value={formState.cfeBill}
                        onChange={(event) =>
                          setFormState((prev) => ({
                            ...prev,
                            cfeBill: event.target.value,
                          }))
                        }
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="energy-kiwapower">Tarifa Kiwapower (MXN/kWh)</Label>
                      <Input
                        id="energy-kiwapower"
                        type="number"
                        min="0"
                        step="0.0001"
                        placeholder="Ej. 1.32"
                        value={formState.kiwapowerRate}
                        onChange={(event) =>
                          setFormState((prev) => ({
                            ...prev,
                            kiwapowerRate: event.target.value,
                          }))
                        }
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="energy-replacement">Tarifa reemplazo (MXN/kWh)</Label>
                      <Input
                        id="energy-replacement"
                        type="number"
                        min="0"
                        step="0.0001"
                        placeholder="Ej. 2.04"
                        value={formState.replacementRate}
                        onChange={(event) =>
                          setFormState((prev) => ({
                            ...prev,
                            replacementRate: event.target.value,
                          }))
                        }
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="energy-cfe-rate">Tarifa CFE (MXN/kWh)</Label>
                      <Input
                        id="energy-cfe-rate"
                        type="number"
                        min="0"
                        step="0.0001"
                        placeholder="Ej. 2.12"
                        value={formState.cfeRate}
                        onChange={(event) =>
                          setFormState((prev) => ({
                            ...prev,
                            cfeRate: event.target.value,
                          }))
                        }
                        required
                      />
                    </div>
                    <div className="md:col-span-2 flex flex-wrap items-center justify-between gap-3">
                      <p className="text-xs text-gray-500">
                        Mes, año y KPIs se calculan automáticamente al guardar.
                      </p>
                      <Button className="gap-2 bg-emerald-600 hover:bg-emerald-700" type="submit">
                        <Plus className="h-4 w-4" />
                        Guardar registro
                      </Button>
                    </div>
                  </form>
                </CardContent>
              </>
            )}
          </Card>

          <Card className="border-gray-200">
            <CardContent className="p-0">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Fecha</TableHead>
                    <TableHead>Generación</TableHead>
                    <TableHead>Consumo</TableHead>
                    <TableHead>Adquirida CFE</TableHead>
                    <TableHead>Vendida CFE</TableHead>
                    <TableHead>Factura CFE</TableHead>
                    <TableHead>Tarifa Kiwapower</TableHead>
                    <TableHead>Tarifa reemplazo</TableHead>
                    <TableHead>Tarifa CFE</TableHead>
                    <TableHead>Cobertura FV bruta</TableHead>
                    <TableHead>Cobertura FV neta</TableHead>
                    <TableHead>Dependencia bruta</TableHead>
                    <TableHead>Dependencia neta</TableHead>
                    <TableHead>Autosuficiencia</TableHead>
                    <TableHead>% Exportación</TableHead>
                    <TableHead>% Importación</TableHead>
                    <TableHead>Índice Imp-Exp</TableHead>
                    {canManageEnergy && <TableHead className="text-right">Acciones</TableHead>}
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {isLoading && (
                    <TableRow>
                      <TableCell colSpan={canManageEnergy ? 18 : 17}>
                        Cargando registros...
                      </TableCell>
                    </TableRow>
                  )}
                  {!isLoading && filteredRecords.length === 0 && (
                    <TableRow>
                      <TableCell colSpan={canManageEnergy ? 18 : 17}>
                        Sin datos para el periodo seleccionado.
                      </TableCell>
                    </TableRow>
                  )}
                  {!isLoading &&
                    filteredRecords.map((record) => {
                      const kpis = buildRecordKpis(record);
                      return (
                        <TableRow key={record.id}>
                          <TableCell className="font-medium text-gray-900">
                            {formatDate(record.date)}
                          </TableCell>
                          <TableCell>
                            {numberFormatter.format(record.generationKwh)} kWh
                          </TableCell>
                          <TableCell>
                            {numberFormatter.format(record.consumptionKwh)} kWh
                          </TableCell>
                          <TableCell>
                            {numberFormatter.format(record.energyAcquiredCfeKwh)} kWh
                          </TableCell>
                          <TableCell>
                            {numberFormatter.format(record.energySoldCfeKwh)} kWh
                          </TableCell>
                          <TableCell>{currencyFormatter.format(record.cfeBill)}</TableCell>
                          <TableCell>{currencyFormatter.format(record.kiwapowerRate)}</TableCell>
                          <TableCell>{currencyFormatter.format(record.replacementRate)}</TableCell>
                          <TableCell>{currencyFormatter.format(record.cfeRate)}</TableCell>
                          <TableCell>
                            {formatPercent(record.generationKwh, record.consumptionKwh)}
                          </TableCell>
                          <TableCell>
                            {formatPercent(
                              record.generationKwh - record.energySoldCfeKwh,
                              record.consumptionKwh,
                            )}
                          </TableCell>
                          <TableCell>
                            {formatPercent(record.energyAcquiredCfeKwh, record.consumptionKwh)}
                          </TableCell>
                          <TableCell>
                            {formatPercent(
                              record.energyAcquiredCfeKwh - record.energySoldCfeKwh,
                              record.consumptionKwh,
                            )}
                          </TableCell>
                          <TableCell>{percentFormatter.format(kpis.selfSufficiency)}</TableCell>
                          <TableCell>
                            {formatPercent(record.energySoldCfeKwh, record.generationKwh)}
                          </TableCell>
                          <TableCell>
                            {formatPercent(record.energyAcquiredCfeKwh, record.consumptionKwh)}
                          </TableCell>
                          <TableCell>{percentFormatter.format(kpis.importExportIndex)}</TableCell>
                          {canManageEnergy && (
                            <TableCell>
                              <div className="flex justify-end gap-2">
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="gap-1"
                                  onClick={() => handleEditRecord(record.id)}
                                >
                                  Editar
                                </Button>
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="gap-1 text-rose-600"
                                  onClick={() => handleDeleteRecord(record.id)}
                                >
                                  <Trash2 className="h-4 w-4" />
                                  Eliminar
                                </Button>
                              </div>
                            </TableCell>
                          )}
                        </TableRow>
                      );
                    })}
                </TableBody>
              </Table>
            </CardContent>
          </Card>
        </section>
      </div>
    </SGIDashboardLayout>
  );
}