"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import { supabase, type Database } from "@/lib/supabase";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { addMonths, getQuarterLabel } from "@/lib/supplier-evaluations";
import { Eye, Pencil, Plus, Search, Trash2 } from "lucide-react";
import { toast } from "sonner";
import { User } from "@supabase/supabase-js";

type Supplier = Database["public"]["Tables"]["suppliers"]["Row"];
type SupplierEvaluation = Database["public"]["Tables"]["supplier_evaluations"]["Row"];

type SupplierFormState = {
  name: string;
  contact: string;
  materialType: string;
  location: string;
  purchaseType: Supplier["purchase_type"];
};

const defaultFormState: SupplierFormState = {
  name: "",
  contact: "",
  materialType: "",
  location: "",
  purchaseType: "direct",
};

const purchaseTypeLabels: Record<Supplier["purchase_type"], string> = {
  direct: "Directas",
  indirect: "Indirectas",
  direct_indirect: "Directa e indirecta",
};

const getPurchaseTypeLabel = (purchaseType: Supplier["purchase_type"]) =>
  purchaseTypeLabels[purchaseType] ?? "Sin tipo";

const allowedManagers = new Set([
  "ehs.mexico@meiwa.com.mx",
  "comercial@meiwa.com.mx",
  "compras@meiwa.com.mx",
  "qc2.mexico@meiwa.com.mx",  
]);

const formatDate = (value: string | null) =>
  value ? new Date(value).toLocaleDateString("es-MX") : "Sin fecha";

const getCurrentQuarter = () => Math.floor(new Date().getMonth() / 3) + 1;

export default function ExternalSuppliersPage() {
  const router = useRouter();
  const [user, setUser] = useState<User | null>(null);
  const [suppliers, setSuppliers] = useState<Supplier[]>([]);
  const [evaluations, setEvaluations] = useState<SupplierEvaluation[]>([]);
  const [loading, setLoading] = useState(true);
  const [search, setSearch] = useState("");
  const [purchaseTypeFilter, setPurchaseTypeFilter] = useState<string>("all");
  const [materialFilter, setMaterialFilter] = useState("");
  const [locationFilter, setLocationFilter] = useState("");
  const [yearFilter, setYearFilter] = useState<string>(() => new Date().getFullYear().toString());
  const [quarterFilter, setQuarterFilter] = useState<string>(() => getCurrentQuarter().toString());
  const [dialogOpen, setDialogOpen] = useState(false);
  const [editingSupplier, setEditingSupplier] = useState<Supplier | null>(null);
  const [formState, setFormState] = useState<SupplierFormState>(defaultFormState);

  const canManage = allowedManagers.has(user?.email ?? "");

  const fetchData = useCallback(async () => {
    setLoading(true);
    const { data: { session } } = await supabase.auth.getSession();

    if (!session?.user) {
      router.push("/");
      return;
    }

    setUser(session.user);

    const [supplierResponse, evaluationResponse] = await Promise.all([
      supabase.from("suppliers").select("*").order("created_at", { ascending: false }),
      supabase.from("supplier_evaluations").select("*"),
    ]);

    if (supplierResponse.error) {
      toast.error("No se pudieron cargar los proveedores.");
      console.error(supplierResponse.error);
    }

    if (evaluationResponse.error) {
      toast.error("No se pudieron cargar las evaluaciones.");
      console.error(evaluationResponse.error);
    }

    setSuppliers(supplierResponse.data ?? []);
    setEvaluations(evaluationResponse.data ?? []);
    setLoading(false);
  }, [router]);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  const supplierMap = useMemo(() => {
    return suppliers.reduce((acc, supplier) => {
      acc.set(supplier.id, supplier);
      return acc;
    }, new Map<string, Supplier>());
  }, [suppliers]);

  const filteredSuppliers = useMemo(() => {
    const term = search.toLowerCase();
    return suppliers.filter((supplier) => {
      const matchesSearch =
        supplier.name.toLowerCase().includes(term) ||
        (supplier.material_type ?? "").toLowerCase().includes(term) ||
        (supplier.location ?? "").toLowerCase().includes(term);
      const matchesPurchase =
        purchaseTypeFilter === "all" || supplier.purchase_type === purchaseTypeFilter;
      const matchesMaterial =
        !materialFilter ||
        (supplier.material_type ?? "").toLowerCase().includes(materialFilter.toLowerCase());
      const matchesLocation =
        !locationFilter ||
        (supplier.location ?? "").toLowerCase().includes(locationFilter.toLowerCase());

      return matchesSearch && matchesPurchase && matchesMaterial && matchesLocation;
    });
  }, [suppliers, search, purchaseTypeFilter, materialFilter, locationFilter]);

  const evaluationsBySupplier = useMemo(() => {
    return evaluations.reduce((acc, evaluation) => {
      const existing = acc.get(evaluation.supplier_id) ?? [];
      existing.push(evaluation);
      acc.set(evaluation.supplier_id, existing);
      return acc;
    }, new Map<string, SupplierEvaluation[]>());
  }, [evaluations]);

  const latestEvaluationBySupplier = useMemo(() => {
    const latestMap = new Map<string, SupplierEvaluation | null>();
    evaluationsBySupplier.forEach((supplierEvaluations, supplierId) => {
      const sorted = [...supplierEvaluations].sort(
        (a, b) => new Date(b.evaluation_date).getTime() - new Date(a.evaluation_date).getTime()
      );
      latestMap.set(supplierId, sorted[0] ?? null);
    });
    return latestMap;
  }, [evaluationsBySupplier]);

  const yearOptions = useMemo(() => {
    const years = new Set<number>();
    evaluations.forEach((evaluation) => years.add(evaluation.year));
    years.add(new Date().getFullYear());
    return Array.from(years).sort((a, b) => b - a);
  }, [evaluations]);

  const filteredEvaluationData = useMemo(() => {
    const allowedSupplierIds = new Set(filteredSuppliers.map((supplier) => supplier.id));
    return evaluations.filter((evaluation) => {
      const matchesYear = yearFilter === "all" || evaluation.year.toString() === yearFilter;
      const matchesQuarter =
        quarterFilter === "all" || evaluation.quarter.toString() === quarterFilter;
      return allowedSupplierIds.has(evaluation.supplier_id) && matchesYear && matchesQuarter;
    });
  }, [evaluations, filteredSuppliers, yearFilter, quarterFilter]);

  const kpiData = useMemo(() => {
    const totalSuppliers = filteredSuppliers.length;
    const evaluationsInPeriod = filteredEvaluationData.length;
    const approvedCount = filteredEvaluationData.filter((evaluation) => evaluation.status === "APROBADO").length;
    const rejectedCount = filteredEvaluationData.filter((evaluation) => evaluation.status === "REPROBADO").length;
    const averageScore =
      evaluationsInPeriod === 0
        ? 0
        : filteredEvaluationData.reduce((sum, evaluation) => sum + evaluation.final_score, 0) /
          evaluationsInPeriod;

    const currentYear = new Date().getFullYear();
    const currentQuarter = getCurrentQuarter();
    const overdueSuppliers = filteredSuppliers.filter((supplier) => {
      const latest = latestEvaluationBySupplier.get(supplier.id);
      if (!latest) return true;
      if (latest.year < currentYear) return true;
      if (latest.year === currentYear && latest.quarter < currentQuarter) return true;
      return false;
    }).length;

    const topByScore = [...filteredEvaluationData]
      .sort((a, b) => b.final_score - a.final_score)
      .slice(0, 5)
      .map((evaluation) => ({
        supplier: supplierMap.get(evaluation.supplier_id)?.name ?? "Proveedor sin nombre",
        score: evaluation.final_score,
      }));

    const bottomByScore = [...filteredEvaluationData]
      .sort((a, b) => a.final_score - b.final_score)
      .slice(0, 5)
      .map((evaluation) => ({
        supplier: supplierMap.get(evaluation.supplier_id)?.name ?? "Proveedor sin nombre",
        score: evaluation.final_score,
      }));

    return {
      totalSuppliers,
      evaluationsInPeriod,
      approvedCount,
      rejectedCount,
      overdueSuppliers,
      averageScore,
      topByScore,
      bottomByScore,
    };
  }, [filteredSuppliers, filteredEvaluationData, latestEvaluationBySupplier, supplierMap]);

  const handleOpenCreate = () => {
    if (!canManage) {
      toast.error("No tienes permisos para registrar proveedores.");
      return;
    }
    setEditingSupplier(null);
    setFormState(defaultFormState);
    setDialogOpen(true);
  };

  const handleOpenEdit = (supplier: Supplier) => {
    if (!canManage) {
      toast.error("No tienes permisos para editar proveedores.");
      return;
    }
    setEditingSupplier(supplier);
    setFormState({
      name: supplier.name,
      contact: supplier.contact_email_or_contact ?? "",
      materialType: supplier.material_type ?? "",
      location: supplier.location ?? "",
      purchaseType: supplier.purchase_type,
    });
    setDialogOpen(true);
  };

  const handleSaveSupplier = async (event: React.FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!canManage) {
      toast.error("No tienes permisos para guardar proveedores.");
      return;
    }
    const payload = {
      name: formState.name.trim(),
      contact_email_or_contact: formState.contact.trim() || null,
      material_type: formState.materialType.trim() || null,
      location: formState.location.trim() || null,
      purchase_type: formState.purchaseType,
    };

    if (!payload.name) {
      toast.error("El nombre del proveedor es obligatorio.");
      return;
    }

    if (editingSupplier) {
      const { error } = await supabase.from("suppliers").update(payload).eq("id", editingSupplier.id);
      if (error) {
        toast.error("No se pudo actualizar el proveedor.");
        console.error(error);
        return;
      }
      toast.success("Proveedor actualizado.");
    } else {
      const { error } = await supabase.from("suppliers").insert(payload);
      if (error) {
        toast.error("No se pudo crear el proveedor.");
        console.error(error);
        return;
      }
      toast.success("Proveedor creado.");
    }

    setDialogOpen(false);
    fetchData();
  };

  const handleDeleteSupplier = async (supplier: Supplier) => {
    if (!canManage) {
      toast.error("No tienes permisos para eliminar proveedores.");
      return;
    }
    const confirmation = window.confirm(`¿Deseas eliminar el proveedor ${supplier.name}?`);
    if (!confirmation) return;

    const { error } = await supabase.from("suppliers").delete().eq("id", supplier.id);
    if (error) {
      toast.error("No se pudo eliminar el proveedor.");
      console.error(error);
      return;
    }

    setSuppliers((prev) => prev.filter((item) => item.id !== supplier.id));
    toast.success("Proveedor eliminado.");
  };

  return (
    <SGIDashboardLayout>
      <div className="space-y-6">
        <div className="flex flex-wrap items-start justify-between gap-4">
          <div>
            <h1 className="text-2xl font-semibold text-gray-900">Servicios externalizados · Proveedores</h1>
            <p className="text-sm text-gray-500">
              Administra proveedores, controla evaluaciones trimestrales y consulta indicadores clave.
            </p>
          </div>
          <div className="flex flex-wrap gap-2">
            <Button
              variant="outline"
              onClick={() => router.push("/meiwa-core/meiwa-sgi/operation/external/evaluations")}
            >
              Evaluaciones de proveedores
            </Button>
            <Button onClick={handleOpenCreate} className="gap-2" disabled={!canManage}>
              <Plus className="h-4 w-4" />
              Nuevo proveedor
            </Button>
          </div>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Filtros globales y KPIs</CardTitle>
          </CardHeader>
          <CardContent className="space-y-6">
            <div className="grid gap-4 md:grid-cols-5">
              <div>
                <Label>Año</Label>
                <Select value={yearFilter} onValueChange={setYearFilter}>
                  <SelectTrigger>
                    <SelectValue placeholder="Todos" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    {yearOptions.map((year) => (
                      <SelectItem key={year} value={year.toString()}>
                        {year}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label>Trimestre</Label>
                <Select value={quarterFilter} onValueChange={setQuarterFilter}>
                  <SelectTrigger>
                    <SelectValue placeholder="Todos" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    {[1, 2, 3, 4].map((quarter) => (
                      <SelectItem key={quarter} value={quarter.toString()}>
                        Q{quarter}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label>Tipo de compra</Label>
                <Select value={purchaseTypeFilter} onValueChange={setPurchaseTypeFilter}>
                  <SelectTrigger>
                    <SelectValue placeholder="Todas" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todas</SelectItem>
                    <SelectItem value="direct">Directas</SelectItem>
                    <SelectItem value="indirect">Indirectas</SelectItem>
                    <SelectItem value="direct_indirect">Directa e indirecta</SelectItem>             
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label>Material</Label>
                <Input
                  value={materialFilter}
                  onChange={(event) => setMaterialFilter(event.target.value)}
                  placeholder="Buscar material"
                />
              </div>
              <div>
                <Label>Ubicación</Label>
                <Input
                  value={locationFilter}
                  onChange={(event) => setLocationFilter(event.target.value)}
                  placeholder="Buscar ubicación"
                />
              </div>
            </div>

            <div className="grid gap-4 md:grid-cols-3 xl:grid-cols-6">
              <Card className="bg-slate-50">
                <CardHeader>
                  <CardTitle className="text-sm text-gray-500">Proveedores totales</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-2xl font-semibold text-gray-900">{kpiData.totalSuppliers}</p>
                </CardContent>
              </Card>
              <Card className="bg-slate-50">
                <CardHeader>
                  <CardTitle className="text-sm text-gray-500">Evaluaciones en periodo</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-2xl font-semibold text-gray-900">{kpiData.evaluationsInPeriod}</p>
                </CardContent>
              </Card>
              <Card className="bg-slate-50">
                <CardHeader>
                  <CardTitle className="text-sm text-gray-500">% Aprobados</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-2xl font-semibold text-emerald-600">
                    {kpiData.evaluationsInPeriod
                      ? Math.round((kpiData.approvedCount / kpiData.evaluationsInPeriod) * 100)
                      : 0}
                    %
                  </p>
                </CardContent>
              </Card>
              <Card className="bg-slate-50">
                <CardHeader>
                  <CardTitle className="text-sm text-gray-500">% Reprobados</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-2xl font-semibold text-rose-600">
                    {kpiData.evaluationsInPeriod
                      ? Math.round((kpiData.rejectedCount / kpiData.evaluationsInPeriod) * 100)
                      : 0}
                    %
                  </p>
                </CardContent>
              </Card>
              <Card className="bg-slate-50">
                <CardHeader>
                  <CardTitle className="text-sm text-gray-500">Proveedores vencidos</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-2xl font-semibold text-amber-600">{kpiData.overdueSuppliers}</p>
                </CardContent>
              </Card>
              <Card className="bg-slate-50">
                <CardHeader>
                  <CardTitle className="text-sm text-gray-500">Promedio ScoreFinal</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-2xl font-semibold text-blue-600">{kpiData.averageScore.toFixed(1)}</p>
                </CardContent>
              </Card>
            </div>

            <div className="grid gap-4 md:grid-cols-2">
              <Card className="bg-white">
                <CardHeader>
                  <CardTitle className="text-sm text-gray-500">Top 5 proveedores</CardTitle>
                </CardHeader>
                <CardContent className="space-y-2">
                  {kpiData.topByScore.length === 0 ? (
                    <p className="text-sm text-gray-500">Sin datos para el periodo.</p>
                  ) : (
                    kpiData.topByScore.map((item) => (
                      <div key={item.supplier} className="flex items-center justify-between text-sm">
                        <span>{item.supplier}</span>
                        <Badge>{item.score.toFixed(1)}</Badge>
                      </div>
                    ))
                  )}
                </CardContent>
              </Card>
              <Card className="bg-white">
                <CardHeader>
                  <CardTitle className="text-sm text-gray-500">Bottom 5 proveedores</CardTitle>
                </CardHeader>
                <CardContent className="space-y-2">
                  {kpiData.bottomByScore.length === 0 ? (
                    <p className="text-sm text-gray-500">Sin datos para el periodo.</p>
                  ) : (
                    kpiData.bottomByScore.map((item) => (
                      <div key={item.supplier} className="flex items-center justify-between text-sm">
                        <span>{item.supplier}</span>
                        <Badge variant="destructive">{item.score.toFixed(1)}</Badge>
                      </div>
                    ))
                  )}
                </CardContent>
              </Card>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="space-y-4">
            <CardTitle>Proveedores</CardTitle>
            <div className="flex flex-wrap gap-2">
              <div className="relative flex-1 min-w-[240px]">
                <Search className="absolute left-3 top-3 h-4 w-4 text-gray-400" />
                <Input
                  className="pl-9"
                  value={search}
                  onChange={(event) => setSearch(event.target.value)}
                  placeholder="Buscar por nombre, material o ubicación"
                />
              </div>
            </div>
          </CardHeader>
          <CardContent>
            {loading ? (
              <p className="text-sm text-gray-500">Cargando proveedores...</p>
            ) : (
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Proveedor</TableHead>
                    <TableHead>Tipo de compra</TableHead>
                    <TableHead>Material</TableHead>
                    <TableHead>Ubicación</TableHead>
                    <TableHead>Última evaluación</TableHead>
                    <TableHead>Próxima evaluación</TableHead>
                    <TableHead>Estado</TableHead>
                    <TableHead className="text-right">Acciones</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {filteredSuppliers.length === 0 ? (
                    <TableRow>
                      <TableCell colSpan={8} className="text-center text-sm text-gray-500">
                        No hay proveedores con esos filtros.
                      </TableCell>
                    </TableRow>
                  ) : (
                    filteredSuppliers.map((supplier) => {
                      const latestEvaluation = latestEvaluationBySupplier.get(supplier.id) ?? null;
                      const nextEvaluationDate = latestEvaluation
                        ? addMonths(new Date(latestEvaluation.evaluation_date), 3)
                        : null;
                      const currentYear = new Date().getFullYear();
                      const currentQuarter = getCurrentQuarter();
                      const status = latestEvaluation
                        ? latestEvaluation.year === currentYear && latestEvaluation.quarter === currentQuarter
                          ? "Al día"
                          : latestEvaluation.year < currentYear ||
                            (latestEvaluation.year === currentYear && latestEvaluation.quarter < currentQuarter)
                          ? "Vencida"
                          : "Al día"
                        : "Pendiente";

                      return (
                        <TableRow key={supplier.id}>
                          <TableCell>
                            <div className="space-y-1">
                              <p className="font-medium text-gray-900">{supplier.name}</p>
                              <p className="text-xs text-gray-500">{supplier.contact_email_or_contact ?? "Sin contacto"}</p>
                            </div>
                          </TableCell>
                          <TableCell>
                            <Badge variant="secondary">
                              {getPurchaseTypeLabel(supplier.purchase_type)}
                            </Badge>
                          </TableCell>
                          <TableCell>{supplier.material_type ?? "Sin material"}</TableCell>
                          <TableCell>{supplier.location ?? "Sin ubicación"}</TableCell>
                          <TableCell>
                            {latestEvaluation ? (
                              <div className="space-y-1">
                                <p className="text-sm text-gray-900">{formatDate(latestEvaluation.evaluation_date)}</p>
                                <p className="text-xs text-gray-500">
                                  {getQuarterLabel(latestEvaluation.year, latestEvaluation.quarter)}
                                </p>
                              </div>
                            ) : (
                              "Pendiente"
                            )}
                          </TableCell>
                          <TableCell>
                            {nextEvaluationDate ? nextEvaluationDate.toLocaleDateString("es-MX") : "Pendiente"}
                          </TableCell>
                          <TableCell>
                            <Badge variant={status === "Vencida" ? "destructive" : status === "Pendiente" ? "outline" : "default"}>
                              {status}
                            </Badge>
                          </TableCell>
                          <TableCell className="text-right">
                            <div className="flex justify-end gap-2">
                              <Button
                                variant="ghost"
                                size="icon"
                                onClick={() =>
                                  router.push(
                                    `/meiwa-core/meiwa-sgi/operation/external/evaluations?supplierId=${supplier.id}`
                                  )
                                }
                              >
                                <Eye className="h-4 w-4" />
                              </Button>
                              <Button variant="ghost" size="icon" onClick={() => handleOpenEdit(supplier)} disabled={!canManage}>
                                <Pencil className="h-4 w-4" />
                              </Button>
                              <Button variant="ghost" size="icon" onClick={() => handleDeleteSupplier(supplier)} disabled={!canManage}>
                                <Trash2 className="h-4 w-4 text-rose-500" />
                              </Button>
                            </div>
                          </TableCell>
                        </TableRow>
                      );
                    })
                  )}
                </TableBody>
              </Table>
            )}
          </CardContent>
        </Card>
      </div>

      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="max-w-lg">
          <DialogHeader>
            <DialogTitle>{editingSupplier ? "Editar proveedor" : "Nuevo proveedor"}</DialogTitle>
          </DialogHeader>
          <form onSubmit={handleSaveSupplier} className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="supplier-name">Nombre del proveedor</Label>
              <Input
                id="supplier-name"
                value={formState.name}
                onChange={(event) => setFormState((prev) => ({ ...prev, name: event.target.value }))}
                required
                disabled={!canManage}
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="supplier-contact">Correo / Contacto</Label>
              <Input
                id="supplier-contact"
                value={formState.contact}
                onChange={(event) => setFormState((prev) => ({ ...prev, contact: event.target.value }))}
                disabled={!canManage}
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="supplier-material">Tipo de material</Label>
              <Input
                id="supplier-material"
                value={formState.materialType}
                onChange={(event) => setFormState((prev) => ({ ...prev, materialType: event.target.value }))}
                disabled={!canManage}
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="supplier-location">Ubicación</Label>
              <Input
                id="supplier-location"
                value={formState.location}
                onChange={(event) => setFormState((prev) => ({ ...prev, location: event.target.value }))}
                disabled={!canManage}
              />
            </div>
            <div className="space-y-2">
              <Label>Tipo de compras</Label>
              <Select
                value={formState.purchaseType}
                onValueChange={(value) =>
                  setFormState((prev) => ({ ...prev, purchaseType: value as Supplier["purchase_type"] }))
                }
                disabled={!canManage}
              >
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="direct">Directas</SelectItem>
                  <SelectItem value="indirect">Indirectas</SelectItem>
                  <SelectItem value="direct_indirect">Directa e indirecta</SelectItem>                  
                </SelectContent>
              </Select>
            </div>
            <div className="flex justify-end gap-2">
              <Button type="button" variant="ghost" onClick={() => setDialogOpen(false)}>
                Cancelar
              </Button>
              <Button type="submit" disabled={!canManage}>
                {editingSupplier ? "Guardar cambios" : "Registrar proveedor"}
              </Button>
            </div>
          </form>
        </DialogContent>
      </Dialog>
    </SGIDashboardLayout>
  );
}