"use client";

import { type FormEvent, useEffect, useMemo, useState } from "react";
import { type User } from "@supabase/supabase-js";
import { toast } from "sonner";
import {
  BarChart3,
  Droplet,
  Filter,
  Flame,
  Fuel,
  LineChart,
  PackageSearch,
  Pencil,
  Plus,
  Trash2,
} from "lucide-react";

import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Textarea } from "@/components/ui/textarea";
import { supabase } from "@/lib/supabase";

type GasolineRecord = {
  id: string;
  date: string;
  month: string;
  year: number;
  budget: number;
  fuelType: "Magna/Diesel";
  averageCostPerLiter: number;
  equivalentLiters: number;
  notes?: string;
};

type GasolineFormState = {
  date: string;
  budget: string;
  fuelType: "Magna/Diesel";
  averageCostPerLiter: string;
  equivalentLiters: string;
  notes: string;
};

type GasRecord = {
  id: string;
  date: string;
  month: string;
  year: number;
  gasType: string;
  tankCapacity: number;
  capacityUnit: "L" | "kg";
  quantity: number;
  totalLiters: number;
  unitCost: number;
  costUnit: "Por litro" | "Por unidad";
  totalCost: number;
  provider: string;
};

type GasFormState = {
  date: string;
  gasType: string;
  tankCapacity: string;
  capacityUnit: "L" | "kg";
  quantity: string;
  totalLiters: string;
  unitCost: string;
  costUnit: "Por litro" | "Por unidad";
  provider: string;
};

const gasolineEnglishColumns = {
  date: "date",
  month: "month",
  year: "year",
  budget: "budget",
  fuelType: "fuel_type",
  averageCostPerLiter: "average_cost_per_liter",
  equivalentLiters: "equivalent_liters",
  notes: "notes",
  createdBy: "created_by",
} as const;

const gasolineSpanishColumns = {
  date: "fecha",
  month: "mes",
  year: "anio",
  budget: "presupuesto",
  fuelType: "tipo_combustible",
  averageCostPerLiter: "costo_promedio_litro",
  equivalentLiters: "litros_equivalentes",
  notes: "observaciones",
  createdBy: "created_by",
} as const;

type GasolineColumnConfig = typeof gasolineEnglishColumns | typeof gasolineSpanishColumns;

const gasEnglishColumns = {
  date: "date",
  month: "month",
  year: "year",
  gasType: "gas_type",
  tankCapacity: "tank_capacity",
  capacityUnit: "capacity_unit",
  quantity: "quantity",
  totalLiters: "total_liters",
  unitCost: "unit_cost",
  costUnit: "cost_unit",
  totalCost: "total_cost",
  provider: "provider",
  createdBy: "created_by",
} as const;

const gasSpanishColumns = {
  date: "fecha",
  month: "mes",
  year: "anio",
  gasType: "tipo_gas",
  tankCapacity: "capacidad_tanque",
  capacityUnit: "unidad_capacidad",
  quantity: "cantidad",
  totalLiters: "litros_totales",
  unitCost: "costo_unitario",
  costUnit: "unidad_costo",
  totalCost: "total",
  provider: "proveedor",
  createdBy: "created_by",
} as const;

type GasColumnConfig = typeof gasEnglishColumns | typeof gasSpanishColumns;

const allowedManagers = new Set([
  "ehs.mexico@meiwa.com.mx",
  "admin2.mexico@meiwa.com.mx",
]);

const numberFormatter = new Intl.NumberFormat("es-MX", {
  minimumFractionDigits: 0,
  maximumFractionDigits: 2,
});

const currencyFormatter = new Intl.NumberFormat("es-MX", {
  style: "currency",
  currency: "MXN",
  minimumFractionDigits: 2,
  maximumFractionDigits: 2,
});

const normalizeMonth = (value?: string | null) => {
  if (!value) return "";
  return value.charAt(0).toUpperCase() + value.slice(1).toLowerCase();
};

const toSpanishMonth = (dateValue: string) =>
  normalizeMonth(
    new Date(`${dateValue}T00:00:00`).toLocaleDateString("es-MX", {
      month: "long",
    }),
  );

const formatDate = (value: string) =>
  new Date(`${value}T00:00:00`).toLocaleDateString("es-MX", {
    day: "2-digit",
    month: "short",
    year: "numeric",
  });

const sortByDateDesc = <T extends { date: string }>(records: T[]) =>
  [...records].sort((first, second) => {
    const firstTime = new Date(`${first.date}T00:00:00`).getTime();
    const secondTime = new Date(`${second.date}T00:00:00`).getTime();
    if (Number.isNaN(firstTime) && Number.isNaN(secondTime)) return 0;
    if (Number.isNaN(firstTime)) return 1;
    if (Number.isNaN(secondTime)) return -1;
    return secondTime - firstTime;
  });

const StatCard = ({
  title,
  value,
  helper,
  icon: Icon,
  accent,
}: {
  title: string;
  value: string;
  helper: string;
  icon: typeof Fuel;
  accent: string;
}) => (
  <Card className="border-gray-200">
    <CardContent className="flex items-start justify-between gap-4 p-6">
      <div>
        <p className="text-xs font-semibold uppercase tracking-[0.2em] text-gray-500">
          {title}
        </p>
        <p className="mt-3 text-2xl font-semibold text-gray-900">{value}</p>
        <p className="mt-2 text-sm text-gray-500">{helper}</p>
      </div>
      <div className={`rounded-2xl p-3 ${accent}`}>
        <Icon className="h-5 w-5 text-white" />
      </div>
    </CardContent>
  </Card>
);

const spanishMonths = [
  "Enero",
  "Febrero",
  "Marzo",
  "Abril",
  "Mayo",
  "Junio",
  "Julio",
  "Agosto",
  "Septiembre",
  "Octubre",
  "Noviembre",
  "Diciembre",
];

const detectGasolineConfig = (
  records: Record<string, unknown>[],
  fallback: GasolineColumnConfig,
): GasolineColumnConfig => {
  const sample = records[0];
  if (!sample) return fallback;
  if ("fecha" in sample || "presupuesto" in sample || "tipo_combustible" in sample) {
    return gasolineSpanishColumns;
  }
  if ("date" in sample || "budget" in sample || "fuel_type" in sample) {
    return gasolineEnglishColumns;
  }
  return fallback;
};

const detectGasConfig = (
  records: Record<string, unknown>[],
  fallback: GasColumnConfig,
): GasColumnConfig => {
  const sample = records[0];
  if (!sample) return fallback;
  if ("fecha" in sample || "tipo_gas" in sample || "proveedor" in sample) {
    return gasSpanishColumns;
  }
  if ("date" in sample || "gas_type" in sample || "provider" in sample) {
    return gasEnglishColumns;
  }
  return fallback;
};

const mapGasolineRecord = (
  record: Record<string, unknown>,
  config: GasolineColumnConfig,
): GasolineRecord => {
  const dateValue = (record[config.date] as string) ?? "";
  const budget = (record[config.budget] as number) ?? 0;
  const averageCostPerLiter = (record[config.averageCostPerLiter] as number) ?? 0;
  const equivalentLiters =
    (record[config.equivalentLiters] as number) ??
    (averageCostPerLiter ? budget / averageCostPerLiter : 0);
  return {
    id: record.id as string,
    date: dateValue,
    month: normalizeMonth((record[config.month] as string) ?? toSpanishMonth(dateValue)),
    year:
      (record[config.year] as number) ??
      new Date(`${dateValue}T00:00:00`).getFullYear(),
    budget,
    fuelType:
      (record[config.fuelType] as GasolineRecord["fuelType"]) ?? "Magna/Diesel",
    averageCostPerLiter,
    equivalentLiters,
    notes: (record[config.notes] as string) ?? "",
  };
};

const mapGasRecord = (record: Record<string, unknown>, config: GasColumnConfig): GasRecord => {
  const dateValue = (record[config.date] as string) ?? "";
  const tankCapacity = (record[config.tankCapacity] as number) ?? 0;
  const quantity = (record[config.quantity] as number) ?? 0;
  const totalLiters =
    (record[config.totalLiters] as number) ?? tankCapacity * quantity;
  const unitCost = (record[config.unitCost] as number) ?? 0;
  const costUnit =
    (record[config.costUnit] as GasRecord["costUnit"]) ?? "Por litro";
  const totalCost =
    (record[config.totalCost] as number) ??
    (costUnit === "Por litro" ? totalLiters * unitCost : quantity * unitCost);
  return {
    id: record.id as string,
    date: dateValue,
    month: normalizeMonth((record[config.month] as string) ?? toSpanishMonth(dateValue)),
    year:
      (record[config.year] as number) ??
      new Date(`${dateValue}T00:00:00`).getFullYear(),
    gasType: (record[config.gasType] as string) ?? "",
    tankCapacity,
    capacityUnit: (record[config.capacityUnit] as GasRecord["capacityUnit"]) ?? "L",
    quantity,
    totalLiters,
    unitCost,
    costUnit,
    totalCost,
    provider: (record[config.provider] as string) ?? "",
  };
};

const buildGasolinePayload = (
  config: GasolineColumnConfig,
  formState: GasolineFormState,
  parsedDate: Date,
  options: { includeDerived?: boolean; includeCreatedBy?: boolean; userId?: string } = {},
) => {
  const budget = Number(formState.budget);
  const averageCostPerLiter = Number(formState.averageCostPerLiter);
  const providedEquivalent = Number(formState.equivalentLiters);
  const computedEquivalent =
    Number.isFinite(budget) && averageCostPerLiter > 0
      ? budget / averageCostPerLiter
      : 0;
  const equivalentLiters =
    Number.isFinite(providedEquivalent) && providedEquivalent > 0
      ? providedEquivalent
      : computedEquivalent;

  const payload: Record<string, unknown> = {
    [config.date]: formState.date,
    [config.budget]: budget,
    [config.fuelType]: formState.fuelType,
    [config.averageCostPerLiter]: averageCostPerLiter,
    [config.equivalentLiters]: equivalentLiters,
    [config.notes]: formState.notes.trim() || null,
  };

  if (options.includeDerived ?? true) {
    payload[config.month] = toSpanishMonth(formState.date);
    payload[config.year] = parsedDate.getFullYear();
  }

  if (options.includeCreatedBy ?? true) {
    payload[config.createdBy] = options.userId;
  }

  return payload;
};

const buildGasPayload = (
  config: GasColumnConfig,
  formState: GasFormState,
  parsedDate: Date,
  options: { includeDerived?: boolean; includeCreatedBy?: boolean; userId?: string } = {},
) => {
  const tankCapacity = Number(formState.tankCapacity);
  const quantity = Number(formState.quantity);
  const providedTotalLiters = Number(formState.totalLiters);
  const totalLiters =
    Number.isFinite(providedTotalLiters) && providedTotalLiters > 0
      ? providedTotalLiters
      : tankCapacity * quantity;
  const unitCost = Number(formState.unitCost);
  const totalCost =
    formState.costUnit === "Por litro" ? totalLiters * unitCost : quantity * unitCost;

  const payload: Record<string, unknown> = {
    [config.date]: formState.date,
    [config.gasType]: formState.gasType.trim(),
    [config.tankCapacity]: tankCapacity,
    [config.capacityUnit]: formState.capacityUnit,
    [config.quantity]: quantity,
    [config.totalLiters]: totalLiters,
    [config.unitCost]: unitCost,
    [config.costUnit]: formState.costUnit,
    [config.totalCost]: totalCost,
    [config.provider]: formState.provider.trim(),
  };

  if (options.includeDerived ?? true) {
    payload[config.month] = toSpanishMonth(formState.date);
    payload[config.year] = parsedDate.getFullYear();
  }

  if (options.includeCreatedBy ?? true) {
    payload[config.createdBy] = options.userId;
  }

  return payload;
};

export default function FuelsPage() {
  const [user, setUser] = useState<User | null>(null);
  const [gasolineRecords, setGasolineRecords] = useState<GasolineRecord[]>([]);
  const [gasRecords, setGasRecords] = useState<GasRecord[]>([]);
  const [gasolineColumnConfig, setGasolineColumnConfig] =
    useState<GasolineColumnConfig>(gasolineEnglishColumns);
  const [gasColumnConfig, setGasColumnConfig] = useState<GasColumnConfig>(gasEnglishColumns);

  const [gasolineFormState, setGasolineFormState] = useState<GasolineFormState>({
    date: "",
    budget: "",
    fuelType: "Magna/Diesel",
    averageCostPerLiter: "",
    equivalentLiters: "",
    notes: "",
  });

  const [gasFormState, setGasFormState] = useState<GasFormState>({
    date: "",
    gasType: "",
    tankCapacity: "",
    capacityUnit: "L",
    quantity: "",
    totalLiters: "",
    unitCost: "",
    costUnit: "Por litro",
    provider: "",
  });

  const [selectedGasolineYear, setSelectedGasolineYear] = useState("Todo");
  const [selectedGasolineMonth, setSelectedGasolineMonth] = useState("Todo");
  const [selectedGasYear, setSelectedGasYear] = useState("Todo");
  const [selectedGasMonth, setSelectedGasMonth] = useState("Todo");

  const canManage = allowedManagers.has(user?.email ?? "");

  const gasolineYears = useMemo(
    () =>
      Array.from(new Set(gasolineRecords.map((record) => record.year)))
        .filter((year) => Number.isFinite(year))
        .sort((a, b) => b - a),
    [gasolineRecords],
  );

  const gasYears = useMemo(
    () =>
      Array.from(new Set(gasRecords.map((record) => record.year)))
        .filter((year) => Number.isFinite(year))
        .sort((a, b) => b - a),
    [gasRecords],
  );

  const gasolineMonths = useMemo(() => {
    const monthSet = new Set(
      gasolineRecords.map((record) => normalizeMonth(record.month)).filter(Boolean),
    );
    return spanishMonths.filter((month) => monthSet.has(month));
  }, [gasolineRecords]);

  const gasMonths = useMemo(() => {
    const monthSet = new Set(gasRecords.map((record) => normalizeMonth(record.month)).filter(Boolean));
    return spanishMonths.filter((month) => monthSet.has(month));
  }, [gasRecords]);

  const filteredGasolineRecords = useMemo(
    () =>
      gasolineRecords.filter((record) => {
        const matchesYear =
          selectedGasolineYear === "Todo" || record.year === Number(selectedGasolineYear);
        const matchesMonth =
          selectedGasolineMonth === "Todo" ||
          normalizeMonth(record.month) === selectedGasolineMonth;
        return matchesYear && matchesMonth;
      }),
    [gasolineRecords, selectedGasolineMonth, selectedGasolineYear],
  );

  const filteredGasRecords = useMemo(
    () =>
      gasRecords.filter((record) => {
        const matchesYear = selectedGasYear === "Todo" || record.year === Number(selectedGasYear);
        const matchesMonth =
          selectedGasMonth === "Todo" || normalizeMonth(record.month) === selectedGasMonth;
        return matchesYear && matchesMonth;
      }),
    [gasRecords, selectedGasMonth, selectedGasYear],
  );

  const gasolineStats = useMemo(() => {
    const totalBudget = filteredGasolineRecords.reduce((acc, record) => acc + record.budget, 0);
    const totalLiters = filteredGasolineRecords.reduce(
      (acc, record) => acc + record.equivalentLiters,
      0,
    );
    const weightedAverageCost = totalLiters
      ? totalBudget / totalLiters
      : filteredGasolineRecords.reduce((acc, record) => acc + record.averageCostPerLiter, 0) /
        (filteredGasolineRecords.length || 1);
    const byType = filteredGasolineRecords.reduce<Record<string, { budget: number; liters: number }>>(
      (acc, record) => {
        acc[record.fuelType] = acc[record.fuelType] ?? { budget: 0, liters: 0 };
        acc[record.fuelType].budget += record.budget;
        acc[record.fuelType].liters += record.equivalentLiters;
        return acc;
      },
      {},
    );

    return {
      totalBudget,
      totalLiters,
      weightedAverageCost,
      byType,
    };
  }, [filteredGasolineRecords]);

  const gasStats = useMemo(() => {
    const totalLiters = filteredGasRecords.reduce((acc, record) => acc + record.totalLiters, 0);
    const totalCost = filteredGasRecords.reduce((acc, record) => acc + record.totalCost, 0);
    const costUnitBasis = filteredGasRecords.reduce((acc, record) => {
      if (record.costUnit === "Por litro") {
        return acc + record.totalLiters;
      }
      return acc + record.quantity;
    }, 0);
    const averageUnitCost = costUnitBasis ? totalCost / costUnitBasis : 0;
    const providerSpend = filteredGasRecords.reduce<Record<string, number>>((acc, record) => {
      acc[record.provider] = (acc[record.provider] ?? 0) + record.totalCost;
      return acc;
    }, {});
    const topProvider = Object.keys(providerSpend).sort(
      (first, second) => (providerSpend[second] ?? 0) - (providerSpend[first] ?? 0),
    )[0];
    const byType = filteredGasRecords.reduce<Record<string, { cost: number; liters: number }>>(
      (acc, record) => {
        acc[record.gasType] = acc[record.gasType] ?? { cost: 0, liters: 0 };
        acc[record.gasType].cost += record.totalCost;
        acc[record.gasType].liters += record.totalLiters;
        return acc;
      },
      {},
    );

    return {
      totalLiters,
      totalCost,
      averageUnitCost,
      topProvider,
      byType,
    };
  }, [filteredGasRecords]);

  useEffect(() => {
    const loadSession = async () => {
      const { data } = await supabase.auth.getSession();
      setUser(data.session?.user ?? null);
    };

    const loadRecords = async () => {
      const [{ data: gasolineData, error: gasolineError }, { data: gasData, error: gasError }] =
        await Promise.all([
          supabase.from("fuel_gasoline_records").select("*"),
          supabase.from("fuel_gas_records").select("*"),
        ]);

      if (gasolineError) {
        toast.error("No pudimos cargar los registros de gasolina.");
      } else {
        const configToUse = detectGasolineConfig(
          (gasolineData ?? []) as Record<string, unknown>[],
          gasolineColumnConfig,
        );
        if (configToUse !== gasolineColumnConfig) {
          setGasolineColumnConfig(configToUse);
        }
        const mapped = (gasolineData ?? []).map((record) =>
          mapGasolineRecord(record as Record<string, unknown>, configToUse),
        );
        setGasolineRecords(sortByDateDesc(mapped));
      }

      if (gasError) {
        toast.error("No pudimos cargar los registros de gases.");
      } else {
        const configToUse = detectGasConfig(
          (gasData ?? []) as Record<string, unknown>[],
          gasColumnConfig,
        );
        if (configToUse !== gasColumnConfig) {
          setGasColumnConfig(configToUse);
        }
        const mapped = (gasData ?? []).map((record) =>
          mapGasRecord(record as Record<string, unknown>, configToUse),
        );
        setGasRecords(sortByDateDesc(mapped));
      }
    };

    loadSession();
    loadRecords();

    const {
      data: { subscription },
    } = supabase.auth.onAuthStateChange((_event, session) => {
      setUser(session?.user ?? null);
    });

    return () => subscription.unsubscribe();
  }, []);

  const handleAddGasolineRecord = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!canManage) return;

    const budget = Number(gasolineFormState.budget);
    const averageCostPerLiter = Number(gasolineFormState.averageCostPerLiter);
    const providedLiters = Number(gasolineFormState.equivalentLiters);
    const computedLiters =
      Number.isFinite(budget) && averageCostPerLiter > 0
        ? budget / averageCostPerLiter
        : 0;
    const equivalentLiters =
      Number.isFinite(providedLiters) && providedLiters > 0 ? providedLiters : computedLiters;

    if (!gasolineFormState.date.trim() || !gasolineFormState.fuelType.trim()) return;
    if (!Number.isFinite(budget) || budget <= 0) return;
    if (!Number.isFinite(averageCostPerLiter) || averageCostPerLiter <= 0) return;
    if (!Number.isFinite(equivalentLiters) || equivalentLiters <= 0) return;

    const parsedDate = new Date(`${gasolineFormState.date}T00:00:00`);
    if (Number.isNaN(parsedDate.valueOf())) return;

    if (!user) {
      toast.error("Necesitamos una sesión activa para guardar el registro.");
      return;
    }

    const fullPayload = buildGasolinePayload(
      gasolineColumnConfig,
      { ...gasolineFormState, equivalentLiters: String(equivalentLiters) },
      parsedDate,
      {
        includeDerived: true,
        includeCreatedBy: true,
        userId: user.id,
      },
    );
    const basePayload = buildGasolinePayload(
      gasolineColumnConfig,
      { ...gasolineFormState, equivalentLiters: String(equivalentLiters) },
      parsedDate,
      {
        includeDerived: false,
        includeCreatedBy: false,
      },
    );

    const insertRecord = (payload: Record<string, unknown>) =>
      supabase.from("fuel_gasoline_records").insert(payload).select().single();

    let { data, error } = await insertRecord(fullPayload);
    let configToUse = gasolineColumnConfig;

    if (error) {
      const baseAttempt = await insertRecord(basePayload);
      if (!baseAttempt.error) {
        data = baseAttempt.data;
        error = null;
      } else if (gasolineColumnConfig === gasolineEnglishColumns) {
        const spanishFullPayload = buildGasolinePayload(
          gasolineSpanishColumns,
          { ...gasolineFormState, equivalentLiters: String(equivalentLiters) },
          parsedDate,
          {
            includeDerived: true,
            includeCreatedBy: true,
            userId: user.id,
          },
        );
        const spanishBasePayload = buildGasolinePayload(
          gasolineSpanishColumns,
          { ...gasolineFormState, equivalentLiters: String(equivalentLiters) },
          parsedDate,
          {
            includeDerived: false,
            includeCreatedBy: false,
          },
        );

        const spanishFull = await insertRecord(spanishFullPayload);
        if (!spanishFull.error) {
          data = spanishFull.data;
          error = null;
          configToUse = gasolineSpanishColumns;
        } else {
          const spanishBase = await insertRecord(spanishBasePayload);
          data = spanishBase.data;
          error = spanishBase.error;
          if (!spanishBase.error) {
            configToUse = gasolineSpanishColumns;
          }
        }
      }
    }

    if (error || !data) {
      toast.error("No pudimos guardar el registro.");
      return;
    }

    if (configToUse !== gasolineColumnConfig) {
      setGasolineColumnConfig(configToUse);
    }

    setGasolineRecords((prev) => [
      mapGasolineRecord(data as Record<string, unknown>, configToUse),
      ...prev,
    ]);

    setGasolineFormState({
      date: "",
      budget: "",
      fuelType: "Magna/Diesel",
      averageCostPerLiter: "",
      equivalentLiters: "",
      notes: "",
    });

    toast.success("Registro de gasolina guardado.");
  };

  const handleEditGasolineRecord = async (recordId: string) => {
    if (!canManage) return;
    const record = gasolineRecords.find((item) => item.id === recordId);
    if (!record) return;

    const date = window.prompt("Fecha de registro (YYYY-MM-DD)", record.date);
    if (!date) return;
    const fuelTypeInput = window.prompt(
      "Tipo de combustible (Magna/Diesel)",
      record.fuelType,
    );
    if (!fuelTypeInput) return;
    const budgetInput = window.prompt("Presupuesto (MXN)", String(record.budget));
    if (!budgetInput) return;
    const averageCostInput = window.prompt(
      "Costo promedio por litro (MXN/L)",
      String(record.averageCostPerLiter),
    );
    if (!averageCostInput) return;
    const equivalentLitersInput = window.prompt(
      "Litros equivalentes (L)",
      String(record.equivalentLiters),
    );
    if (!equivalentLitersInput) return;
    const notes = window.prompt("Observaciones (opcional)", record.notes ?? "") ?? "";

    const budget = Number(budgetInput);
    const averageCostPerLiter = Number(averageCostInput);
    const equivalentLiters = Number(equivalentLitersInput);
    if (!Number.isFinite(budget) || budget <= 0) return;
    if (!Number.isFinite(averageCostPerLiter) || averageCostPerLiter <= 0) return;
    if (!Number.isFinite(equivalentLiters) || equivalentLiters <= 0) return;

    const parsedDate = new Date(`${date}T00:00:00`);
    if (Number.isNaN(parsedDate.valueOf())) return;

    const normalizedFuelType: GasolineRecord["fuelType"] = "Magna/Diesel";

    const fullPayload = {
      [gasolineColumnConfig.date]: date,
      [gasolineColumnConfig.month]: toSpanishMonth(date),
      [gasolineColumnConfig.year]: parsedDate.getFullYear(),
      [gasolineColumnConfig.budget]: budget,
      [gasolineColumnConfig.fuelType]: normalizedFuelType,
      [gasolineColumnConfig.averageCostPerLiter]: averageCostPerLiter,
      [gasolineColumnConfig.equivalentLiters]: equivalentLiters,
      [gasolineColumnConfig.notes]: notes.trim() || null,
    };

    const updateRecord = (payload: Record<string, unknown>) =>
      supabase
        .from("fuel_gasoline_records")
        .update(payload)
        .eq("id", recordId)
        .select()
        .single();

    let { data, error } = await updateRecord(fullPayload);
    let configToUse = gasolineColumnConfig;

    const basePayload = {
      [gasolineColumnConfig.date]: date,
      [gasolineColumnConfig.budget]: budget,
      [gasolineColumnConfig.fuelType]: normalizedFuelType,
      [gasolineColumnConfig.averageCostPerLiter]: averageCostPerLiter,
      [gasolineColumnConfig.equivalentLiters]: equivalentLiters,
      [gasolineColumnConfig.notes]: notes.trim() || null,
    };

    if (error) {
      const baseAttempt = await updateRecord(basePayload);
      if (!baseAttempt.error) {
        data = baseAttempt.data;
        error = null;
      } else if (gasolineColumnConfig === gasolineEnglishColumns) {
        const spanishFullPayload = {
          [gasolineSpanishColumns.date]: date,
          [gasolineSpanishColumns.month]: toSpanishMonth(date),
          [gasolineSpanishColumns.year]: parsedDate.getFullYear(),
          [gasolineSpanishColumns.budget]: budget,
          [gasolineSpanishColumns.fuelType]: normalizedFuelType,
          [gasolineSpanishColumns.averageCostPerLiter]: averageCostPerLiter,
          [gasolineSpanishColumns.equivalentLiters]: equivalentLiters,
          [gasolineSpanishColumns.notes]: notes.trim() || null,
        };
        const spanishBasePayload = {
          [gasolineSpanishColumns.date]: date,
          [gasolineSpanishColumns.budget]: budget,
          [gasolineSpanishColumns.fuelType]: normalizedFuelType,
          [gasolineSpanishColumns.averageCostPerLiter]: averageCostPerLiter,
          [gasolineSpanishColumns.equivalentLiters]: equivalentLiters,
          [gasolineSpanishColumns.notes]: notes.trim() || null,
        };

        const spanishFull = await updateRecord(spanishFullPayload);
        if (!spanishFull.error) {
          data = spanishFull.data;
          error = null;
          configToUse = gasolineSpanishColumns;
        } else {
          const spanishBase = await updateRecord(spanishBasePayload);
          data = spanishBase.data;
          error = spanishBase.error;
          if (!spanishBase.error) {
            configToUse = gasolineSpanishColumns;
          }
        }
      }
    }

    if (error || !data) {
      toast.error("No pudimos actualizar el registro.");
      return;
    }

    if (configToUse !== gasolineColumnConfig) {
      setGasolineColumnConfig(configToUse);
    }

    setGasolineRecords((prev) =>
      prev.map((item) =>
        item.id === recordId
          ? mapGasolineRecord(data as Record<string, unknown>, configToUse)
          : item,
      ),
    );
    toast.success("Registro actualizado.");
  };

  const handleDeleteGasolineRecord = async (recordId: string) => {
    if (!canManage) return;
    const confirmed = window.confirm("¿Deseas eliminar este registro?");
    if (!confirmed) return;

    const { error } = await supabase.from("fuel_gasoline_records").delete().eq("id", recordId);
    if (error) {
      toast.error("No pudimos eliminar el registro.");
      return;
    }

    setGasolineRecords((prev) => prev.filter((item) => item.id !== recordId));
    toast.success("Registro eliminado.");
  };

  const handleAddGasRecord = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!canManage) return;

    const tankCapacity = Number(gasFormState.tankCapacity);
    const quantity = Number(gasFormState.quantity);
    const unitCost = Number(gasFormState.unitCost);
    const providedLiters = Number(gasFormState.totalLiters);
    const totalLiters =
      Number.isFinite(providedLiters) && providedLiters > 0
        ? providedLiters
        : tankCapacity * quantity;
    const totalCost =
      gasFormState.costUnit === "Por litro" ? totalLiters * unitCost : quantity * unitCost;

    if (!gasFormState.date.trim() || !gasFormState.gasType.trim() || !gasFormState.provider.trim()) {
      return;
    }
    if (!Number.isFinite(tankCapacity) || tankCapacity <= 0) return;
    if (!Number.isFinite(quantity) || quantity <= 0) return;
    if (!Number.isFinite(unitCost) || unitCost <= 0) return;
    if (!Number.isFinite(totalLiters) || totalLiters <= 0) return;
    if (!Number.isFinite(totalCost) || totalCost <= 0) return;

    const parsedDate = new Date(`${gasFormState.date}T00:00:00`);
    if (Number.isNaN(parsedDate.valueOf())) return;

    if (!user) {
      toast.error("Necesitamos una sesión activa para guardar el registro.");
      return;
    }

    const fullPayload = buildGasPayload(
      gasColumnConfig,
      { ...gasFormState, totalLiters: String(totalLiters) },
      parsedDate,
      {
        includeDerived: true,
        includeCreatedBy: true,
        userId: user.id,
      },
    );
    const basePayload = buildGasPayload(
      gasColumnConfig,
      { ...gasFormState, totalLiters: String(totalLiters) },
      parsedDate,
      {
        includeDerived: false,
        includeCreatedBy: false,
      },
    );

    const insertRecord = (payload: Record<string, unknown>) =>
      supabase.from("fuel_gas_records").insert(payload).select().single();

    let { data, error } = await insertRecord(fullPayload);
    let configToUse = gasColumnConfig;

    if (error) {
      const baseAttempt = await insertRecord(basePayload);
      if (!baseAttempt.error) {
        data = baseAttempt.data;
        error = null;
      } else if (gasColumnConfig === gasEnglishColumns) {
        const spanishFullPayload = buildGasPayload(
          gasSpanishColumns,
          { ...gasFormState, totalLiters: String(totalLiters) },
          parsedDate,
          {
            includeDerived: true,
            includeCreatedBy: true,
            userId: user.id,
          },
        );
        const spanishBasePayload = buildGasPayload(
          gasSpanishColumns,
          { ...gasFormState, totalLiters: String(totalLiters) },
          parsedDate,
          {
            includeDerived: false,
            includeCreatedBy: false,
          },
        );

        const spanishFull = await insertRecord(spanishFullPayload);
        if (!spanishFull.error) {
          data = spanishFull.data;
          error = null;
          configToUse = gasSpanishColumns;
        } else {
          const spanishBase = await insertRecord(spanishBasePayload);
          data = spanishBase.data;
          error = spanishBase.error;
          if (!spanishBase.error) {
            configToUse = gasSpanishColumns;
          }
        }
      }
    }

    if (error || !data) {
      toast.error("No pudimos guardar el registro.");
      return;
    }

    if (configToUse !== gasColumnConfig) {
      setGasColumnConfig(configToUse);
    }

    setGasRecords((prev) => [
      mapGasRecord(data as Record<string, unknown>, configToUse),
      ...prev,
    ]);

    setGasFormState({
      date: "",
      gasType: "",
      tankCapacity: "",
      capacityUnit: "L",
      quantity: "",
      totalLiters: "",
      unitCost: "",
      costUnit: "Por litro",
      provider: "",
    });

    toast.success("Registro de gas guardado.");
  };

  const handleEditGasRecord = async (recordId: string) => {
    if (!canManage) return;
    const record = gasRecords.find((item) => item.id === recordId);
    if (!record) return;

    const date = window.prompt("Fecha de registro (YYYY-MM-DD)", record.date);
    if (!date) return;
    const gasType = window.prompt("Tipo de gas", record.gasType);
    if (!gasType) return;
    const tankCapacityInput = window.prompt(
      `Capacidad del tanque (${record.capacityUnit})`,
      String(record.tankCapacity),
    );
    if (!tankCapacityInput) return;
    const capacityUnit =
      (window.prompt("Unidad de capacidad (L o kg)", record.capacityUnit) ?? "L") === "kg"
        ? "kg"
        : "L";
    const quantityInput = window.prompt("Cantidad", String(record.quantity));
    if (!quantityInput) return;
    const totalLitersInput = window.prompt(
      `Total (${capacityUnit})`,
      String(record.totalLiters),
    );
    if (!totalLitersInput) return;
    const unitCostInput = window.prompt("Costo unitario", String(record.unitCost));
    if (!unitCostInput) return;
    const costUnitInput =
      (window.prompt("Costo por (Por litro o Por unidad)", record.costUnit) ?? "Por litro").toLowerCase() ===
      "por unidad"
        ? "Por unidad"
        : "Por litro";
    const provider = window.prompt("Proveedor", record.provider);
    if (!provider) return;

    const tankCapacity = Number(tankCapacityInput);
    const quantity = Number(quantityInput);
    const totalLiters = Number(totalLitersInput);
    const unitCost = Number(unitCostInput);
    if (!Number.isFinite(tankCapacity) || tankCapacity <= 0) return;
    if (!Number.isFinite(quantity) || quantity <= 0) return;
    if (!Number.isFinite(totalLiters) || totalLiters <= 0) return;
    if (!Number.isFinite(unitCost) || unitCost <= 0) return;

    const parsedDate = new Date(`${date}T00:00:00`);
    if (Number.isNaN(parsedDate.valueOf())) return;

    const totalCost = costUnitInput === "Por litro" ? totalLiters * unitCost : quantity * unitCost;

    const fullPayload = {
      [gasColumnConfig.date]: date,
      [gasColumnConfig.month]: toSpanishMonth(date),
      [gasColumnConfig.year]: parsedDate.getFullYear(),
      [gasColumnConfig.gasType]: gasType.trim(),
      [gasColumnConfig.tankCapacity]: tankCapacity,
      [gasColumnConfig.capacityUnit]: capacityUnit,
      [gasColumnConfig.quantity]: quantity,
      [gasColumnConfig.totalLiters]: totalLiters,
      [gasColumnConfig.unitCost]: unitCost,
      [gasColumnConfig.costUnit]: costUnitInput,
      [gasColumnConfig.totalCost]: totalCost,
      [gasColumnConfig.provider]: provider.trim(),
    };

    const updateRecord = (payload: Record<string, unknown>) =>
      supabase.from("fuel_gas_records").update(payload).eq("id", recordId).select().single();

    let { data, error } = await updateRecord(fullPayload);
    let configToUse = gasColumnConfig;

    const basePayload = {
      [gasColumnConfig.date]: date,
      [gasColumnConfig.gasType]: gasType.trim(),
      [gasColumnConfig.tankCapacity]: tankCapacity,
      [gasColumnConfig.capacityUnit]: capacityUnit,
      [gasColumnConfig.quantity]: quantity,
      [gasColumnConfig.totalLiters]: totalLiters,
      [gasColumnConfig.unitCost]: unitCost,
      [gasColumnConfig.costUnit]: costUnitInput,
      [gasColumnConfig.totalCost]: totalCost,
      [gasColumnConfig.provider]: provider.trim(),
    };

    if (error) {
      const baseAttempt = await updateRecord(basePayload);
      if (!baseAttempt.error) {
        data = baseAttempt.data;
        error = null;
      } else if (gasColumnConfig === gasEnglishColumns) {
        const spanishFullPayload = {
          [gasSpanishColumns.date]: date,
          [gasSpanishColumns.month]: toSpanishMonth(date),
          [gasSpanishColumns.year]: parsedDate.getFullYear(),
          [gasSpanishColumns.gasType]: gasType.trim(),
          [gasSpanishColumns.tankCapacity]: tankCapacity,
          [gasSpanishColumns.capacityUnit]: capacityUnit,
          [gasSpanishColumns.quantity]: quantity,
          [gasSpanishColumns.totalLiters]: totalLiters,
          [gasSpanishColumns.unitCost]: unitCost,
          [gasSpanishColumns.costUnit]: costUnitInput,
          [gasSpanishColumns.totalCost]: totalCost,
          [gasSpanishColumns.provider]: provider.trim(),
        };
        const spanishBasePayload = {
          [gasSpanishColumns.date]: date,
          [gasSpanishColumns.gasType]: gasType.trim(),
          [gasSpanishColumns.tankCapacity]: tankCapacity,
          [gasSpanishColumns.capacityUnit]: capacityUnit,
          [gasSpanishColumns.quantity]: quantity,
          [gasSpanishColumns.totalLiters]: totalLiters,
          [gasSpanishColumns.unitCost]: unitCost,
          [gasSpanishColumns.costUnit]: costUnitInput,
          [gasSpanishColumns.totalCost]: totalCost,
          [gasSpanishColumns.provider]: provider.trim(),
        };

        const spanishFull = await updateRecord(spanishFullPayload);
        if (!spanishFull.error) {
          data = spanishFull.data;
          error = null;
          configToUse = gasSpanishColumns;
        } else {
          const spanishBase = await updateRecord(spanishBasePayload);
          data = spanishBase.data;
          error = spanishBase.error;
          if (!spanishBase.error) {
            configToUse = gasSpanishColumns;
          }
        }
      }
    }

    if (error || !data) {
      toast.error("No pudimos actualizar el registro.");
      return;
    }

    if (configToUse !== gasColumnConfig) {
      setGasColumnConfig(configToUse);
    }

    setGasRecords((prev) =>
      prev.map((item) =>
        item.id === recordId ? mapGasRecord(data as Record<string, unknown>, configToUse) : item,
      ),
    );
    toast.success("Registro actualizado.");
  };

  const handleDeleteGasRecord = async (recordId: string) => {
    if (!canManage) return;
    const confirmed = window.confirm("¿Deseas eliminar este registro?");
    if (!confirmed) return;

    const { error } = await supabase.from("fuel_gas_records").delete().eq("id", recordId);
    if (error) {
      toast.error("No pudimos eliminar el registro.");
      return;
    }

    setGasRecords((prev) => prev.filter((item) => item.id !== recordId));
    toast.success("Registro eliminado.");
  };

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-10 bg-gradient-to-br from-slate-50 via-white to-amber-50">
        <section className="relative overflow-hidden rounded-3xl border border-amber-100 bg-white px-6 py-10 shadow-sm sm:px-10">
          <div className="absolute inset-0 bg-gradient-to-r from-amber-500/10 to-rose-500/10" />
          <div className="relative space-y-4">
            <p className="text-xs font-semibold uppercase tracking-[0.5em] text-amber-600">
              Operación · Combustibles
            </p>
            <h1 className="text-3xl font-bold text-gray-900 sm:text-4xl">
              Gasolina y gases industriales
            </h1>
            <p className="max-w-8xl text-base text-gray-600 xs:text-lg">
              Administra el presupuesto de gasolina y los consumos de gases con filtros por periodo,
              indicadores clave y control de accesos para managers.
            </p>
          </div>
        </section>

        <Tabs defaultValue="gasoline" className="space-y-8">
          <TabsList className="grid w-full max-w-xl grid-cols-2">
            <TabsTrigger value="gasoline">Gasolina</TabsTrigger>
            <TabsTrigger value="gases">Gases</TabsTrigger>
          </TabsList>

          <TabsContent value="gasoline" className="space-y-8">
            <section className="grid gap-6 lg:grid-cols-[2fr_1fr]">
              <Card className="border-amber-100">
                <CardHeader>
                  <CardTitle className="text-xl text-gray-900">Indicadores de gasolina</CardTitle>
                  <p className="text-sm text-gray-500">
                    Presupuesto y consumo por periodo para Magna/Diesel.
                  </p>
                </CardHeader>
                <CardContent className="grid gap-4 sm:grid-cols-2">
                  <StatCard
                    title="Presupuesto total"
                    value={currencyFormatter.format(gasolineStats.totalBudget)}
                    helper="Acumulado del periodo"
                    icon={Fuel}
                    accent="bg-amber-500"
                  />
                  <StatCard
                    title="Litros equivalentes"
                    value={`${numberFormatter.format(gasolineStats.totalLiters)} L`}
                    helper="Consumo estimado"
                    icon={Droplet}
                    accent="bg-rose-500"
                  />
                  <StatCard
                    title="Costo promedio"
                    value={`${currencyFormatter.format(gasolineStats.weightedAverageCost)}/L`}
                    helper="Promedio ponderado"
                    icon={Filter}
                    accent="bg-amber-500"
                  />
                  <StatCard
                    title="Tipos monitoreados"
                    value={`${Object.keys(gasolineStats.byType).length} tipos`}
                    helper="Magna/Diesel"
                    icon={BarChart3}
                    accent="bg-rose-500"
                  />
                </CardContent>
              </Card>

              <Card className="border-rose-100">
                <CardHeader>
                  <CardTitle className="text-lg text-gray-900">Distribución por tipo</CardTitle>
                  <p className="text-sm text-gray-500">
                    Totales del periodo seleccionado.
                  </p>
                </CardHeader>
                <CardContent className="space-y-4">
                  {(["Magna/Diesel"] as const).map((type) => {
                    const totals = gasolineStats.byType[type] ?? { budget: 0, liters: 0 };
                    return (
                      <div
                        key={type}
                        className="rounded-2xl border border-amber-100 bg-amber-50 px-4 py-3"
                      >
                        <p className="text-xs font-semibold uppercase text-amber-700">{type}</p>
                        <p className="mt-2 text-lg font-semibold text-gray-900">
                          {currencyFormatter.format(totals.budget)}
                        </p>
                        <p className="text-xs text-gray-500">
                          {numberFormatter.format(totals.liters)} litros
                        </p>
                      </div>
                    );
                  })}
                </CardContent>
              </Card>
            </section>

            <section className="rounded-3xl border border-amber-100 bg-white p-6 shadow-sm">
              <div className="grid gap-4 sm:grid-cols-3">
                <div className="space-y-2">
                  <Label className="text-xs font-semibold uppercase tracking-[0.2em] text-gray-500">
                    Filtro de año
                  </Label>
                  <select
                    className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm text-slate-900 shadow-sm"
                    value={selectedGasolineYear}
                    onChange={(event) => setSelectedGasolineYear(event.target.value)}
                  >
                    <option value="Todo">Todo</option>
                    {gasolineYears.map((year) => (
                      <option key={year} value={String(year)}>
                        {year}
                      </option>
                    ))}
                  </select>
                </div>
                <div className="space-y-2">
                  <Label className="text-xs font-semibold uppercase tracking-[0.2em] text-gray-500">
                    Filtro de mes
                  </Label>
                  <select
                    className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm text-slate-900 shadow-sm"
                    value={selectedGasolineMonth}
                    onChange={(event) => setSelectedGasolineMonth(event.target.value)}
                  >
                    <option value="Todo">Todo</option>
                    {gasolineMonths.map((month) => (
                      <option key={month} value={month}>
                        {month}
                      </option>
                    ))}
                  </select>
                </div>
                <div className="rounded-2xl border border-amber-100 bg-amber-50 px-4 py-5">
                  <p className="text-xs font-semibold uppercase text-amber-700">
                    Registros filtrados
                  </p>
                  <p className="mt-2 text-2xl font-semibold text-gray-900">
                    {filteredGasolineRecords.length} registros
                  </p>
                  <p className="mt-1 text-xs text-gray-500">
                    Ajusta los filtros para ver otros periodos.
                  </p>
                </div>
              </div>
            </section>

            <section className="space-y-4">
              <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
                <div>
                  <h2 className="text-2xl font-semibold text-gray-900">
                    Registro de gasolina
                  </h2>
                  <p className="text-sm text-gray-500">
                    Captura presupuestos y litros equivalentes por tipo de combustible.
                  </p>
                </div>
              </div>

              <Card className="border-amber-100">
                {canManage && (
                  <>
                    <CardHeader>
                      <CardTitle className="text-lg text-gray-900">Nuevo registro</CardTitle>
                      <p className="text-sm text-gray-500">
                        Los litros equivalentes pueden calcularse automáticamente con presupuesto y
                        costo promedio.
                      </p>
                    </CardHeader>
                    <CardContent>
                      <form
                        className="grid gap-4 rounded-2xl border border-amber-100 bg-amber-50 p-4 md:grid-cols-2"
                        onSubmit={handleAddGasolineRecord}
                      >
                        <div className="space-y-2">
                          <Label htmlFor="gasoline-date">Fecha de registro</Label>
                          <Input
                            id="gasoline-date"
                            type="date"
                            value={gasolineFormState.date}
                            onChange={(event) =>
                              setGasolineFormState((prev) => ({
                                ...prev,
                                date: event.target.value,
                              }))
                            }
                          />
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="gasoline-type">Tipo de combustible</Label>
                          <select
                            id="gasoline-type"
                            className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm text-slate-900 shadow-sm"
                            value={gasolineFormState.fuelType}
                            onChange={(event) =>
                              setGasolineFormState((prev) => ({
                                ...prev,
                                fuelType: event.target.value as GasolineFormState["fuelType"],
                              }))
                            }
                          >
                            <option value="Magna/Diesel">Magna/Diesel</option>
                          </select>
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="gasoline-budget">Presupuesto (MXN)</Label>
                          <Input
                            id="gasoline-budget"
                            type="number"
                            min="0"
                            step="0.01"
                            placeholder="Ej. 45000"
                            value={gasolineFormState.budget}
                            onChange={(event) =>
                              setGasolineFormState((prev) => ({
                                ...prev,
                                budget: event.target.value,
                              }))
                            }
                          />
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="gasoline-cost">Costo promedio por litro (MXN/L)</Label>
                          <Input
                            id="gasoline-cost"
                            type="number"
                            min="0"
                            step="0.01"
                            placeholder="Ej. 24.5"
                            value={gasolineFormState.averageCostPerLiter}
                            onChange={(event) =>
                              setGasolineFormState((prev) => ({
                                ...prev,
                                averageCostPerLiter: event.target.value,
                              }))
                            }
                          />
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="gasoline-liters">Litros equivalentes (L)</Label>
                          <Input
                            id="gasoline-liters"
                            type="number"
                            min="0"
                            step="0.01"
                            placeholder="Ej. 1836"
                            value={gasolineFormState.equivalentLiters}
                            onChange={(event) =>
                              setGasolineFormState((prev) => ({
                                ...prev,
                                equivalentLiters: event.target.value,
                              }))
                            }
                          />
                          <p className="text-xs text-gray-500">
                            Si se deja vacío se calcula automáticamente con presupuesto y costo.
                          </p>
                        </div>
                        <div className="space-y-2 md:col-span-2">
                          <Label htmlFor="gasoline-notes">Observaciones</Label>
                          <Textarea
                            id="gasoline-notes"
                            rows={3}
                            placeholder="Notas adicionales"
                            value={gasolineFormState.notes}
                            onChange={(event) =>
                              setGasolineFormState((prev) => ({
                                ...prev,
                                notes: event.target.value,
                              }))
                            }
                          />
                        </div>
                        <div className="md:col-span-2 flex justify-end">
                          <Button type="submit" className="gap-2">
                            <Plus className="h-4 w-4" />
                            Guardar registro
                          </Button>
                        </div>
                      </form>
                    </CardContent>
                  </>
                )}

                <CardContent>
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>Fecha</TableHead>
                        <TableHead>Tipo</TableHead>
                        <TableHead>Presupuesto</TableHead>
                        <TableHead>Costo promedio/L</TableHead>
                        <TableHead>Litros equivalentes</TableHead>
                        <TableHead>Observaciones</TableHead>
                        <TableHead className="text-right">Acciones</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {filteredGasolineRecords.map((record) => (
                        <TableRow key={record.id}>
                          <TableCell>{formatDate(record.date)}</TableCell>
                          <TableCell>
                            <Badge className="bg-amber-100 text-amber-700">
                              {record.fuelType}
                            </Badge>
                          </TableCell>
                          <TableCell>{currencyFormatter.format(record.budget)}</TableCell>
                          <TableCell>
                            {currencyFormatter.format(record.averageCostPerLiter)}
                          </TableCell>
                          <TableCell>{numberFormatter.format(record.equivalentLiters)} L</TableCell>
                          <TableCell className="max-w-[220px] truncate">
                            {record.notes || "—"}
                          </TableCell>
                          <TableCell className="text-right">
                            {canManage ? (
                              <div className="flex justify-end gap-2">
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="gap-1"
                                  onClick={() => handleEditGasolineRecord(record.id)}
                                >
                                  <Pencil className="h-3 w-3" />
                                  Editar
                                </Button>
                                <Button
                                  size="sm"
                                  variant="destructive"
                                  className="gap-1"
                                  onClick={() => handleDeleteGasolineRecord(record.id)}
                                >
                                  <Trash2 className="h-3 w-3" />
                                  Eliminar
                                </Button>
                              </div>
                            ) : (
                              <span className="text-xs text-gray-400">Solo lectura</span>
                            )}
                          </TableCell>
                        </TableRow>
                      ))}
                      {!filteredGasolineRecords.length && (
                        <TableRow>
                          <TableCell colSpan={7} className="py-6 text-center text-sm text-gray-500">
                            No hay registros para los filtros seleccionados.
                          </TableCell>
                        </TableRow>
                      )}
                    </TableBody>
                  </Table>
                </CardContent>
              </Card>
            </section>
          </TabsContent>

          <TabsContent value="gases" className="space-y-8">
            <section className="grid gap-6 lg:grid-cols-[2fr_1fr]">
              <Card className="border-rose-100">
                <CardHeader>
                  <CardTitle className="text-xl text-gray-900">Indicadores de gases</CardTitle>
                  <p className="text-sm text-gray-500">
                    Control de volumen, gasto y proveedores para cada tipo de gas.
                  </p>
                </CardHeader>
                <CardContent className="grid gap-4 sm:grid-cols-2">
                  <StatCard
                    title="Total registrado"
                    value={`${numberFormatter.format(gasStats.totalLiters)} L/kg`}
                    helper="Volumen equivalente"
                    icon={Flame}
                    accent="bg-rose-500"
                  />
                  <StatCard
                    title="Gasto total"
                    value={currencyFormatter.format(gasStats.totalCost)}
                    helper="Acumulado del periodo"
                    icon={LineChart}
                    accent="bg-amber-500"
                  />
                  <StatCard
                    title="Costo unitario promedio"
                    value={currencyFormatter.format(gasStats.averageUnitCost)}
                    helper="Ponderado por litros o unidades"
                    icon={Filter}
                    accent="bg-rose-500"
                  />
                  <StatCard
                    title="Proveedor principal"
                    value={gasStats.topProvider ?? "Sin proveedor"}
                    helper="Basado en gasto"
                    icon={PackageSearch}
                    accent="bg-amber-500"
                  />
                </CardContent>
              </Card>

              <Card className="border-amber-100">
                <CardHeader>
                  <CardTitle className="text-lg text-gray-900">Totales por tipo</CardTitle>
                  <p className="text-sm text-gray-500">
                    Distribución de gasto por tipo de gas.
                  </p>
                </CardHeader>
                <CardContent className="space-y-4">
                  {Object.keys(gasStats.byType).length ? (
                    Object.entries(gasStats.byType).map(([type, totals]) => (
                      <div
                        key={type}
                        className="rounded-2xl border border-rose-100 bg-rose-50 px-4 py-3"
                      >
                        <p className="text-xs font-semibold uppercase text-rose-700">{type}</p>
                        <p className="mt-2 text-lg font-semibold text-gray-900">
                          {currencyFormatter.format(totals.cost)}
                        </p>
                        <p className="text-xs text-gray-500">
                          {numberFormatter.format(totals.liters)} L/kg
                        </p>
                      </div>
                    ))
                  ) : (
                    <div className="rounded-2xl border border-slate-100 bg-slate-50 px-4 py-4 text-sm text-gray-500">
                      Sin datos para el periodo seleccionado.
                    </div>
                  )}
                </CardContent>
              </Card>
            </section>

            <section className="rounded-3xl border border-rose-100 bg-white p-6 shadow-sm">
              <div className="grid gap-4 sm:grid-cols-3">
                <div className="space-y-2">
                  <Label className="text-xs font-semibold uppercase tracking-[0.2em] text-gray-500">
                    Filtro de año
                  </Label>
                  <select
                    className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm text-slate-900 shadow-sm"
                    value={selectedGasYear}
                    onChange={(event) => setSelectedGasYear(event.target.value)}
                  >
                    <option value="Todo">Todo</option>
                    {gasYears.map((year) => (
                      <option key={year} value={String(year)}>
                        {year}
                      </option>
                    ))}
                  </select>
                </div>
                <div className="space-y-2">
                  <Label className="text-xs font-semibold uppercase tracking-[0.2em] text-gray-500">
                    Filtro de mes
                  </Label>
                  <select
                    className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm text-slate-900 shadow-sm"
                    value={selectedGasMonth}
                    onChange={(event) => setSelectedGasMonth(event.target.value)}
                  >
                    <option value="Todo">Todo</option>
                    {gasMonths.map((month) => (
                      <option key={month} value={month}>
                        {month}
                      </option>
                    ))}
                  </select>
                </div>
                <div className="rounded-2xl border border-rose-100 bg-rose-50 px-4 py-5">
                  <p className="text-xs font-semibold uppercase text-rose-700">
                    Registros filtrados
                  </p>
                  <p className="mt-2 text-2xl font-semibold text-gray-900">
                    {filteredGasRecords.length} registros
                  </p>
                  <p className="mt-1 text-xs text-gray-500">
                    Ajusta los filtros para ver otros periodos.
                  </p>
                </div>
              </div>
            </section>

            <section className="space-y-4">
              <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
                <div>
                  <h2 className="text-2xl font-semibold text-gray-900">Registro de gases</h2>
                  <p className="text-sm text-gray-500">
                    Captura consumo, costos y proveedores por tipo de gas.
                  </p>
                </div>
              </div>

              <Card className="border-rose-100">
                {canManage && (
                  <>
                    <CardHeader>
                      <CardTitle className="text-lg text-gray-900">Nuevo registro</CardTitle>
                      <p className="text-sm text-gray-500">
                        Define si el costo es por litro o por unidad para el cálculo del total.
                      </p>
                    </CardHeader>
                    <CardContent>
                      <form
                        className="grid gap-4 rounded-2xl border border-rose-100 bg-rose-50 p-4 md:grid-cols-2"
                        onSubmit={handleAddGasRecord}
                      >
                        <div className="space-y-2">
                          <Label htmlFor="gas-date">Fecha de registro</Label>
                          <Input
                            id="gas-date"
                            type="date"
                            value={gasFormState.date}
                            onChange={(event) =>
                              setGasFormState((prev) => ({
                                ...prev,
                                date: event.target.value,
                              }))
                            }
                          />
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="gas-type">Tipo de gas</Label>
                          <Input
                            id="gas-type"
                            placeholder="Ej. Argón, Nitrógeno"
                            value={gasFormState.gasType}
                            onChange={(event) =>
                              setGasFormState((prev) => ({
                                ...prev,
                                gasType: event.target.value,
                              }))
                            }
                          />
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="gas-capacity">Capacidad del tanque</Label>
                          <Input
                            id="gas-capacity"
                            type="number"
                            min="0"
                            step="0.01"
                            placeholder="Ej. 50"
                            value={gasFormState.tankCapacity}
                            onChange={(event) =>
                              setGasFormState((prev) => ({
                                ...prev,
                                tankCapacity: event.target.value,
                              }))
                            }
                          />
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="gas-capacity-unit">Unidad de capacidad</Label>
                          <select
                            id="gas-capacity-unit"
                            className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm text-slate-900 shadow-sm"
                            value={gasFormState.capacityUnit}
                            onChange={(event) =>
                              setGasFormState((prev) => ({
                                ...prev,
                                capacityUnit: event.target.value === "kg" ? "kg" : "L",
                              }))
                            }
                          >
                            <option value="L">Litros (L)</option>
                            <option value="kg">Kilogramos (kg)</option>
                          </select>
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="gas-quantity">Cantidad</Label>
                          <Input
                            id="gas-quantity"
                            type="number"
                            min="0"
                            step="0.01"
                            placeholder="Ej. 4"
                            value={gasFormState.quantity}
                            onChange={(event) =>
                              setGasFormState((prev) => ({
                                ...prev,
                                quantity: event.target.value,
                              }))
                            }
                          />
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="gas-total-liters">Total (L/kg)</Label>
                          <Input
                            id="gas-total-liters"
                            type="number"
                            min="0"
                            step="0.01"
                            placeholder="Ej. 200"
                            value={gasFormState.totalLiters}
                            onChange={(event) =>
                              setGasFormState((prev) => ({
                                ...prev,
                                totalLiters: event.target.value,
                              }))
                            }
                          />
                          <p className="text-xs text-gray-500">
                            Si se deja vacío, se calcula con capacidad × cantidad.
                          </p>
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="gas-unit-cost">Costo unitario</Label>
                          <Input
                            id="gas-unit-cost"
                            type="number"
                            min="0"
                            step="0.01"
                            placeholder="Ej. 12.5"
                            value={gasFormState.unitCost}
                            onChange={(event) =>
                              setGasFormState((prev) => ({
                                ...prev,
                                unitCost: event.target.value,
                              }))
                            }
                          />
                        </div>
                        <div className="space-y-2">
                          <Label htmlFor="gas-cost-unit">Unidad de costo</Label>
                          <select
                            id="gas-cost-unit"
                            className="h-10 w-full rounded-md border border-slate-200 bg-white px-3 text-sm text-slate-900 shadow-sm"
                            value={gasFormState.costUnit}
                            onChange={(event) =>
                              setGasFormState((prev) => ({
                                ...prev,
                                costUnit:
                                  event.target.value === "Por unidad" ? "Por unidad" : "Por litro",
                              }))
                            }
                          >
                            <option value="Por litro">Por litro</option>
                            <option value="Por unidad">Por unidad/tanque</option>
                          </select>
                        </div>
                        <div className="space-y-2 md:col-span-2">
                          <Label htmlFor="gas-provider">Proveedor</Label>
                          <Input
                            id="gas-provider"
                            placeholder="Ej. Praxair"
                            value={gasFormState.provider}
                            onChange={(event) =>
                              setGasFormState((prev) => ({
                                ...prev,
                                provider: event.target.value,
                              }))
                            }
                          />
                        </div>
                        <div className="md:col-span-2 flex justify-end">
                          <Button type="submit" className="gap-2">
                            <Plus className="h-4 w-4" />
                            Guardar registro
                          </Button>
                        </div>
                      </form>
                    </CardContent>
                  </>
                )}

                <CardContent>
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>Fecha</TableHead>
                        <TableHead>Tipo de gas</TableHead>
                        <TableHead>Capacidad</TableHead>
                        <TableHead>Cantidad</TableHead>
                        <TableHead>Total</TableHead>
                        <TableHead>Costo unitario</TableHead>
                        <TableHead>Total (MXN)</TableHead>
                        <TableHead>Proveedor</TableHead>
                        <TableHead className="text-right">Acciones</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {filteredGasRecords.map((record) => (
                        <TableRow key={record.id}>
                          <TableCell>{formatDate(record.date)}</TableCell>
                          <TableCell>
                            <Badge className="bg-rose-100 text-rose-700">
                              {record.gasType}
                            </Badge>
                          </TableCell>
                          <TableCell>
                            {numberFormatter.format(record.tankCapacity)} {record.capacityUnit}
                          </TableCell>
                          <TableCell>{numberFormatter.format(record.quantity)}</TableCell>
                          <TableCell>
                            {numberFormatter.format(record.totalLiters)} {record.capacityUnit}
                          </TableCell>
                          <TableCell>
                            {currencyFormatter.format(record.unitCost)}{" "}
                            {record.costUnit === "Por litro" ? "/L" : "/unidad"}
                          </TableCell>
                          <TableCell>{currencyFormatter.format(record.totalCost)}</TableCell>
                          <TableCell className="max-w-[180px] truncate">
                            {record.provider}
                          </TableCell>
                          <TableCell className="text-right">
                            {canManage ? (
                              <div className="flex justify-end gap-2">
                                <Button
                                  size="sm"
                                  variant="outline"
                                  className="gap-1"
                                  onClick={() => handleEditGasRecord(record.id)}
                                >
                                  <Pencil className="h-3 w-3" />
                                  Editar
                                </Button>
                                <Button
                                  size="sm"
                                  variant="destructive"
                                  className="gap-1"
                                  onClick={() => handleDeleteGasRecord(record.id)}
                                >
                                  <Trash2 className="h-3 w-3" />
                                  Eliminar
                                </Button>
                              </div>
                            ) : (
                              <span className="text-xs text-gray-400">Solo lectura</span>
                            )}
                          </TableCell>
                        </TableRow>
                      ))}
                      {!filteredGasRecords.length && (
                        <TableRow>
                          <TableCell colSpan={9} className="py-6 text-center text-sm text-gray-500">
                            No hay registros para los filtros seleccionados.
                          </TableCell>
                        </TableRow>
                      )}
                    </TableBody>
                  </Table>
                </CardContent>
              </Card>
            </section>
          </TabsContent>
        </Tabs>
      </div>
    </SGIDashboardLayout>
  );
}