"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import Link from "next/link";
import {
  AlertTriangle,
  ArrowRight,
  CalendarClock,
  CheckCircle2,
  ClipboardList,
  FileCheck2,
  LineChart,
  ShieldCheck,
  Target,
} from "lucide-react";
import type { User } from "@supabase/supabase-js";

import { supabase } from "@/lib/supabase";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { ChartContainer, ChartTooltip, ChartTooltipContent } from "@/components/ui/chart";
import {
  CartesianGrid,
  Line,
  LineChart as RechartsLineChart,
  Pie,
  PieChart,
  XAxis,
  YAxis,
} from "recharts";

const allowedManagers = new Set([
  "ehs.mexico@meiwa.com.mx",
  "auxiliar.ehs@meiwa.com.mx",
]);

type DashboardKpis = {
  actionsOpen: number;
  actionsClosed: number;  
  ncOpen: number;
  ncClosed: number;  
  improvementsClosed: number;
  improvementsTotal: number;
  objectivesMet: number;
  objectivesTotal: number;
  objectivesCompletionRate: number | null;
  lastAuditDate: string | null;
  lastManagementReviewDate: string | null;
};

type AnnouncementItem = {
  id: string;
  title: string;
  category: string | null;
  publish_date: string | null;
};

type AwarenessModuleItem = {
  id: string;
  title: string | null;
  category: string | null;
  created_at: string | null;
  is_active?: boolean | null;
};

type IndicatorMonthlyRecord = {
  year: number | null;
  jan: number | null;
  feb: number | null;
  mar: number | null;
  apr: number | null;
  may: number | null;
  jun: number | null;
  jul: number | null;
  aug: number | null;
  sep: number | null;
  oct: number | null;
  nov: number | null;
  dec: number | null;
};

type MonthKey =
  | "jan"
  | "feb"
  | "mar"
  | "apr"
  | "may"
  | "jun"
  | "jul"
  | "aug"
  | "sep"
  | "oct"
  | "nov"
  | "dec";

const monthOptions: { key: MonthKey; label: string; full: string }[] = [
  { key: "jan", label: "ENE", full: "Enero" },
  { key: "feb", label: "FEB", full: "Febrero" },
  { key: "mar", label: "MAR", full: "Marzo" },
  { key: "apr", label: "ABR", full: "Abril" },
  { key: "may", label: "MAY", full: "Mayo" },
  { key: "jun", label: "JUN", full: "Junio" },
  { key: "jul", label: "JUL", full: "Julio" },
  { key: "aug", label: "AGO", full: "Agosto" },
  { key: "sep", label: "SEP", full: "Septiembre" },
  { key: "oct", label: "OCT", full: "Octubre" },
  { key: "nov", label: "NOV", full: "Noviembre" },
  { key: "dec", label: "DIC", full: "Diciembre" },
];

const formatDate = (value?: string | null) => {
  if (!value) return "Sin fecha";
  const parsed = new Date(value);
  if (Number.isNaN(parsed.getTime())) return "Sin fecha";
  return parsed.toLocaleDateString("es-MX", {
    day: "2-digit",
    month: "short",
    year: "numeric",
  });
};

const formatCount = (value?: number | null) => (value == null ? "—" : value.toLocaleString("es-MX"));

const emptyKpis: DashboardKpis = {
  actionsOpen: 0,
  actionsClosed: 0,  
  ncOpen: 0,
  ncClosed: 0,  
  improvementsClosed: 0,
  improvementsTotal: 0,
  objectivesMet: 0,
  objectivesTotal: 0,
  objectivesCompletionRate: null,
  lastAuditDate: null,
  lastManagementReviewDate: null,
};

export default function SGIDashboard() {
  const router = useRouter();
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [canManage, setCanManage] = useState(false);
  const [kpis, setKpis] = useState<DashboardKpis>(emptyKpis);
  const [recentAnnouncements, setRecentAnnouncements] = useState<AnnouncementItem[]>([]);
  const [recentAwarenessModules, setRecentAwarenessModules] = useState<AwarenessModuleItem[]>([]);
  const [environmentTrendData, setEnvironmentTrendData] = useState<
    { month: string; short: string; value: number | null }[]
  >([]);
  const [environmentTrendYear, setEnvironmentTrendYear] = useState<number | null>(null);
  const [environmentTrendLoading, setEnvironmentTrendLoading] = useState(false);

  const loadDashboardData = useCallback(async (currentUser: User) => {
    const isManager = allowedManagers.has(currentUser.email ?? "");
    setCanManage(isManager);
    setEnvironmentTrendLoading(true);

    const userEmail = currentUser.email ?? "";
    const actionQuery = supabase
      .from("action_plan_matrix")
      .select("id", { count: "exact", head: true })
      .not("status", "in", "(COMPLETO,CERRADO,CERRADA)");

    const actionClosedQuery = supabase
      .from("action_plan_matrix")
      .select("id", { count: "exact", head: true })
      .in("status", ["COMPLETO", "CERRADO", "CERRADA"]);

    if (!isManager) {
      actionQuery.eq("owner", userEmail);
      actionClosedQuery.eq("owner", userEmail);      
    }

    const ncQuery = supabase
      .from("nc_corrective_actions")
      .select("id", { count: "exact", head: true })
      .not("status", "in", "(COMPLETO,CERRADO,CERRADA)");

    const ncClosedQuery = supabase
      .from("nc_corrective_actions")
      .select("id", { count: "exact", head: true })
      .in("status", ["COMPLETO", "CERRADO", "CERRADA"]);

    if (!isManager) {
      ncQuery.eq("implementation_owner", userEmail);
      ncClosedQuery.eq("implementation_owner", userEmail);      
    }

    const improvementClosedQuery = supabase
      .from("continuous_improvement_register")
      .select("id", { count: "exact", head: true })
      .in("status", ["verified", "closed"]);

    const improvementTotalQuery = supabase
      .from("continuous_improvement_register")
      .select("id", { count: "exact", head: true });

    if (!isManager) {
      improvementClosedQuery.eq("created_by", currentUser.id);
      improvementTotalQuery.eq("created_by", currentUser.id);
    }

    const objectivesCompletedQuery = supabase
      .from("objectives")
      .select("id", { count: "exact", head: true })
      .eq("status", "Cumplido");

    const objectivesTotalQuery = supabase.from("objectives").select("id", { count: "exact", head: true });

    if (!isManager) {
      objectivesCompletedQuery.eq("responsible", userEmail);
      objectivesTotalQuery.eq("responsible", userEmail);
    }

    const auditQuery = supabase
      .from("internal_audit_sessions")
      .select("audit_date, end_date")
      .order("audit_date", { ascending: false })
      .order("end_date", { ascending: false })
      .limit(1);

    let announcementQuery = supabase
      .from("communications_posts")
      .select("id, title, category, publish_date")
      .order("publish_date", { ascending: false })
      .limit(5);

    if (!isManager) {
      announcementQuery = announcementQuery.eq("status", "published");
    }

    let awarenessQuery = supabase
      .from("awareness_modules")
      .select("id, title, category, created_at, is_active")
      .order("created_at", { ascending: false })
      .limit(5);

    if (!isManager) {
      awarenessQuery = awarenessQuery.eq("is_active", true);
    }

    const [
      actionsResponse,
      actionsClosedResponse,      
      ncResponse,
      ncClosedResponse,      
      improvementsClosedResponse,
      improvementsTotalResponse,
      objectivesCompletedResponse,
      objectivesTotalResponse,
      auditResponse,
      announcementsResponse,
      awarenessResponse,
      environmentIndicatorsResponse,      
    ] = await Promise.all([
      actionQuery,
      actionClosedQuery,      
      ncQuery,
      ncClosedQuery,      
      improvementClosedQuery,
      improvementTotalQuery,
      objectivesCompletedQuery,
      objectivesTotalQuery,
      auditQuery,
      announcementQuery,
      awarenessQuery,
      supabase
        .from("indicators_yearly")
        .select(
          "year, indicator_type, jan, feb, mar, apr, may, jun, jul, aug, sep, oct, nov, dec",
        )
        .eq("indicator_type", "environment")
        .order("year", { ascending: false }),      
    ]);

    if (actionsResponse.error) {
      console.error(actionsResponse.error);
    }

    if (actionsClosedResponse.error) {
      console.error(actionsClosedResponse.error);
    }

    if (ncResponse.error) {
      console.error(ncResponse.error);
    }

    if (ncClosedResponse.error) {
      console.error(ncClosedResponse.error);
    }

    if (improvementsClosedResponse.error) {
      console.error(improvementsClosedResponse.error);
    }

    if (improvementsTotalResponse.error) {
      console.error(improvementsTotalResponse.error);
    }

    if (objectivesCompletedResponse.error) {
      console.error(objectivesCompletedResponse.error);
    }

    if (objectivesTotalResponse.error) {
      console.error(objectivesTotalResponse.error);
    }

    if (auditResponse.error) {
      console.error(auditResponse.error);
    }

    if (announcementsResponse.error) {
      console.error(announcementsResponse.error);
    }

    if (awarenessResponse.error) {
      console.error(awarenessResponse.error);
    }

    if (environmentIndicatorsResponse.error) {
      console.error(environmentIndicatorsResponse.error);
    }

    const indicatorData = (environmentIndicatorsResponse.data ?? []) as IndicatorMonthlyRecord[];
    const latestYear = indicatorData.find((record) => record.year != null)?.year ?? null;
    const latestYearRecords = latestYear
      ? indicatorData.filter((record) => record.year === latestYear)
      : [];
    const trendData = monthOptions.map((month) => {
      const monthValues = latestYearRecords
        .map((record) => record[month.key])
        .filter((value): value is number => value !== null && value !== undefined);
      const average = monthValues.length
        ? Number(
            (
              monthValues.reduce((sum, value) => sum + value, 0) / monthValues.length
            ).toFixed(2),
          )
        : null;
      return {
        month: month.full,
        short: month.label,
        value: average,
      };
    });

    setEnvironmentTrendYear(latestYear);
    setEnvironmentTrendData(trendData);
    setEnvironmentTrendLoading(false);

    const totalObjectives = objectivesTotalResponse.count ?? 0;
    const completedObjectives = objectivesCompletedResponse.count ?? 0;
    const completionRate = totalObjectives
      ? Math.round((completedObjectives / totalObjectives) * 100)
      : null;

    const auditRecord = auditResponse.data?.[0];
    const lastAuditDate = auditRecord?.audit_date ?? auditRecord?.end_date ?? null;

    let lastManagementReviewDate: string | null = null;
    const closedReviewResponse = await supabase
      .from("management_review_sessions")
      .select("review_date")
      .eq("status", "closed")
      .order("review_date", { ascending: false })
      .limit(1);

    if (closedReviewResponse.error) {
      console.error(closedReviewResponse.error);
    }

    if (closedReviewResponse.data?.length) {
      lastManagementReviewDate = closedReviewResponse.data[0].review_date ?? null;
    } else {
      const fallbackReviewResponse = await supabase
        .from("management_review_sessions")
        .select("review_date")
        .order("review_date", { ascending: false })
        .limit(1);

      if (fallbackReviewResponse.error) {
        console.error(fallbackReviewResponse.error);
      }

      lastManagementReviewDate = fallbackReviewResponse.data?.[0]?.review_date ?? null;
    }

    setKpis({
      actionsOpen: actionsResponse.count ?? 0,
      actionsClosed: actionsClosedResponse.count ?? 0,      
      ncOpen: ncResponse.count ?? 0,
      ncClosed: ncClosedResponse.count ?? 0,      
      improvementsClosed: improvementsClosedResponse.count ?? 0,
      improvementsTotal: improvementsTotalResponse.count ?? 0,
      objectivesMet: completedObjectives,
      objectivesTotal: totalObjectives,
      objectivesCompletionRate: completionRate,
      lastAuditDate,
      lastManagementReviewDate,
    });

    setRecentAnnouncements((announcementsResponse.data ?? []) as AnnouncementItem[]);
    setRecentAwarenessModules((awarenessResponse.data ?? []) as AwarenessModuleItem[]);
  }, []);

  useEffect(() => {
    const loadSession = async () => {
      const { data: { session } } = await supabase.auth.getSession();

      if (!session?.user) {
        router.push("/meiwa-core/meiwa-sgi");
        setLoading(false);
        return;
      }

      setUser(session.user);
      await loadDashboardData(session.user);
      setLoading(false);
    };

    loadSession();

    const { data: { subscription } } = supabase.auth.onAuthStateChange((_event, session) => {
      if (!session?.user) {
        router.push("/meiwa-core/meiwa-sgi");
        setLoading(false);
        return;
      }

      setUser(session.user);
      loadDashboardData(session.user).finally(() => {
        setLoading(false);
      });
    });

    return () => subscription.unsubscribe();
  }, [loadDashboardData, router]);

  const kpiCards = useMemo(
    () => [
      {
        id: "actions",
        title: canManage ? "Acciones abiertas" : "Mis acciones abiertas",
        value: formatCount(kpis.actionsOpen),
        helper: "Plan de acciones",
        href: "/meiwa-core/meiwa-sgi/planning/action-plan",
        icon: ClipboardList,
      },
      {
        id: "nc",
        title: canManage ? "No conformidades abiertas" : "Mis no conformidades abiertas",
        value: formatCount(kpis.ncOpen),
        helper: "Acciones correctivas",
        href: "/meiwa-core/meiwa-sgi/improvement/nonconformity-corrective-action",
        icon: AlertTriangle,
      },
      {
        id: "improvements",
        title: canManage ? "Mejoras realizadas" : "Mis mejoras realizadas",
        value: formatCount(kpis.improvementsClosed),
        helper: `Total: ${formatCount(kpis.improvementsTotal)}`,
        href: "/meiwa-core/meiwa-sgi/improvement/continuous-improvement",
        icon: CheckCircle2,
      },
      {
        id: "objectives",
        title: canManage ? "Objetivos cumplidos" : "Mis objetivos cumplidos",
        value: formatCount(kpis.objectivesMet),
        helper: `${formatCount(kpis.objectivesTotal)} total${
          kpis.objectivesCompletionRate != null ? ` • ${kpis.objectivesCompletionRate}%` : ""
        }`,
        href: "/meiwa-core/meiwa-sgi/planning/environmental-objectives",
        icon: Target,
      },
      {
        id: "audit",
        title: "Última auditoría interna",
        value: formatDate(kpis.lastAuditDate),
        helper: "Historial de auditorías",
        href: "/meiwa-core/meiwa-sgi/evaluation/internal-audit",
        icon: CalendarClock,
      },
      {
        id: "review",
        title: "Última revisión por la dirección",
        value: formatDate(kpis.lastManagementReviewDate),
        helper: "Revisión gerencial",
        href: "/meiwa-core/meiwa-sgi/evaluation/management-review",
        icon: ShieldCheck,
      },
    ],
    [canManage, kpis]
  );

  const environmentTrendHasData = useMemo(
    () => environmentTrendData.some((item) => item.value !== null && item.value !== undefined),
    [environmentTrendData],
  );

  const environmentChartConfig = useMemo(
    () => ({
      value: {
        label: "Promedio mensual",
        color: "hsl(160 84% 39%)",
      },
    }),
    [],
  );

  const statusChartConfig = useMemo(
    () => ({
      open: {
        label: "Abiertas",
        color: "hsl(24 94% 50%)",  
      },
      closed: {
        label: "Cerradas",
        color: "hsl(142 71% 45%)",
      },
      met: {
        label: "Cumplidos",
        color: "hsl(142 71% 45%)",
      },
      remaining: {
        label: "Pendientes",
        color: "hsl(45 93% 47%)",
      },
    }),
    [],
  );

  const actionsStatusData = useMemo(
    () => [
      { name: "Abiertas", value: kpis.actionsOpen, fill: "var(--color-open)" },
      { name: "Cerradas", value: kpis.actionsClosed, fill: "var(--color-closed)" },
    ],
    [kpis.actionsClosed, kpis.actionsOpen],
  );

  const ncStatusData = useMemo(
    () => [
      { name: "Abiertas", value: kpis.ncOpen, fill: "var(--color-open)" },
      { name: "Cerradas", value: kpis.ncClosed, fill: "var(--color-closed)" },
    ],
    [kpis.ncClosed, kpis.ncOpen],
  );

  const objectivesRemaining = useMemo(
    () => Math.max(kpis.objectivesTotal - kpis.objectivesMet, 0),
    [kpis.objectivesMet, kpis.objectivesTotal],
  );

  const objectivesStatusData = useMemo(
    () => [
      { name: "Cumplidos", value: kpis.objectivesMet, fill: "var(--color-met)" },
      { name: "Pendientes", value: objectivesRemaining, fill: "var(--color-remaining)" },
    ],
    [kpis.objectivesMet, objectivesRemaining],
  );

  const quickAccessLinks = [
    {
      id: "certificate",
      title: "Ir a Certificado",
      description: "Consulta el certificado vigente del SGI.",
      href: "/meiwa-core/meiwa-sgi/certificate",
      icon: FileCheck2,
    },
    {
      id: "policy",
      title: "Ir a Política",
      description: "Revisa la política ambiental y de calidad.",
      href: "/meiwa-core/meiwa-sgi/policies",
      icon: ShieldCheck,
    },
    {
      id: "action-plan",
      title: "Plan de acciones",
      description: "Da seguimiento a compromisos y acciones abiertas.",
      href: "/meiwa-core/meiwa-sgi/planning/action-plan",
      icon: ClipboardList,
    },
    {
      id: "internal-audit",
      title: "Auditoría interna",
      description: "Consulta sesiones, hallazgos y resultados.",
      href: "/meiwa-core/meiwa-sgi/evaluation/internal-audit",
      icon: CalendarClock,
    },
    {
      id: "monitoring",
      title: "Monitoring & Measurement",
      description: "Indicadores y mediciones clave del SGI.",
      href: "/meiwa-core/meiwa-sgi/evaluation/monitoring-measurement",
      icon: LineChart,
    },
    {
      id: "nc",
      title: "Nonconformity & Corrective Action",
      description: "Hallazgos, acciones correctivas y seguimiento.",
      href: "/meiwa-core/meiwa-sgi/improvement/nonconformity-corrective-action",
      icon: AlertTriangle,
    },
  ];

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (!user) {
    return null;
  }

  return (
    <SGIDashboardLayout>
      <div className="space-y-6">
        <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Dashboard</h1>
            <p className="text-gray-600">
              Bienvenido de vuelta, {user.user_metadata?.full_name || user.email}
            </p>
          </div>
        </div>
        <div className="space-y-10">
          <section className="rounded-3xl bg-gradient-to-br from-green-500 via-green-400 to-emerald-300 px-8 py-10 text-white shadow-lg">
            <div className="max-w-2xl space-y-4">
              <p className="text-sm font-semibold uppercase tracking-[0.5em] text-white/90">
                Sistema de Gestión Integral
              </p>
            </div>
          </section>
        </div>

        <section className="space-y-4">
          <div className="flex items-center justify-between">
            <h2 className="text-xl font-semibold text-gray-900">Distribución de indicadores</h2>
          </div>
          <div className="grid gap-4 lg:grid-cols-3">
            <Card className="border border-gray-200">
              <CardHeader>
                <CardTitle className="text-base">Acciones abiertas vs cerradas</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                {kpis.actionsOpen + kpis.actionsClosed === 0 ? (
                  <div className="flex h-[220px] items-center justify-center rounded-xl border border-dashed border-gray-200 text-sm text-gray-500">
                    Sin acciones registradas.
                  </div>
                ) : (
                  <ChartContainer config={statusChartConfig} className="h-[220px] w-full">
                    <PieChart>
                      <Pie
                        data={actionsStatusData}
                        dataKey="value"
                        nameKey="name"
                        innerRadius={55}
                        outerRadius={85}
                        paddingAngle={4}
                      />
                      <ChartTooltip content={<ChartTooltipContent nameKey="name" />} />
                    </PieChart>
                  </ChartContainer>
                )}
                <div className="flex flex-wrap justify-center gap-4 text-xs text-gray-500">
                  {actionsStatusData.map((item) => (
                    <div key={item.name} className="flex items-center gap-2">
                      <span className="h-2.5 w-2.5 rounded-full" style={{ backgroundColor: item.fill }} />
                      <span>
                        {item.name}: {formatCount(item.value)}
                      </span>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>

            <Card className="border border-gray-200">
              <CardHeader>
                <CardTitle className="text-base">No conformidades abiertas vs cerradas</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                {kpis.ncOpen + kpis.ncClosed === 0 ? (
                  <div className="flex h-[220px] items-center justify-center rounded-xl border border-dashed border-gray-200 text-sm text-gray-500">
                    Sin no conformidades registradas.
                  </div>
                ) : (
                  <ChartContainer config={statusChartConfig} className="h-[220px] w-full">
                    <PieChart>
                      <Pie
                        data={ncStatusData}
                        dataKey="value"
                        nameKey="name"
                        innerRadius={55}
                        outerRadius={85}
                        paddingAngle={4}
                      />
                      <ChartTooltip content={<ChartTooltipContent nameKey="name" />} />
                    </PieChart>
                  </ChartContainer>
                )}
                <div className="flex flex-wrap justify-center gap-4 text-xs text-gray-500">
                  {ncStatusData.map((item) => (
                    <div key={item.name} className="flex items-center gap-2">
                      <span className="h-2.5 w-2.5 rounded-full" style={{ backgroundColor: item.fill }} />
                      <span>
                        {item.name}: {formatCount(item.value)}
                      </span>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>

            <Card className="border border-gray-200">
              <CardHeader>
                <CardTitle className="text-base">Objetivos cumplidos vs totales</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                {kpis.objectivesTotal === 0 ? (
                  <div className="flex h-[220px] items-center justify-center rounded-xl border border-dashed border-gray-200 text-sm text-gray-500">
                    Sin objetivos registrados.
                  </div>
                ) : (
                  <ChartContainer config={statusChartConfig} className="h-[220px] w-full">
                    <PieChart>
                      <Pie
                        data={objectivesStatusData}
                        dataKey="value"
                        nameKey="name"
                        innerRadius={55}
                        outerRadius={85}
                        paddingAngle={4}
                      />
                      <ChartTooltip content={<ChartTooltipContent nameKey="name" />} />
                    </PieChart>
                  </ChartContainer>
                )}
                <div className="flex flex-wrap justify-center gap-4 text-xs text-gray-500">
                  {objectivesStatusData.map((item) => (
                    <div key={item.name} className="flex items-center gap-2">
                      <span className="h-2.5 w-2.5 rounded-full" style={{ backgroundColor: item.fill }} />
                      <span>
                        {item.name}: {formatCount(item.value)}
                      </span>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>
        </section>
        
        <section className="space-y-4">
          <div className="flex items-center justify-between">
            <h2 className="text-xl font-semibold text-gray-900">Indicadores clave</h2>
          </div>
          <div className="grid gap-4 sm:grid-cols-2 xl:grid-cols-3">
            {kpiCards.map((card) => (
              <Link key={card.id} href={card.href} className="group">
                <Card className="h-full border border-gray-200 transition-all duration-200 hover:border-emerald-300 hover:shadow-md">
                  <CardHeader className="flex flex-row items-center justify-between gap-4">
                    <div>
                      <CardTitle className="text-sm font-semibold text-gray-600">{card.title}</CardTitle>
                      <div className="mt-2 text-2xl font-bold text-gray-900">{card.value}</div>
                    </div>
                    <div className="rounded-full bg-emerald-50 p-3 text-emerald-600">
                      <card.icon className="h-5 w-5" />
                    </div>
                  </CardHeader>
                  <CardContent className="flex items-center justify-between text-sm text-gray-500">
                    <span>{card.helper}</span>
                    <ArrowRight className="h-4 w-4 opacity-0 transition-opacity duration-200 group-hover:opacity-100" />
                  </CardContent>
                </Card>
              </Link>
            ))}
          </div>
        </section>
        
        <section className="space-y-4">
          <div className="flex flex-col gap-2 sm:flex-row sm:items-center sm:justify-between">
            <div>
              <h2 className="text-xl font-semibold text-gray-900">
                Evolución mensual de indicadores ambientales
              </h2>
              <p className="text-sm text-gray-500">
                Promedio mensual del último año con datos disponibles.
              </p>
            </div>
            <Card className="border border-emerald-100 bg-emerald-50 px-3 py-2 text-sm text-emerald-700">
              {environmentTrendYear ? `Año ${environmentTrendYear}` : "Sin año disponible"}
            </Card>
          </div>
          <Card className="border border-emerald-100">
            <CardContent className="p-6">
              {environmentTrendLoading ? (
                <div className="flex items-center justify-center rounded-xl border border-dashed border-emerald-200 bg-emerald-50 p-6 text-sm text-emerald-700">
                  Cargando evolución mensual...
                </div>
              ) : environmentTrendHasData ? (
                <ChartContainer config={environmentChartConfig} className="h-[280px] w-full">
                  <RechartsLineChart data={environmentTrendData} margin={{ left: 8, right: 16, top: 12 }}>
                    <CartesianGrid strokeDasharray="4 4" />
                    <XAxis dataKey="short" />
                    <YAxis
                      tickLine={false}
                      axisLine={false}
                      tickFormatter={(value) => (value === null ? "" : `${value}`)}
                    />
                    <ChartTooltip
                      content={
                        <ChartTooltipContent
                          indicator="line"
                          labelFormatter={(_, payload) =>
                            payload?.[0]?.payload?.month ?? "Mes"
                          }
                        />
                      }
                    />
                    <Line
                      type="monotone"
                      dataKey="value"
                      stroke="var(--color-value)"
                      strokeWidth={2}
                      dot={{ r: 3, strokeWidth: 2 }}
                      activeDot={{ r: 5 }}
                      connectNulls
                    />
                  </RechartsLineChart>
                </ChartContainer>
              ) : (
                <div className="rounded-xl border border-dashed border-emerald-200 bg-emerald-50 p-6 text-center text-sm text-emerald-700">
                  No hay datos suficientes para graficar la evolución mensual.
                </div>
              )}
            </CardContent>
          </Card>
        </section>

        <section className="grid gap-6 lg:grid-cols-2">
          <Card className="border border-gray-200">
            <CardHeader className="flex flex-row items-center justify-between">
              <div>
                <CardTitle className="text-lg">Actividad reciente</CardTitle>
                <p className="text-sm text-gray-500">Últimos comunicados publicados.</p>
              </div>
              <Button asChild variant="outline" size="sm">
                <Link href="/meiwa-core/meiwa-sgi/support/comunication">Ver todos</Link>
              </Button>
            </CardHeader>
            <CardContent>
              <ul className="space-y-4">
                {recentAnnouncements.length ? (
                  recentAnnouncements.map((item) => (
                    <li
                      key={item.id}
                      className="flex flex-col gap-3 sm:flex-row sm:items-start sm:justify-between"
                    >
                      <div>
                        <p className="font-medium text-gray-900">{item.title}</p>
                        <p className="text-sm text-gray-500">
                          {item.category ?? "Sin categoría"} • {formatDate(item.publish_date)}
                        </p>
                      </div>
                      <div className="flex items-center gap-3 text-xs text-gray-400">
                        <span>Comunicado</span>
                        <Button asChild variant="ghost" size="sm" className="h-7 px-2 text-xs">
                          <Link href={`/meiwa-core/meiwa-sgi/support/comunication?postId=${item.id}`}>Ver comunicado</Link>
                        </Button>
                      </div>
                    </li>
                  ))
                ) : (
                  <li className="text-sm text-gray-500">Aún no hay comunicados recientes.</li>
                )}
              </ul>
            </CardContent>
          </Card>

          <Card className="border border-gray-200">
            <CardHeader className="flex flex-row items-center justify-between">
              <div>
                <CardTitle className="text-lg">Awareness</CardTitle>
                <p className="text-sm text-gray-500">Módulos más recientes de toma de conciencia.</p>
              </div>
              <Button asChild variant="outline" size="sm">
                <Link href="/meiwa-core/meiwa-sgi/support/awareness">Ir a awareness</Link>
              </Button>
            </CardHeader>
            <CardContent>
              <ul className="space-y-4">
                {recentAwarenessModules.length ? (
                  recentAwarenessModules.map((module) => (
                    <li
                      key={module.id}
                      className="flex flex-col gap-3 sm:flex-row sm:items-start sm:justify-between"
                    >
                      <div>
                        <p className="font-medium text-gray-900">{module.title ?? "Sin título"}</p>
                        <p className="text-sm text-gray-500">
                          {module.category ?? "Sin categoría"} • {formatDate(module.created_at)}
                        </p>
                      </div>
                      <div className="flex items-center gap-3 text-xs text-gray-400">
                        <span>{module.is_active === false ? "Inactivo" : "Activo"}</span>
                        <Button asChild variant="ghost" size="sm" className="h-7 px-2 text-xs">
                          <Link href={`/meiwa-core/meiwa-sgi/support/awareness/modules/view?moduleId=${module.id}`}>
                            Ver módulo
                          </Link>
                        </Button>
                      </div>
                    </li>
                  ))
                ) : (
                  <li className="text-sm text-gray-500">Aún no hay módulos recientes.</li>
                )}
              </ul>
            </CardContent>
          </Card>
        </section>

        <section className="space-y-4">
          <h2 className="text-xl font-semibold text-gray-900">Accesos rápidos</h2>
          <div className="grid gap-4 md:grid-cols-2 xl:grid-cols-3">
            {quickAccessLinks.map((link) => (
              <Card key={link.id} className="border border-gray-200">
                <CardContent className="flex h-full flex-col gap-3 p-6">
                  <div className="flex items-center gap-3 text-emerald-600">
                    <link.icon className="h-5 w-5" />
                    <p className="font-semibold text-gray-900">{link.title}</p>
                  </div>
                  <p className="text-sm text-gray-500">{link.description}</p>
                  <Button asChild variant="outline" size="sm" className="mt-auto w-fit">
                    <Link href={link.href}>Abrir módulo</Link>
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>
        </section>

      </div>
    </SGIDashboardLayout>
  );
}