"use client";

import { useCallback, useEffect, useMemo, useRef, useState } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import type { User } from "@supabase/supabase-js";
import {
  CalendarDays,
  ClipboardList,
  FileSearch,
  Filter,
  ListChecks,
  Plus,
  RefreshCcw,
  ShieldCheck,
  Trash2,  
} from "lucide-react";
import { toast } from "sonner";

import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Dialog,
  DialogContent,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Textarea } from "@/components/ui/textarea";
import { supabase } from "@/lib/supabase";

const allowedManagers = new Set([
  "ehs.mexico@meiwa.com.mx",
  "auxiliar.ehs@meiwa.com.mx",
]);

const classificationOptions = ["Ambiental", "Calidad", "Seguridad"];

const statusOptions = [
  "COMPLETO",
  "RETRASADO",
  "EN PROCESO",
  "PLANEADO",
  "SIN ASIGNAR",
  "REABIERTO",
] as const;

type StatusOption = (typeof statusOptions)[number];

const resultOptions = ["Conformidad", "No conformidad"] as const;

type ResultOption = (typeof resultOptions)[number];

type ActionPlanRecord = {
  id: string;
  action_id: string | null;
  created_at: string | null;
  source: string | null;
  classification: string | null;
  description: string | null;
  comments: string | null;
  root_cause_analysis: string | null;
  action_plan: string | null;
  owner: string | null;
  target_date: string | null;
  start_date: string | null;
  end_date: string | null;
  evidence_url: string | null;
  result: ResultOption | null;
  result_comment: string | null;
  review_date: string | null;
  status: StatusOption | null;
  updated_at: string | null;
  updated_by: string | null;
};

type FormState = {
  action_id: string;
  source: string;
  classification: string;
  description: string;
  comments: string;
  root_cause_analysis: string;
  action_plan: string;
  owner: string;
  target_date: string;
  start_date: string;
  end_date: string;
  evidence_url: string;
  result: ResultOption | "";
  result_comment: string;
  review_date: string;
};

const buildEmptyFormState = (): FormState => ({
  action_id: "",
  source: "",
  classification: "",
  description: "",
  comments: "",
  root_cause_analysis: "",
  action_plan: "",
  owner: "",
  target_date: "",
  start_date: "",
  end_date: "",
  evidence_url: "",
  result: "",
  result_comment: "",
  review_date: "",
});

const buildActionId = () => {
  const year = new Date().getFullYear();
  const random = Math.floor(1000 + Math.random() * 9000);
  return `AP-${year}-${random}`;
};

const formatDate = (value?: string | null) => {
  if (!value) return "Sin fecha";
  const parsed = new Date(value);
  if (Number.isNaN(parsed.getTime())) return "Sin fecha";
  return parsed.toLocaleDateString("es-MX");
};

const formatDateInput = (value?: string | null) => {
  if (!value) return "";
  return new Date(value).toISOString().split("T")[0] ?? "";
};

const normalizeDate = (value?: string | null) => {
  if (!value) return null;
  const parsed = new Date(value);
  if (Number.isNaN(parsed.getTime())) return null;
  parsed.setHours(0, 0, 0, 0);
  return parsed;
};

const getStatusBadge = (status: StatusOption) => {
  const styles: Record<StatusOption, string> = {
    COMPLETO: "bg-emerald-100 text-emerald-700",
    RETRASADO: "bg-red-100 text-red-700",
    "EN PROCESO": "bg-amber-100 text-amber-700",
    PLANEADO: "bg-blue-100 text-blue-700",
    "SIN ASIGNAR": "bg-slate-200 text-slate-700",
    REABIERTO: "bg-orange-100 text-orange-700",
  };
  return styles[status] ?? "bg-slate-100 text-slate-700";
};

const getResultBadge = (result: ResultOption | null) => {
  if (result === "Conformidad") return "bg-emerald-100 text-emerald-700";
  if (result === "No conformidad") return "bg-red-100 text-red-700";
  return "bg-slate-100 text-slate-700";
};

const getClassificationBadge = (classification: string | null) => {
  if (!classification) return "bg-slate-100 text-slate-700";
  if (classification.includes("mayor")) return "bg-red-100 text-red-700";
  if (classification.includes("menor")) return "bg-amber-100 text-amber-700";
  if (classification.includes("Observación")) return "bg-blue-100 text-blue-700";
  return "bg-emerald-100 text-emerald-700";
};

const computeStatus = (payload: {
  owner?: string | null;
  target_date?: string | null;
  start_date?: string | null;
  end_date?: string | null;
  result?: ResultOption | null;
}): StatusOption => {
  if (payload.result === "No conformidad") return "REABIERTO";
  if (payload.end_date) return "COMPLETO";
  const today = normalizeDate(new Date().toISOString());
  const targetDate = normalizeDate(payload.target_date ?? null);
  if (targetDate && today && today > targetDate) return "RETRASADO";
  if (payload.start_date) return "EN PROCESO";
  if (payload.owner || payload.target_date) return "PLANEADO";
  return "SIN ASIGNAR";
};

export default function ActionPlanPage() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const hasPrefilledSource = useRef(false);
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [records, setRecords] = useState<ActionPlanRecord[]>([]);
  const [searchText, setSearchText] = useState("");
  const [classificationFilter, setClassificationFilter] = useState("all");
  const [statusFilter, setStatusFilter] = useState("all");
  const [sourceFilter, setSourceFilter] = useState("all");
  const [ownerFilter, setOwnerFilter] = useState("all");
  const [createdFrom, setCreatedFrom] = useState("");
  const [createdTo, setCreatedTo] = useState("");
  const [targetFrom, setTargetFrom] = useState("");
  const [targetTo, setTargetTo] = useState("");
  const [endFrom, setEndFrom] = useState("");
  const [endTo, setEndTo] = useState("");
  const [dialogOpen, setDialogOpen] = useState(false);
  const [dialogMode, setDialogMode] = useState<"new" | "edit">("new");
  const [activeRecord, setActiveRecord] = useState<ActionPlanRecord | null>(null);
  const [formState, setFormState] = useState<FormState>(buildEmptyFormState());
  const [saving, setSaving] = useState(false);
  const [deletingId, setDeletingId] = useState<string | null>(null);  
  const sourceParam = searchParams.get("source");
  const ownerParam = searchParams.get("owner");
  const descriptionParam = searchParams.get("description");
  const commentsParam = searchParams.get("comments");
  const rootCauseParam = searchParams.get("rootCause");
  const actionPlanParam = searchParams.get("actionPlan");
  const targetDateParam = searchParams.get("targetDate");

  useEffect(() => {
    if (
      hasPrefilledSource.current ||
      (!sourceParam &&
        !ownerParam &&
        !descriptionParam &&
        !commentsParam &&
        !rootCauseParam &&
        !actionPlanParam &&
        !targetDateParam)
    ) {
      return;
    }
    setDialogMode("new");
    setActiveRecord(null);
    setFormState({
      ...buildEmptyFormState(),
      action_id: buildActionId(),
      source: sourceParam ?? "",
      owner: ownerParam ?? "",
      description: descriptionParam ?? "",
      comments: commentsParam ?? "",
      root_cause_analysis: rootCauseParam ?? "",
      action_plan: actionPlanParam ?? "",
      target_date: targetDateParam ?? "",
    });
    setDialogOpen(true);
    hasPrefilledSource.current = true;
  }, [
    actionPlanParam,
    commentsParam,
    descriptionParam,
    ownerParam,
    rootCauseParam,
    sourceParam,
    targetDateParam,
  ]);

  const canManage = allowedManagers.has(user?.email ?? "");

  const fetchRecords = useCallback(async () => {
    const { data, error } = await supabase
      .from("action_plan_matrix")
      .select("*")
      .order("created_at", { ascending: false });

    if (error) {
      toast.error("No pudimos cargar el plan de acción.");
      return;
    }

    const normalized = (data ?? []).map((record: ActionPlanRecord) => {
      const computedStatus = computeStatus({
        owner: record.owner,
        target_date: record.target_date,
        start_date: record.start_date,
        end_date: record.end_date,
        result: record.result,
      });

      return {
        ...record,
        status: computedStatus,
      };
    });

    setRecords(normalized);
  }, []);

  const loadSession = useCallback(async () => {
    setLoading(true);
    const { data } = await supabase.auth.getSession();
    const currentUser = data.session?.user ?? null;
    if (!currentUser) {
      router.push("/");
      return;
    }

    setUser(currentUser);
    await fetchRecords();
    setLoading(false);
  }, [fetchRecords, router]);

  useEffect(() => {
    loadSession();

    const {
      data: { subscription },
    } = supabase.auth.onAuthStateChange((_event, session) => {
      const nextUser = session?.user ?? null;
      setUser(nextUser);
      if (!nextUser) {
        router.push("/");
      }
    });

    return () => {
      subscription.unsubscribe();
    };
  }, [loadSession, router]);

  const uniqueSources = useMemo(() => {
    const values = new Set(
      records.map((record) => record.source).filter((value): value is string => Boolean(value))
    );
    return ["all", ...Array.from(values)];
  }, [records]);

  const uniqueOwners = useMemo(() => {
    const values = new Set(
      records.map((record) => record.owner).filter((value): value is string => Boolean(value))
    );
    return ["all", ...Array.from(values)];
  }, [records]);

  const filteredRecords = useMemo(() => {
    return records.filter((record) => {
      if (classificationFilter !== "all" && record.classification !== classificationFilter) {
        return false;
      }
      if (statusFilter !== "all" && record.status !== statusFilter) {
        return false;
      }
      if (sourceFilter !== "all" && record.source !== sourceFilter) {
        return false;
      }
      if (ownerFilter !== "all" && record.owner !== ownerFilter) {
        return false;
      }

      const search = searchText.trim().toLowerCase();
      if (search) {
        const haystack = [
          record.action_id,
          record.description,
          record.comments,
          record.action_plan,
          record.root_cause_analysis,
          record.owner,
        ]
          .filter(Boolean)
          .join(" ")
          .toLowerCase();
        if (!haystack.includes(search)) return false;
      }

      const createdDate = normalizeDate(record.created_at ?? null);
      const createdFromDate = normalizeDate(createdFrom || null);
      const createdToDate = normalizeDate(createdTo || null);

      if (createdFromDate && createdDate && createdDate < createdFromDate) return false;
      if (createdToDate && createdDate && createdDate > createdToDate) return false;

      const targetDate = normalizeDate(record.target_date ?? null);
      const targetFromDate = normalizeDate(targetFrom || null);
      const targetToDate = normalizeDate(targetTo || null);

      if (targetFromDate && targetDate && targetDate < targetFromDate) return false;
      if (targetToDate && targetDate && targetDate > targetToDate) return false;

      const endDate = normalizeDate(record.end_date ?? null);
      const endFromDate = normalizeDate(endFrom || null);
      const endToDate = normalizeDate(endTo || null);

      if (endFromDate && endDate && endDate < endFromDate) return false;
      if (endToDate && endDate && endDate > endToDate) return false;

      return true;
    });
  }, [
    classificationFilter,
    createdFrom,
    createdTo,
    endFrom,
    endTo,
    ownerFilter,
    records,
    searchText,
    sourceFilter,
    statusFilter,
    targetFrom,
    targetTo,
  ]);

  const totalsByStatus = useMemo(() => {
    return statusOptions.reduce<Record<StatusOption, number>>((acc, status) => {
      acc[status] = records.filter((record) => record.status === status).length;
      return acc;
    }, {} as Record<StatusOption, number>);
  }, [records]);

  const handleOpenNew = () => {
    setDialogMode("new");
    setActiveRecord(null);
    setFormState({ ...buildEmptyFormState(), action_id: buildActionId() });
    setDialogOpen(true);
  };

  const handleOpenEdit = (record: ActionPlanRecord) => {
    setDialogMode("edit");
    setActiveRecord(record);
    setFormState({
      action_id: record.action_id ?? buildActionId(),
      source: record.source ?? "",
      classification: record.classification ?? "",
      description: record.description ?? "",
      comments: record.comments ?? "",
      root_cause_analysis: record.root_cause_analysis ?? "",
      action_plan: record.action_plan ?? "",
      owner: record.owner ?? "",
      target_date: formatDateInput(record.target_date),
      start_date: formatDateInput(record.start_date),
      end_date: formatDateInput(record.end_date),
      evidence_url: record.evidence_url ?? "",
      result: record.result ?? "",
      result_comment: record.result_comment ?? "",
      review_date: formatDateInput(record.review_date),
    });
    setDialogOpen(true);
  };

  const handleFormChange = (field: keyof FormState, value: string) => {
    setFormState((prev) => ({ ...prev, [field]: value }));
  };

  const handleSave = async () => {
    if (!user) {
      toast.error("Necesitas iniciar sesión para guardar.");
      return;
    }

    if (!formState.action_id) {
      toast.error("El ID de acción es obligatorio.");
      return;
    }

    if (!formState.classification) {
      toast.error("Selecciona una clasificación válida.");
      return;
    }

    if (!formState.description) {
      toast.error("La descripción de la acción es obligatoria.");
      return;
    }

    setSaving(true);

    const computedStatus = computeStatus({
      owner: formState.owner,
      target_date: formState.target_date,
      start_date: formState.start_date,
      end_date: formState.end_date,
      result: formState.result || null,
    });

    const basePayload = {
      action_id: formState.action_id,
      source: formState.source || null,
      classification: formState.classification,
      description: formState.description,
      comments: formState.comments,
      root_cause_analysis: formState.root_cause_analysis,
      action_plan: formState.action_plan,
      owner: formState.owner || null,
      target_date: formState.target_date || null,
      start_date: formState.start_date || null,
      end_date: formState.end_date || null,
      evidence_url: formState.evidence_url || null,
      result: formState.result || null,
      result_comment: formState.result_comment || null,
      review_date: formState.review_date || null,
      status: computedStatus,
      updated_at: new Date().toISOString(),
      updated_by: user.email ?? user.id,
    };

    if (dialogMode === "new") {
      const { error } = await supabase.from("action_plan_matrix").insert({
        ...basePayload,
        created_at: new Date().toISOString(),
      });

      if (error) {
        toast.error("No pudimos guardar el registro.");
        setSaving(false);
        return;
      }

      toast.success("Acción registrada.");
    } else if (activeRecord) {
      const { error } = await supabase
        .from("action_plan_matrix")
        .update(basePayload)
        .eq("id", activeRecord.id);

      if (error) {
        toast.error("No pudimos actualizar el registro.");
        setSaving(false);
        return;
      }

      toast.success("Acción actualizada.");
    }

    await fetchRecords();
    setSaving(false);
    setDialogOpen(false);
  };

  const handleRefresh = async () => {
    setLoading(true);
    await fetchRecords();
    setLoading(false);
  };

  const handleDelete = async (record: ActionPlanRecord) => {
    if (!canManage) {
      toast.error("No tienes permisos para eliminar registros.");
      return;
    }

    const confirmed = window.confirm(
      `¿Eliminar la acción ${record.action_id ?? "seleccionada"}? Esta acción no se puede deshacer.`
    );
    if (!confirmed) return;

    setDeletingId(record.id);
    const { error } = await supabase.from("action_plan_matrix").delete().eq("id", record.id);

    if (error) {
      toast.error("No pudimos eliminar el registro.");
      setDeletingId(null);
      return;
    }

    toast.success("Acción eliminada.");
    await fetchRecords();
    setDeletingId(null);
  };

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-10 bg-gradient-to-br from-slate-50 via-white to-emerald-50 pb-20">
        <section className="relative overflow-hidden rounded-3xl border border-emerald-100 bg-white px-6 py-10 shadow-sm sm:px-10">
          <div className="absolute inset-0 bg-gradient-to-r from-emerald-500/10 to-lime-400/10" />
          <div className="relative space-y-4">
            <p className="text-xs font-semibold uppercase tracking-[0.5em] text-emerald-600">
              PLANEACIÓN
            </p>
            <h1 className="text-3xl font-bold text-gray-900 sm:text-4xl">Plan de acción</h1>
            <p className="max-w-4xl text-base text-gray-600 sm:text-lg">
              Matriz de acciones con seguimiento integral: clasificación, planes de acción,
              responsables y evidencia de cierre.
            </p>
          </div>
        </section>

        <section className="grid gap-6 lg:grid-cols-4">
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between space-y-0">
              <CardTitle className="text-sm font-semibold text-gray-600">Total de acciones</CardTitle>
              <ListChecks className="h-5 w-5 text-emerald-600" />
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold text-gray-900">{records.length}</p>
              <p className="text-xs text-gray-500">Registros activos y cerrados.</p>
            </CardContent>
          </Card>
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between space-y-0">
              <CardTitle className="text-sm font-semibold text-gray-600">En proceso / planeadas</CardTitle>
              <CalendarDays className="h-5 w-5 text-amber-500" />
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold text-gray-900">
                {(totalsByStatus["EN PROCESO"] ?? 0) + (totalsByStatus["PLANEADO"] ?? 0)}
              </p>
              <p className="text-xs text-gray-500">Acciones activas o por iniciar.</p>
            </CardContent>
          </Card>
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between space-y-0">
              <CardTitle className="text-sm font-semibold text-gray-600">Completadas</CardTitle>
              <ShieldCheck className="h-5 w-5 text-emerald-500" />
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold text-gray-900">{totalsByStatus.COMPLETO ?? 0}</p>
              <p className="text-xs text-gray-500">Acciones cerradas correctamente.</p>
            </CardContent>
          </Card>
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between space-y-0">
              <CardTitle className="text-sm font-semibold text-gray-600">Retrasadas</CardTitle>
              <FileSearch className="h-5 w-5 text-red-500" />
            </CardHeader>
            <CardContent>
              <p className="text-3xl font-bold text-gray-900">{totalsByStatus.RETRASADO ?? 0}</p>
              <p className="text-xs text-gray-500">Requieren atención inmediata.</p>
            </CardContent>
          </Card>
        </section>

        <section className="space-y-4 rounded-3xl border border-emerald-100 bg-white p-6 shadow-sm">
          <div className="flex flex-wrap items-center justify-between gap-4">
            <div className="flex items-center gap-2 text-sm font-semibold text-emerald-700">
              <Filter className="h-4 w-4" />
              Filtros y acciones
            </div>
            <div className="flex flex-wrap gap-2">
              <Button
                variant="outline"
                className="border-emerald-200 text-emerald-700"
                onClick={handleRefresh}
                disabled={loading}
              >
                <RefreshCcw className="mr-2 h-4 w-4" />
                Actualizar
              </Button>
              {canManage && (
                <Button onClick={handleOpenNew} className="bg-emerald-600 text-white">
                  <Plus className="mr-2 h-4 w-4" />
                  Agregar acción
                </Button>
              )}
            </div>
          </div>

          <div className="grid gap-4 md:grid-cols-2 xl:grid-cols-4">
            <div className="space-y-2">
              <Label>Clasificación</Label>
              <Select value={classificationFilter} onValueChange={setClassificationFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Todas" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todas</SelectItem>
                  {classificationOptions.map((classification) => (
                    <SelectItem key={classification} value={classification}>
                      {classification}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Estado</Label>
              <Select value={statusFilter} onValueChange={setStatusFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Todos" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todos</SelectItem>
                  {statusOptions.map((status) => (
                    <SelectItem key={status} value={status}>
                      {status}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Fuente</Label>
              <Select value={sourceFilter} onValueChange={setSourceFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Todas" />
                </SelectTrigger>
                <SelectContent>
                  {uniqueSources.map((source) => (
                    <SelectItem key={source} value={source}>
                      {source === "all" ? "Todas" : source}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Responsable</Label>
              <Select value={ownerFilter} onValueChange={setOwnerFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Todos" />
                </SelectTrigger>
                <SelectContent>
                  {uniqueOwners.map((owner) => (
                    <SelectItem key={owner} value={owner}>
                      {owner === "all" ? "Todos" : owner}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>

          <div className="grid gap-4 md:grid-cols-2 xl:grid-cols-4">
            <div className="space-y-2">
              <Label>Búsqueda</Label>
              <Input
                placeholder="Buscar por ID, descripción o responsable"
                value={searchText}
                onChange={(event) => setSearchText(event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Fecha (desde/hasta)</Label>
              <div className="flex gap-2">
                <Input type="date" value={createdFrom} onChange={(e) => setCreatedFrom(e.target.value)} />
                <Input type="date" value={createdTo} onChange={(e) => setCreatedTo(e.target.value)} />
              </div>
            </div>
            <div className="space-y-2">
              <Label>Fecha meta (desde/hasta)</Label>
              <div className="flex gap-2">
                <Input type="date" value={targetFrom} onChange={(e) => setTargetFrom(e.target.value)} />
                <Input type="date" value={targetTo} onChange={(e) => setTargetTo(e.target.value)} />
              </div>
            </div>
            <div className="space-y-2">
              <Label>Fecha término (desde/hasta)</Label>
              <div className="flex gap-2">
                <Input type="date" value={endFrom} onChange={(e) => setEndFrom(e.target.value)} />
                <Input type="date" value={endTo} onChange={(e) => setEndTo(e.target.value)} />
              </div>
            </div>
          </div>
        </section>

        <section className="rounded-3xl border border-emerald-100 bg-white p-6 shadow-sm">
          <div className="mb-4 flex items-center justify-between">
            <div>
              <h2 className="text-lg font-semibold text-gray-900">Matriz de acciones</h2>
              <p className="text-sm text-gray-500">
                {filteredRecords.length} registros visibles de {records.length} en total.
              </p>
            </div>
          </div>

          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>ID Acción</TableHead>
                  <TableHead>Fecha</TableHead>
                  <TableHead>Fuente</TableHead>
                  <TableHead>Clasificación</TableHead>
                  <TableHead>Descripción / Acción</TableHead>
                  <TableHead>Comentarios</TableHead>
                  <TableHead>Análisis de causa raíz</TableHead>
                  <TableHead>Plan de acción</TableHead>
                  <TableHead>Responsable</TableHead>
                  <TableHead>Fecha meta</TableHead>
                  <TableHead>Fecha inicio</TableHead>
                  <TableHead>Fecha término</TableHead>
                  <TableHead>Evidencia</TableHead>
                  <TableHead>Resultado</TableHead>
                  <TableHead>Comentario de resultado</TableHead>
                  <TableHead>Fecha revisión</TableHead>
                  <TableHead>Estado</TableHead>
                  <TableHead>Acciones</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {filteredRecords.map((record) => (
                  <TableRow key={record.id}>
                    <TableCell className="whitespace-nowrap font-medium text-gray-900">
                      {record.action_id ?? "-"}
                    </TableCell>
                    <TableCell className="whitespace-nowrap text-sm text-gray-600">
                      {formatDate(record.created_at)}
                    </TableCell>
                    <TableCell className="min-w-[160px] text-sm text-gray-600">
                      {record.source ?? "Sin fuente"}
                    </TableCell>
                    <TableCell>
                      <Badge className={getClassificationBadge(record.classification)}>
                        {record.classification || "Sin clasificación"}
                      </Badge>
                    </TableCell>
                    <TableCell className="min-w-[240px] text-sm text-gray-600">
                      {record.description || "Sin descripción"}
                    </TableCell>
                    <TableCell className="min-w-[200px] text-sm text-gray-600">
                      {record.comments || "Sin comentarios"}
                    </TableCell>
                    <TableCell className="min-w-[220px] text-sm text-gray-600">
                      {record.root_cause_analysis || "Sin análisis"}
                    </TableCell>
                    <TableCell className="min-w-[220px] text-sm text-gray-600">
                      {record.action_plan || "Sin plan"}
                    </TableCell>
                    <TableCell className="min-w-[160px] text-sm text-gray-600">
                      {record.owner || "Sin responsable"}
                    </TableCell>
                    <TableCell className="whitespace-nowrap text-sm text-gray-600">
                      {formatDate(record.target_date)}
                    </TableCell>
                    <TableCell className="whitespace-nowrap text-sm text-gray-600">
                      {formatDate(record.start_date)}
                    </TableCell>
                    <TableCell className="whitespace-nowrap text-sm text-gray-600">
                      {formatDate(record.end_date)}
                    </TableCell>
                    <TableCell className="min-w-[160px] text-sm text-emerald-700">
                      {record.evidence_url ? (
                        <a
                          href={record.evidence_url}
                          className="font-medium underline decoration-emerald-300 underline-offset-4 hover:text-emerald-900"
                          target="_blank"
                          rel="noreferrer"
                        >
                          Ver evidencia
                        </a>
                      ) : (
                        <span className="text-gray-500">Sin evidencia</span>
                      )}
                    </TableCell>
                    <TableCell>
                      <Badge className={getResultBadge(record.result)}>
                        {record.result ?? "Sin resultado"}
                      </Badge>
                    </TableCell>
                    <TableCell className="min-w-[200px] text-sm text-gray-600">
                      {record.result_comment || "Sin comentario"}
                    </TableCell>
                    <TableCell className="whitespace-nowrap text-sm text-gray-600">
                      {formatDate(record.review_date)}
                    </TableCell>
                    <TableCell>
                      <Badge className={getStatusBadge(record.status ?? "SIN ASIGNAR")}>
                        {record.status ?? "SIN ASIGNAR"}
                      </Badge>
                    </TableCell>
                    <TableCell className="text-right">
                      <div className="flex justify-end gap-2">
                        <Button variant="outline" size="sm" onClick={() => handleOpenEdit(record)}>
                          Ver / Editar
                        </Button>
                        {canManage && (
                          <Button
                            variant="destructive"
                            size="sm"
                            onClick={() => handleDelete(record)}
                            disabled={deletingId === record.id}
                          >
                            <Trash2 className="mr-1 h-4 w-4" />
                            Eliminar
                          </Button>
                        )}
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
                {!filteredRecords.length && (
                  <TableRow>
                    <TableCell colSpan={18} className="py-6 text-center text-sm text-gray-500">
                      {loading ? "Cargando acciones..." : "No hay acciones registradas."}
                    </TableCell>
                  </TableRow>
                )}
              </TableBody>
            </Table>
          </div>
        </section>
      </div>

      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="max-h-[90vh] overflow-y-auto sm:max-w-3xl">
          <DialogHeader>
            <DialogTitle>
              {dialogMode === "new" ? "Registrar acción" : "Detalle de la acción"}
            </DialogTitle>
          </DialogHeader>

          <div className="grid gap-4 md:grid-cols-2">
            <div className="space-y-2">
              <Label>ID Acción</Label>
              <Input
                value={formState.action_id}
                onChange={(event) => handleFormChange("action_id", event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Fuente</Label>
              <Input
                value={formState.source}
                onChange={(event) => handleFormChange("source", event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Clasificación</Label>
              <Select
                value={formState.classification}
                onValueChange={(value) => handleFormChange("classification", value)}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Selecciona" />
                </SelectTrigger>
                <SelectContent>
                  {classificationOptions.map((classification) => (
                    <SelectItem key={classification} value={classification}>
                      {classification}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Responsable</Label>
              <Input
                value={formState.owner}
                onChange={(event) => handleFormChange("owner", event.target.value)}
              />
            </div>
            <div className="space-y-2 md:col-span-2">
              <Label>Descripción / Acción</Label>
              <Textarea
                value={formState.description}
                onChange={(event) => handleFormChange("description", event.target.value)}
              />
            </div>
            <div className="space-y-2 md:col-span-2">
              <Label>Comentarios</Label>
              <Textarea
                value={formState.comments}
                onChange={(event) => handleFormChange("comments", event.target.value)}
              />
            </div>
            <div className="space-y-2 md:col-span-2">
              <Label>Análisis de causa raíz</Label>
              <Textarea
                value={formState.root_cause_analysis}
                onChange={(event) => handleFormChange("root_cause_analysis", event.target.value)}
              />
            </div>
            <div className="space-y-2 md:col-span-2">
              <Label>Plan de acción</Label>
              <Textarea
                value={formState.action_plan}
                onChange={(event) => handleFormChange("action_plan", event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Fecha meta</Label>
              <Input
                type="date"
                value={formState.target_date}
                onChange={(event) => handleFormChange("target_date", event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Fecha inicio</Label>
              <Input
                type="date"
                value={formState.start_date}
                onChange={(event) => handleFormChange("start_date", event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Fecha término</Label>
              <Input
                type="date"
                value={formState.end_date}
                onChange={(event) => handleFormChange("end_date", event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Evidencia (link)</Label>
              <Input
                type="url"
                placeholder="https://"
                value={formState.evidence_url}
                onChange={(event) => handleFormChange("evidence_url", event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Resultado</Label>
              <Select
                value={formState.result}
                onValueChange={(value) => handleFormChange("result", value)}
                disabled={!canManage}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Selecciona" />
                </SelectTrigger>
                <SelectContent>
                  {resultOptions.map((option) => (
                    <SelectItem key={option} value={option}>
                      {option}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <Label>Comentario de resultado</Label>
              <Input
                value={formState.result_comment}
                onChange={(event) => handleFormChange("result_comment", event.target.value)}
                disabled={!canManage}
              />
            </div>
            <div className="space-y-2">
              <Label>Fecha revisión</Label>
              <Input
                type="date"
                value={formState.review_date}
                onChange={(event) => handleFormChange("review_date", event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <Label>Estado calculado</Label>
              <Input
                value={computeStatus({
                  owner: formState.owner,
                  target_date: formState.target_date,
                  start_date: formState.start_date,
                  end_date: formState.end_date,
                  result: formState.result || null,
                })}
                readOnly
              />
            </div>
          </div>

          <DialogFooter className="mt-4">
            <Button variant="outline" onClick={() => setDialogOpen(false)}>
              Cancelar
            </Button>
            <Button onClick={handleSave} disabled={saving || !canManage}>
              {saving ? "Guardando..." : "Guardar"}
            </Button>
          </DialogFooter>
          {!canManage && (
            <p className="text-xs text-red-500">
              Solo los managers autorizados pueden guardar cambios.
            </p>
          )}
        </DialogContent>
      </Dialog>
    </SGIDashboardLayout>
  );
}
