"use client";

import { useCallback, useEffect, useMemo, useState, type FormEvent } from "react";
import Link from "next/link";
import { supabase } from "@/lib/supabase";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Checkbox } from "@/components/ui/checkbox";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Textarea } from "@/components/ui/textarea";
import {
  Activity,
  CircleDot,
  Filter,
  Link2,
  Plus,
  RefreshCw,
  Target,
  Trash2,
} from "lucide-react";
import { toast } from "sonner";
import type { User } from "@supabase/supabase-js";

const allowedManagers = new Set(["ehs.mexico@meiwa.com.mx", "auxiliar.ehs@meiwa.com.mx"]);
const actionSourceLabel = "Aspectos ambientales";

const buildActionPlanHref = (aspect: EnvironmentalAspect) => {
  const params = new URLSearchParams({
    source: actionSourceLabel,
    description: aspect.aspect ?? "",
    comments: aspect.process_activity ?? "",
    actionPlan: aspect.operational_control ?? "",
  });
  return `/meiwa-core/meiwa-sgi/planning/action-plan?${params.toString()}`;
};

type LifecycleMatrixRow = {
  id: string;
  lifecycle_phase: string | null;
  process_type: "internal" | "external" | null;
  main_inputs: string | null;
  main_outputs: string | null;
  influence_control: "influence" | "control" | null;
  order_index: number | null;
};

type EnvironmentalAspect = {
  id: string;
  aspect: string | null;
  element: string | null;
  process_activity: string | null;
  impact_normal: string | null;
  impact_abnormal: string | null;
  impact_emergency: string | null;
  operational_control: string | null;
  control_responsible_area: string | null;
  emergency_plan_link: string | null;
  technical_instructions_link: string | null;
  magnitude_m: number | null;
  control_influence_fci: number | null;
  legal_requirement_rl: number | null;
  frequency_f: number | null;
  extension_e: number | null;
  expected_value_ve: number | null;
  is_significant: boolean | null;
};

type AspectLifecycleLink = {
  id: string;
  aspect_id: string;
  lifecycle_id: string;
};

type LifecycleFormState = {
  lifecycle_phase: string;
  process_type: "internal" | "external";
  main_inputs: string;
  main_outputs: string;
  influence_control: "influence" | "control";
  order_index: string;
};

type AspectFormState = {
  aspect: string;
  element: string;
  process_activity: string;
  impact_normal: string;
  impact_abnormal: string;
  impact_emergency: string;
  operational_control: string;
  control_responsible_area: string;
  emergency_plan_link: string;
  technical_instructions_link: string;
  magnitude_m: string;
  control_influence_fci: string;
  legal_requirement_rl: string;
  frequency_f: string;
  extension_e: string;
};

const lifecycleEmptyForm: LifecycleFormState = {
  lifecycle_phase: "",
  process_type: "internal",
  main_inputs: "",
  main_outputs: "",
  influence_control: "influence",
  order_index: "",
};

const aspectEmptyForm: AspectFormState = {
  aspect: "",
  element: "",
  process_activity: "",
  impact_normal: "",
  impact_abnormal: "",
  impact_emergency: "",
  operational_control: "",
  control_responsible_area: "",
  emergency_plan_link: "",
  technical_instructions_link: "",
  magnitude_m: "1",
  control_influence_fci: "1",
  legal_requirement_rl: "1",
  frequency_f: "1",
  extension_e: "1",
};

const significanceLabels: Record<"significant" | "not_significant", string> = {
  significant: "Significativo",
  not_significant: "No significativo",
};

const computeExpectedValue = (aspect: AspectFormState | EnvironmentalAspect) => {
  const toNumber = (value: string | number | null | undefined) => {
    if (typeof value === "number") return value;
    if (value === null || value === undefined || value === "") return 0;
    const parsed = Number(value);
    return Number.isNaN(parsed) ? 0 : parsed;
  };

  return (
    toNumber(aspect.magnitude_m) +
    toNumber(aspect.control_influence_fci) +
    toNumber(aspect.legal_requirement_rl) +
    toNumber(aspect.frequency_f) +
    toNumber(aspect.extension_e)
  );
};

const toNumberOrNull = (value: string) => {
  if (value === "") return null;
  const parsed = Number(value);
  return Number.isNaN(parsed) ? null : parsed;
};

const diagramSize = 320;
const diagramRadius = 120;

export default function AspectsImpactsPage() {
  const [user, setUser] = useState<User | null>(null);
  const [canManage, setCanManage] = useState(false);
  const [loading, setLoading] = useState(false);
  const [lifecycleRows, setLifecycleRows] = useState<LifecycleMatrixRow[]>([]);
  const [aspects, setAspects] = useState<EnvironmentalAspect[]>([]);
  const [links, setLinks] = useState<AspectLifecycleLink[]>([]);
  const [selectedPhaseId, setSelectedPhaseId] = useState<string | null>(null);
  const [selectedPhaseFilter, setSelectedPhaseFilter] = useState<string>("all");
  const [significanceFilter, setSignificanceFilter] = useState("all");
  const [processFilter, setProcessFilter] = useState("");
  const [controlInfluenceFilter, setControlInfluenceFilter] = useState("all");
  const [legalRequirementFilter, setLegalRequirementFilter] = useState("all");
  const [activeTab, setActiveTab] = useState("lifecycle");
  const [lifecycleDialogOpen, setLifecycleDialogOpen] = useState(false);
  const [editingLifecycle, setEditingLifecycle] = useState<LifecycleMatrixRow | null>(null);
  const [lifecycleForm, setLifecycleForm] = useState<LifecycleFormState>(lifecycleEmptyForm);
  const [aspectDialogOpen, setAspectDialogOpen] = useState(false);
  const [editingAspect, setEditingAspect] = useState<EnvironmentalAspect | null>(null);
  const [aspectForm, setAspectForm] = useState<AspectFormState>(aspectEmptyForm);
  const [aspectLifecycleSelection, setAspectLifecycleSelection] = useState<string[]>([]);
  const [linkDialogOpen, setLinkDialogOpen] = useState(false);
  const [linkLifecycle, setLinkLifecycle] = useState<LifecycleMatrixRow | null>(null);
  const [linkAspectSelection, setLinkAspectSelection] = useState<string[]>([]);

  const loadData = useCallback(async () => {
    setLoading(true);
    const [lifecycleResponse, aspectResponse, linkResponse] = await Promise.all([
      supabase
        .from("lifecycle_matrix")
        .select("*")
        .order("order_index", { ascending: true })
        .order("created_at", { ascending: true }),
      supabase
        .from("environmental_aspects")
        .select("*")
        .order("created_at", { ascending: false }),
      supabase
        .from("aspect_lifecycle_links")
        .select("*")
        .order("created_at", { ascending: false }),
    ]);

    if (lifecycleResponse.error) {
      toast.error("No se pudieron cargar las fases del ciclo de vida.");
    } else {
      setLifecycleRows(lifecycleResponse.data ?? []);
    }

    if (aspectResponse.error) {
      toast.error("No se pudieron cargar los aspectos ambientales.");
    } else {
      setAspects(aspectResponse.data ?? []);
    }

    if (linkResponse.error) {
      toast.error("No se pudieron cargar los enlaces de fases y aspectos.");
    } else {
      setLinks(linkResponse.data ?? []);
    }
    setLoading(false);
  }, []);

  useEffect(() => {
    const loadSession = async () => {
      const { data } = await supabase.auth.getSession();
      const currentUser = data.session?.user ?? null;
      setUser(currentUser);
      setCanManage(allowedManagers.has(currentUser?.email ?? ""));
    };

    loadSession();
    loadData();
  }, [loadData]);

  const lifecycleById = useMemo(() => {
    const map = new Map<string, LifecycleMatrixRow>();
    lifecycleRows.forEach((row) => map.set(row.id, row));
    return map;
  }, [lifecycleRows]);

  const aspectById = useMemo(() => {
    const map = new Map<string, EnvironmentalAspect>();
    aspects.forEach((row) => map.set(row.id, row));
    return map;
  }, [aspects]);

  const linksByAspect = useMemo(() => {
    const map = new Map<string, string[]>();
    links.forEach((link) => {
      const list = map.get(link.aspect_id) ?? [];
      list.push(link.lifecycle_id);
      map.set(link.aspect_id, list);
    });
    return map;
  }, [links]);

  const linksByLifecycle = useMemo(() => {
    const map = new Map<string, string[]>();
    links.forEach((link) => {
      const list = map.get(link.lifecycle_id) ?? [];
      list.push(link.aspect_id);
      map.set(link.lifecycle_id, list);
    });
    return map;
  }, [links]);

  const enrichedAspects = useMemo(() => {
    return aspects.map((aspect) => {
      const expected = computeExpectedValue(aspect);
      const significant = expected >= 10;
      return {
        ...aspect,
        expected_value_ve: expected,
        is_significant: significant,
      };
    });
  }, [aspects]);

  const filteredAspects = useMemo(() => {
    return enrichedAspects.filter((aspect) => {
      const lifecycleIds = linksByAspect.get(aspect.id) ?? [];
      const selectedLifecycle = selectedPhaseId ?? (selectedPhaseFilter !== "all" ? selectedPhaseFilter : null);
      if (selectedLifecycle && !lifecycleIds.includes(selectedLifecycle)) {
        return false;
      }

      if (processFilter) {
        const processValue = aspect.process_activity?.toLowerCase() ?? "";
        if (!processValue.includes(processFilter.toLowerCase())) {
          return false;
        }
      }

      if (significanceFilter !== "all") {
        const shouldBeSignificant = significanceFilter === "significant";
        if ((aspect.expected_value_ve ?? 0) >= 10 !== shouldBeSignificant) {
          return false;
        }
      }

      if (controlInfluenceFilter !== "all") {
        const matches = lifecycleIds.some((id) => {
          const lifecycle = lifecycleById.get(id);
          return lifecycle?.influence_control === controlInfluenceFilter;
        });
        if (!matches) return false;
      }

      if (legalRequirementFilter === "required") {
        if ((aspect.legal_requirement_rl ?? 0) !== 2) {
          return false;
        }
      }

      return true;
    });
  }, [
    enrichedAspects,
    linksByAspect,
    selectedPhaseId,
    selectedPhaseFilter,
    processFilter,
    significanceFilter,
    controlInfluenceFilter,
    legalRequirementFilter,
    lifecycleById,
  ]);

  const kpis = useMemo(() => {
    const total = aspects.length;
    const significant = enrichedAspects.filter((aspect) => (aspect.expected_value_ve ?? 0) >= 10).length;
    const percentage = total ? Math.round((significant / total) * 100) : 0;

    const significantByPhase = lifecycleRows.map((phase) => {
      const aspectIds = linksByLifecycle.get(phase.id) ?? [];
      const count = aspectIds.filter((aspectId) => {
        const aspect = aspectById.get(aspectId);
        if (!aspect) return false;
        return computeExpectedValue(aspect) >= 10;
      }).length;
      return { phase, count };
    });

    const significantByInfluence = ["control", "influence"].map((type) => {
      const lifecycleIds = lifecycleRows
        .filter((phase) => phase.influence_control === type)
        .map((phase) => phase.id);
      const count = enrichedAspects.filter((aspect) => {
        if ((aspect.expected_value_ve ?? 0) < 10) return false;
        const lifecycleIdsForAspect = linksByAspect.get(aspect.id) ?? [];
        return lifecycleIdsForAspect.some((id) => lifecycleIds.includes(id));
      }).length;
      return { type, count };
    });

    return {
      total,
      significant,
      percentage,
      significantByPhase,
      significantByInfluence,
    };
  }, [aspects, enrichedAspects, lifecycleRows, linksByLifecycle, aspectById, linksByAspect]);

  const openLifecycleDialog = (row?: LifecycleMatrixRow) => {
    if (row) {
      setEditingLifecycle(row);
      setLifecycleForm({
        lifecycle_phase: row.lifecycle_phase ?? "",
        process_type: row.process_type ?? "internal",
        main_inputs: row.main_inputs ?? "",
        main_outputs: row.main_outputs ?? "",
        influence_control: row.influence_control ?? "influence",
        order_index: row.order_index?.toString() ?? "",
      });
    } else {
      setEditingLifecycle(null);
      setLifecycleForm(lifecycleEmptyForm);
    }
    setLifecycleDialogOpen(true);
  };

  const handleLifecycleSubmit = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!canManage) return;

    const payload = {
      lifecycle_phase: lifecycleForm.lifecycle_phase.trim(),
      process_type: lifecycleForm.process_type,
      main_inputs: lifecycleForm.main_inputs.trim() || null,
      main_outputs: lifecycleForm.main_outputs.trim() || null,
      influence_control: lifecycleForm.influence_control,
      order_index: toNumberOrNull(lifecycleForm.order_index),
      updated_at: new Date().toISOString(),
    };

    let response;
    if (editingLifecycle) {
      response = await supabase
        .from("lifecycle_matrix")
        .update(payload)
        .eq("id", editingLifecycle.id);
    } else {
      response = await supabase.from("lifecycle_matrix").insert({
        ...payload,
        created_at: new Date().toISOString(),
      });
    }

    if (response.error) {
      toast.error("No se pudo guardar la fase del ciclo de vida.");
      return;
    }

    toast.success("Fase del ciclo de vida guardada.");
    setLifecycleDialogOpen(false);
    await loadData();
  };

  const handleLifecycleDelete = async (row: LifecycleMatrixRow) => {
    if (!canManage) return;
    const confirmDelete = window.confirm(
      "¿Quieres eliminar esta fase? Se perderán los vínculos con aspectos ambientales.",
    );
    if (!confirmDelete) return;

    const response = await supabase.from("lifecycle_matrix").delete().eq("id", row.id);
    if (response.error) {
      toast.error("No se pudo eliminar la fase.");
      return;
    }
    toast.success("Fase eliminada.");
    await loadData();
  };

  const openAspectDialog = (row?: EnvironmentalAspect) => {
    if (row) {
      setEditingAspect(row);
      setAspectForm({
        aspect: row.aspect ?? "",
        element: row.element ?? "",
        process_activity: row.process_activity ?? "",
        impact_normal: row.impact_normal ?? "",
        impact_abnormal: row.impact_abnormal ?? "",
        impact_emergency: row.impact_emergency ?? "",
        operational_control: row.operational_control ?? "",
        control_responsible_area: row.control_responsible_area ?? "",
        emergency_plan_link: row.emergency_plan_link ?? "",
        technical_instructions_link: row.technical_instructions_link ?? "",
        magnitude_m: row.magnitude_m?.toString() ?? "1",
        control_influence_fci: row.control_influence_fci?.toString() ?? "1",
        legal_requirement_rl: row.legal_requirement_rl?.toString() ?? "1",
        frequency_f: row.frequency_f?.toString() ?? "1",
        extension_e: row.extension_e?.toString() ?? "1",
      });
      setAspectLifecycleSelection(linksByAspect.get(row.id) ?? []);
    } else {
      setEditingAspect(null);
      setAspectForm(aspectEmptyForm);
      setAspectLifecycleSelection([]);
    }
    setAspectDialogOpen(true);
  };

  const handleAspectSubmit = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!canManage) return;

    const expectedValue = computeExpectedValue(aspectForm);
    const payload = {
      aspect: aspectForm.aspect.trim(),
      element: aspectForm.element.trim() || null,
      process_activity: aspectForm.process_activity.trim() || null,
      impact_normal: aspectForm.impact_normal.trim() || null,
      impact_abnormal: aspectForm.impact_abnormal.trim() || null,
      impact_emergency: aspectForm.impact_emergency.trim() || null,
      operational_control: aspectForm.operational_control.trim() || null,
      control_responsible_area: aspectForm.control_responsible_area.trim() || null,
      emergency_plan_link: aspectForm.emergency_plan_link.trim() || null,
      technical_instructions_link: aspectForm.technical_instructions_link.trim() || null,
      magnitude_m: toNumberOrNull(aspectForm.magnitude_m),
      control_influence_fci: toNumberOrNull(aspectForm.control_influence_fci),
      legal_requirement_rl: toNumberOrNull(aspectForm.legal_requirement_rl),
      frequency_f: toNumberOrNull(aspectForm.frequency_f),
      extension_e: toNumberOrNull(aspectForm.extension_e),
      expected_value_ve: expectedValue,
      is_significant: expectedValue >= 10,
      updated_at: new Date().toISOString(),
      updated_by: user?.email ?? null,
    };

    let response;
    let aspectId = editingAspect?.id;
    if (editingAspect) {
      response = await supabase.from("environmental_aspects").update(payload).eq("id", editingAspect.id);
    } else {
      const insertResponse = await supabase
        .from("environmental_aspects")
        .insert({
          ...payload,
          created_at: new Date().toISOString(),
          created_by: user?.email ?? null,
        })
        .select("id")
        .single();
      response = insertResponse;
      aspectId = insertResponse.data?.id ?? null;
    }

    if (response?.error || !aspectId) {
      toast.error("No se pudo guardar el aspecto ambiental.");
      return;
    }

    await supabase.from("aspect_lifecycle_links").delete().eq("aspect_id", aspectId);
    if (aspectLifecycleSelection.length) {
      const linkPayload = aspectLifecycleSelection.map((lifecycleId) => ({
        aspect_id: aspectId,
        lifecycle_id: lifecycleId,
      }));
      const linkResponse = await supabase.from("aspect_lifecycle_links").insert(linkPayload);
      if (linkResponse.error) {
        toast.error("No se pudieron guardar los enlaces del aspecto.");
      }
    }

    toast.success("Aspecto ambiental guardado.");
    setAspectDialogOpen(false);
    await loadData();
  };

  const handleAspectDelete = async (row: EnvironmentalAspect) => {
    if (!canManage) return;
    const confirmDelete = window.confirm("¿Quieres eliminar este aspecto ambiental?");
    if (!confirmDelete) return;

    await supabase.from("aspect_lifecycle_links").delete().eq("aspect_id", row.id);
    const response = await supabase.from("environmental_aspects").delete().eq("id", row.id);
    if (response.error) {
      toast.error("No se pudo eliminar el aspecto ambiental.");
      return;
    }
    toast.success("Aspecto ambiental eliminado.");
    await loadData();
  };

  const openLinkDialog = (row: LifecycleMatrixRow) => {
    setLinkLifecycle(row);
    setLinkAspectSelection(linksByLifecycle.get(row.id) ?? []);
    setLinkDialogOpen(true);
  };

  const handleLinkSubmit = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!canManage || !linkLifecycle) return;
    const lifecycleId = linkLifecycle.id;

    await supabase.from("aspect_lifecycle_links").delete().eq("lifecycle_id", lifecycleId);
    if (linkAspectSelection.length) {
      const payload = linkAspectSelection.map((aspectId) => ({
        aspect_id: aspectId,
        lifecycle_id: lifecycleId,
      }));
      const response = await supabase.from("aspect_lifecycle_links").insert(payload);
      if (response.error) {
        toast.error("No se pudieron actualizar los enlaces de la fase.");
        return;
      }
    }

    toast.success("Aspectos vinculados a la fase.");
    setLinkDialogOpen(false);
    await loadData();
  };

  const handleRefresh = async () => {
    await loadData();
  };

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-8 bg-gradient-to-br from-slate-50 via-white to-emerald-50 pb-16">
        <section className="rounded-3xl border border-emerald-100 bg-white p-8 shadow-sm">
          <div className="flex flex-col gap-4 lg:flex-row lg:items-center lg:justify-between">
            <div>
              <p className="text-xs font-semibold uppercase tracking-[0.4em] text-emerald-600">
                Planificación
              </p>
              <h1 className="mt-2 text-3xl font-bold text-gray-900">Aspectos e Impactos</h1>
              <p className="mt-2 max-w-3xl text-sm text-gray-600">
                Documenta aspectos e impactos ambientales con enfoque de ciclo de vida, criterios
                de significancia y evidencia conforme a ISO 14001:2015
              </p>
            </div>
            <div className="flex flex-wrap items-center gap-2">
              <Badge className="bg-emerald-100 text-emerald-700">
                {canManage ? "Edición habilitada" : "Solo lectura"}
              </Badge>
              <Button variant="secondary" className="gap-2" onClick={handleRefresh} disabled={loading}>
                <RefreshCw className="h-4 w-4" /> Actualizar
              </Button>
            </div>
          </div>
        </section>

        <section className="grid gap-4 lg:grid-cols-4">
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-sm font-medium text-gray-600">Total de aspectos</CardTitle>
              <Activity className="h-4 w-4 text-emerald-500" />
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-semibold text-gray-900">{kpis.total}</div>
              <p className="mt-2 text-xs text-gray-500">Registro consolidado de aspectos.</p>
            </CardContent>
          </Card>
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-sm font-medium text-gray-600">Total significativos</CardTitle>
              <Target className="h-4 w-4 text-amber-500" />
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-semibold text-gray-900">{kpis.significant}</div>
              <p className="mt-2 text-xs text-gray-500">Requieren plan de acción.</p>
            </CardContent>
          </Card>
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-sm font-medium text-gray-600">% significativos</CardTitle>
              <CircleDot className="h-4 w-4 text-emerald-500" />
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-semibold text-gray-900">{kpis.percentage}%</div>
              <p className="mt-2 text-xs text-gray-500">Proporción total de aspectos.</p>
            </CardContent>
          </Card>
          <Card className="border-emerald-100">
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-sm font-medium text-gray-600">Significativos por control</CardTitle>
              <Filter className="h-4 w-4 text-emerald-500" />
            </CardHeader>
            <CardContent className="space-y-2">
              {kpis.significantByInfluence.map((item) => (
                <div key={item.type} className="flex items-center justify-between text-sm">
                  <span className="text-gray-600 capitalize">{item.type}</span>
                  <Badge className="bg-emerald-100 text-emerald-700">{item.count}</Badge>
                </div>
              ))}
            </CardContent>
          </Card>
        </section>

        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-8">
          <TabsList className="grid w-full max-w-xl grid-cols-2">
            <TabsTrigger value="lifecycle">Ciclo de vida</TabsTrigger>
            <TabsTrigger value="aspects">Aspectos e impactos</TabsTrigger>
          </TabsList>

          <TabsContent value="lifecycle" className="space-y-8">
            <section className="grid gap-6 xl:grid-cols-[minmax(0,2fr)_minmax(0,1fr)]">
              <Card className="border-emerald-100">
                <CardHeader className="flex flex-col gap-2 lg:flex-row lg:items-center lg:justify-between">
                  <div>
                    <CardTitle>Matriz de ciclo de vida</CardTitle>
                    <p className="text-sm text-gray-500">
                      Identifica fases, entradas/salidas y aspectos asociados al ciclo de vida.
                    </p>
                  </div>
                  {canManage && (
                    <Button className="gap-2" onClick={() => openLifecycleDialog()}>
                      <Plus className="h-4 w-4" /> Nueva fase
                    </Button>
                  )}
                </CardHeader>
                <CardContent className="overflow-x-auto">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>Fase</TableHead>
                        <TableHead>Tipo proceso</TableHead>
                        <TableHead>Entradas principales</TableHead>
                        <TableHead>Salidas principales</TableHead>
                        <TableHead>Aspectos ambientales</TableHead>
                        <TableHead>Influencia / Control</TableHead>
                        {canManage && <TableHead className="text-right">Acciones</TableHead>}
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {lifecycleRows.map((row) => {
                        const aspectIds = linksByLifecycle.get(row.id) ?? [];
                        const aspectLabels = aspectIds
                          .map((aspectId) => aspectById.get(aspectId))
                          .filter(Boolean) as EnvironmentalAspect[];
                        return (
                          <TableRow key={row.id}>
                            <TableCell className="font-semibold text-gray-900">
                              {row.lifecycle_phase ?? "-"}
                            </TableCell>
                            <TableCell className="capitalize">
                              {row.process_type === "internal" ? "Interno" : "Externo"}
                            </TableCell>
                            <TableCell className="text-sm text-gray-600">
                              {row.main_inputs ?? "-"}
                            </TableCell>
                            <TableCell className="text-sm text-gray-600">
                              {row.main_outputs ?? "-"}
                            </TableCell>
                            <TableCell>
                              <div className="flex flex-wrap gap-2">
                                {aspectLabels.length ? (
                                  aspectLabels.map((aspect) => (
                                    <Badge key={aspect.id} variant="secondary">
                                      {aspect.aspect ?? "Aspecto"}
                                    </Badge>
                                  ))
                                ) : (
                                  <span className="text-sm text-gray-500">Sin vínculos</span>
                                )}
                                {canManage && (
                                  <Button
                                    variant="outline"
                                    size="sm"
                                    className="gap-2"
                                    onClick={() => openLinkDialog(row)}
                                  >
                                    <Link2 className="h-4 w-4" /> Vincular
                                  </Button>
                                )}
                              </div>
                            </TableCell>
                            <TableCell className="capitalize">
                              {row.influence_control === "control" ? "Control" : "Influencia"}
                            </TableCell>
                            {canManage && (
                              <TableCell className="text-right">
                                <div className="flex items-center justify-end gap-2">
                                  <Button
                                    variant="secondary"
                                    size="sm"
                                    onClick={() => openLifecycleDialog(row)}
                                  >
                                    Editar
                                  </Button>
                                  <Button
                                    variant="ghost"
                                    size="icon"
                                    onClick={() => handleLifecycleDelete(row)}
                                  >
                                    <Trash2 className="h-4 w-4 text-rose-500" />
                                  </Button>
                                </div>
                              </TableCell>
                            )}
                          </TableRow>
                        );
                      })}
                    </TableBody>
                  </Table>
                </CardContent>
              </Card>

              <Card className="border-emerald-100">
                <CardHeader>
                  <CardTitle>Diagrama circular del ciclo de vida</CardTitle>
                  <p className="text-sm text-gray-500">
                    Haz clic en una fase para filtrar los aspectos ambientales asociados.
                  </p>
                </CardHeader>
                <CardContent className="flex flex-col items-center gap-6">
                  <div
                    className="relative rounded-full border border-emerald-100 bg-gradient-to-br from-white via-emerald-50 to-white"
                    style={{ width: diagramSize, height: diagramSize }}
                  >
                    {lifecycleRows.length ? (
                      lifecycleRows.map((phase, index) => {
                        const totalPhases = lifecycleRows.length;
                        const angle = (index / totalPhases) * Math.PI * 2 - Math.PI / 2;
                        const x = diagramSize / 2 + diagramRadius * Math.cos(angle);
                        const y = diagramSize / 2 + diagramRadius * Math.sin(angle);
                        const aspectIds = linksByLifecycle.get(phase.id) ?? [];
                        const significantCount = aspectIds.filter((aspectId) => {
                          const aspect = aspectById.get(aspectId);
                          if (!aspect) return false;
                          return computeExpectedValue(aspect) >= 10;
                        }).length;
                        const isSelected = selectedPhaseId === phase.id;

                        return (
                          <button
                            key={phase.id}
                            type="button"
                            onClick={() =>
                              setSelectedPhaseId((prev) => (prev === phase.id ? null : phase.id))
                            }
                            className={`absolute -translate-x-1/2 -translate-y-1/2 rounded-2xl border px-3 py-2 text-left text-xs shadow-sm transition-all duration-200 ${
                              isSelected
                                ? "border-emerald-500 bg-emerald-500 text-white"
                                : "border-emerald-100 bg-white text-gray-700 hover:border-emerald-300"
                            }`}
                            style={{ left: x, top: y, width: 140 }}
                          >
                            <p className="font-semibold">{phase.lifecycle_phase ?? "Fase"}</p>
                            <p className="mt-1 flex items-center justify-between text-[11px]">
                              <span>Aspectos: {aspectIds.length}</span>
                              <span>Sig.: {significantCount}</span>
                            </p>
                          </button>
                        );
                      })
                    ) : (
                      <div className="absolute inset-0 flex items-center justify-center text-sm text-gray-500">
                        No hay fases registradas.
                      </div>
                    )}
                    <div className="absolute left-1/2 top-1/2 flex -translate-x-1/2 -translate-y-1/2 flex-col items-center gap-1 rounded-full border border-emerald-100 bg-white px-4 py-3 text-center shadow-sm">
                      <span className="text-xs font-semibold text-emerald-600">Ciclo de vida</span>
                      <span className="text-[11px] text-gray-500">ISO 14001: 6.1.2</span>
                    </div>
                  </div>
                  <div className="flex flex-wrap items-center justify-center gap-2">
                    {selectedPhaseId ? (
                      <Badge className="bg-emerald-100 text-emerald-700">
                        Filtro activo: {lifecycleById.get(selectedPhaseId)?.lifecycle_phase ?? "Fase"}
                      </Badge>
                    ) : (
                      <Badge variant="secondary">Sin filtro activo</Badge>
                    )}
                    {selectedPhaseId && (
                      <Button variant="outline" size="sm" onClick={() => setSelectedPhaseId(null)}>
                        Limpiar
                      </Button>
                    )}
                  </div>
                  <div className="w-full space-y-2">
                    <p className="text-xs font-semibold uppercase tracking-[0.3em] text-emerald-600">
                      Significativos por fase
                    </p>
                    <div className="space-y-2">
                      {kpis.significantByPhase.map((item) => (
                        <div
                          key={item.phase.id}
                          className="flex items-center justify-between rounded-2xl border border-emerald-100 bg-white px-3 py-2 text-sm"
                        >
                          <span className="text-gray-600">{item.phase.lifecycle_phase ?? "Fase"}</span>
                          <Badge className="bg-emerald-100 text-emerald-700">{item.count}</Badge>
                        </div>
                      ))}
                      {!kpis.significantByPhase.length && (
                        <p className="text-xs text-gray-500">Sin fases registradas.</p>
                      )}
                    </div>
                  </div>
                </CardContent>
              </Card>
            </section>
          </TabsContent>

          <TabsContent value="aspects" className="space-y-8">
            <Card className="border-emerald-100">
              <CardHeader className="flex flex-col gap-4 lg:flex-row lg:items-center lg:justify-between">
                <div>
                  <CardTitle>Matriz de aspectos ambientales</CardTitle>
                  <p className="text-sm text-gray-500">
                    Evalúa significancia con criterios y comunica aspectos significativos.
                  </p>
                </div>
                {canManage && (
                  <Button className="gap-2" onClick={() => openAspectDialog()}>
                    <Plus className="h-4 w-4" /> Nuevo aspecto
                  </Button>
                )}
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid gap-4 md:grid-cols-2 xl:grid-cols-4">
                  <div>
                    <Label className="text-xs text-gray-500">Fase del ciclo de vida</Label>
                    <Select value={selectedPhaseFilter} onValueChange={setSelectedPhaseFilter}>
                      <SelectTrigger>
                        <SelectValue placeholder="Todas" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="all">Todas</SelectItem>
                        {lifecycleRows.map((row) => (
                          <SelectItem key={row.id} value={row.id}>
                            {row.lifecycle_phase ?? "Fase"}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                  <div>
                    <Label className="text-xs text-gray-500">Proceso / actividad</Label>
                    <Input
                      value={processFilter}
                      onChange={(event) => setProcessFilter(event.target.value)}
                      placeholder="Buscar proceso"
                    />
                  </div>
                  <div>
                    <Label className="text-xs text-gray-500">Significancia</Label>
                    <Select value={significanceFilter} onValueChange={setSignificanceFilter}>
                      <SelectTrigger>
                        <SelectValue placeholder="Todas" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="all">Todas</SelectItem>
                        <SelectItem value="significant">Solo significativos</SelectItem>
                        <SelectItem value="not_significant">No significativos</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  <div>
                    <Label className="text-xs text-gray-500">Control / influencia</Label>
                    <Select value={controlInfluenceFilter} onValueChange={setControlInfluenceFilter}>
                      <SelectTrigger>
                        <SelectValue placeholder="Todas" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="all">Todas</SelectItem>
                        <SelectItem value="control">Control</SelectItem>
                        <SelectItem value="influence">Influencia</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                </div>
                <div className="grid gap-4 md:grid-cols-2">
                  <div>
                    <Label className="text-xs text-gray-500">Requisito legal (RL=2)</Label>
                    <Select value={legalRequirementFilter} onValueChange={setLegalRequirementFilter}>
                      <SelectTrigger>
                        <SelectValue placeholder="Todos" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="all">Todos</SelectItem>
                        <SelectItem value="required">Solo con requisito legal</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                </div>
                <div className="overflow-x-auto">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>Identificación</TableHead>
                        <TableHead>Aspecto ambiental</TableHead>
                        <TableHead>Elemento</TableHead>
                        <TableHead>Origen en ciclo de vida</TableHead>
                        <TableHead>Proceso / actividad</TableHead>
                        <TableHead>Impacto normal</TableHead>
                        <TableHead>Impacto anormal</TableHead>
                        <TableHead>Impacto emergencia</TableHead>
                        <TableHead>Control operacional</TableHead>
                        <TableHead>Área responsable</TableHead>
                        <TableHead>Plan emergencia</TableHead>
                        <TableHead>Instrucciones técnicas</TableHead>
                        <TableHead>M</TableHead>
                        <TableHead>FCI</TableHead>
                        <TableHead>RL</TableHead>
                        <TableHead>F</TableHead>
                        <TableHead>E</TableHead>
                        <TableHead>VE</TableHead>
                        <TableHead>Significancia</TableHead>
                        <TableHead>Crear acción</TableHead>
                        {canManage && <TableHead className="text-right">Acciones</TableHead>}
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {filteredAspects.map((aspect) => {
                        const lifecycleIds = linksByAspect.get(aspect.id) ?? [];
                        const lifecycleBadges = lifecycleIds
                          .map((id) => lifecycleById.get(id))
                          .filter(Boolean) as LifecycleMatrixRow[];
                        const expected = aspect.expected_value_ve ?? 0;
                        const isSignificant = expected >= 10;
                        return (
                          <TableRow key={aspect.id}>
                            <TableCell className="text-xs text-gray-500">
                              {aspect.id.slice(0, 8).toUpperCase()}
                            </TableCell>
                            <TableCell className="font-semibold text-gray-900">
                              {aspect.aspect ?? "-"}
                            </TableCell>
                            <TableCell className="text-sm text-gray-600">
                              {aspect.element ?? "-"}
                            </TableCell>
                            <TableCell>
                              <div className="flex flex-wrap gap-2">
                                {lifecycleBadges.length ? (
                                  lifecycleBadges.map((phase) => (
                                    <Badge key={phase.id} variant="secondary">
                                      {phase.lifecycle_phase ?? "Fase"}
                                    </Badge>
                                  ))
                                ) : (
                                  <span className="text-sm text-gray-500">Sin fase</span>
                                )}
                              </div>
                            </TableCell>
                            <TableCell className="text-sm text-gray-600">
                              {aspect.process_activity ?? "-"}
                            </TableCell>
                            <TableCell className="text-sm text-gray-600">
                              {aspect.impact_normal ?? "-"}
                            </TableCell>
                            <TableCell className="text-sm text-gray-600">
                              {aspect.impact_abnormal ?? "-"}
                            </TableCell>
                            <TableCell className="text-sm text-gray-600">
                              {aspect.impact_emergency ?? "-"}
                            </TableCell>
                            <TableCell className="text-sm text-gray-600">
                              {aspect.operational_control ?? "-"}
                            </TableCell>
                            <TableCell className="text-sm text-gray-600">
                              {aspect.control_responsible_area ?? "-"}
                            </TableCell>
                            <TableCell className="text-sm text-emerald-600">
                              {aspect.emergency_plan_link ? (
                                <a
                                  href={aspect.emergency_plan_link}
                                  target="_blank"
                                  rel="noreferrer"
                                  className="underline"
                                >
                                  Ver plan
                                </a>
                              ) : (
                                "-"
                              )}
                            </TableCell>
                            <TableCell className="text-sm text-emerald-600">
                              {aspect.technical_instructions_link ? (
                                <a
                                  href={aspect.technical_instructions_link}
                                  target="_blank"
                                  rel="noreferrer"
                                  className="underline"
                                >
                                  Ver instrucciones
                                </a>
                              ) : (
                                "-"
                              )}
                            </TableCell>
                            <TableCell>{aspect.magnitude_m ?? "-"}</TableCell>
                            <TableCell>{aspect.control_influence_fci ?? "-"}</TableCell>
                            <TableCell>{aspect.legal_requirement_rl ?? "-"}</TableCell>
                            <TableCell>{aspect.frequency_f ?? "-"}</TableCell>
                            <TableCell>{aspect.extension_e ?? "-"}</TableCell>
                            <TableCell className="font-semibold text-gray-900">{expected}</TableCell>
                            <TableCell>
                              <Badge
                                className={
                                  isSignificant
                                    ? "bg-amber-100 text-amber-800"
                                    : "bg-emerald-100 text-emerald-700"
                                }
                              >
                                {isSignificant
                                  ? significanceLabels.significant
                                  : significanceLabels.not_significant}
                              </Badge>
                            </TableCell>
                            <TableCell>
                              <Button
                                asChild
                                size="sm"
                                className="bg-sky-500 text-white hover:bg-sky-600"
                              >
                                <Link href={buildActionPlanHref(aspect)}>
                                  Crear acción
                                </Link>
                              </Button>
                            </TableCell>                            
                            {canManage && (
                              <TableCell className="text-right">
                                <div className="flex items-center justify-end gap-2">
                                  <Button
                                    variant="secondary"
                                    size="sm"
                                    onClick={() => openAspectDialog(aspect)}
                                  >
                                    Editar
                                  </Button>
                                  <Button
                                    variant="ghost"
                                    size="icon"
                                    onClick={() => handleAspectDelete(aspect)}
                                  >
                                    <Trash2 className="h-4 w-4 text-rose-500" />
                                  </Button>
                                </div>
                              </TableCell>
                            )}
                          </TableRow>
                        );
                      })}
                    </TableBody>
                  </Table>
                </div>
              </CardContent>
            </Card>

            <Card className="border-emerald-100">
              <CardHeader>
                <CardTitle>Metodología y criterios de significancia</CardTitle>
                <p className="text-sm text-gray-500">
                  Información documentada para determinar significancia según ISO 14001 (6.1.2).
                </p>
              </CardHeader>
              <CardContent className="grid gap-4 md:grid-cols-2">
                <div className="rounded-2xl border border-emerald-100 bg-white p-4">
                  <h3 className="text-sm font-semibold text-gray-900">Magnitud (M)</h3>
                  <ul className="mt-2 space-y-1 text-sm text-gray-600">
                    <li>3: Alteración muy significativa (&gt;50%).</li>
                    <li>2: Cambios entre 10% y 50%.</li>
                    <li>1: Cambios menores a 10%.</li>
                  </ul>
                </div>
                <div className="rounded-2xl border border-emerald-100 bg-white p-4">
                  <h3 className="text-sm font-semibold text-gray-900">Factor control/influencia (FCI)</h3>
                  <ul className="mt-2 space-y-1 text-sm text-gray-600">
                    <li>2: No existe control/influencia.</li>
                    <li>1: Existe al menos un control/influencia.</li>
                  </ul>
                </div>
                <div className="rounded-2xl border border-emerald-100 bg-white p-4">
                  <h3 className="text-sm font-semibold text-gray-900">Requisito legal (RL)</h3>
                  <ul className="mt-2 space-y-1 text-sm text-gray-600">
                    <li>2: Existe requisito legal asociado.</li>
                    <li>1: No existe requisito legal asociado.</li>
                  </ul>
                </div>
                <div className="rounded-2xl border border-emerald-100 bg-white p-4">
                  <h3 className="text-sm font-semibold text-gray-900">Frecuencia (F)</h3>
                  <ul className="mt-2 space-y-1 text-sm text-gray-600">
                    <li>3: &gt;3 veces/año.</li>
                    <li>2: 2–3 veces/año.</li>
                    <li>1: 1 vez/año.</li>
                  </ul>
                </div>
                <div className="rounded-2xl border border-emerald-100 bg-white p-4">
                  <h3 className="text-sm font-semibold text-gray-900">Extensión (E)</h3>
                  <ul className="mt-2 space-y-1 text-sm text-gray-600">
                    <li>2: Fuera de zona de trabajo o percibido por comunidad.</li>
                    <li>1: Circunscrito al área fuente/zona trabajo.</li>
                  </ul>
                </div>
                <div className="rounded-2xl border border-emerald-100 bg-white p-4">
                  <h3 className="text-sm font-semibold text-gray-900">Umbral VE</h3>
                  <ul className="mt-2 space-y-1 text-sm text-gray-600">
                    <li>&gt;=10: Significativo (requiere plan de acción).</li>
                    <li>&lt;10: No significativo.</li>
                  </ul>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>

      <Dialog open={lifecycleDialogOpen} onOpenChange={setLifecycleDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>{editingLifecycle ? "Editar fase" : "Nueva fase"}</DialogTitle>
          </DialogHeader>
          <form className="space-y-4" onSubmit={handleLifecycleSubmit}>
            <div className="grid gap-4 md:grid-cols-2">
              <div>
                <Label>Fase del ciclo de vida</Label>
                <Input
                  value={lifecycleForm.lifecycle_phase}
                  onChange={(event) =>
                    setLifecycleForm((prev) => ({ ...prev, lifecycle_phase: event.target.value }))
                  }
                  required
                />
              </div>
              <div>
                <Label>Tipo proceso</Label>
                <Select
                  value={lifecycleForm.process_type}
                  onValueChange={(value: "internal" | "external") =>
                    setLifecycleForm((prev) => ({ ...prev, process_type: value }))
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="internal">Interno</SelectItem>
                    <SelectItem value="external">Externo</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label>Entradas principales</Label>
                <Textarea
                  value={lifecycleForm.main_inputs}
                  onChange={(event) =>
                    setLifecycleForm((prev) => ({ ...prev, main_inputs: event.target.value }))
                  }
                />
              </div>
              <div>
                <Label>Salidas principales</Label>
                <Textarea
                  value={lifecycleForm.main_outputs}
                  onChange={(event) =>
                    setLifecycleForm((prev) => ({ ...prev, main_outputs: event.target.value }))
                  }
                />
              </div>
              <div>
                <Label>Influencia / Control</Label>
                <Select
                  value={lifecycleForm.influence_control}
                  onValueChange={(value: "influence" | "control") =>
                    setLifecycleForm((prev) => ({ ...prev, influence_control: value }))
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="influence">Influencia</SelectItem>
                    <SelectItem value="control">Control</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label>Orden</Label>
                <Input
                  type="number"
                  value={lifecycleForm.order_index}
                  onChange={(event) =>
                    setLifecycleForm((prev) => ({ ...prev, order_index: event.target.value }))
                  }
                />
              </div>
            </div>
            <div className="flex justify-end gap-2">
              <Button type="button" variant="secondary" onClick={() => setLifecycleDialogOpen(false)}>
                Cancelar
              </Button>
              <Button type="submit">Guardar</Button>
            </div>
          </form>
        </DialogContent>
      </Dialog>

      <Dialog open={aspectDialogOpen} onOpenChange={setAspectDialogOpen}>
        <DialogContent className="max-w-4xl">
          <DialogHeader>
            <DialogTitle>{editingAspect ? "Editar aspecto" : "Nuevo aspecto"}</DialogTitle>
          </DialogHeader>
          <form className="max-h-[75vh] space-y-4 overflow-y-auto pr-2" onSubmit={handleAspectSubmit}>
            <div className="grid gap-4 md:grid-cols-2">
              <div>
                <Label>Identificación del aspecto</Label>
                <Input
                  value={aspectForm.aspect}
                  onChange={(event) => setAspectForm((prev) => ({ ...prev, aspect: event.target.value }))}
                  required
                />
              </div>
              <div>
                <Label>Elemento</Label>
                <Input
                  value={aspectForm.element}
                  onChange={(event) => setAspectForm((prev) => ({ ...prev, element: event.target.value }))}
                />
              </div>
              <div>
                <Label>Proceso / actividad</Label>
                <Input
                  value={aspectForm.process_activity}
                  onChange={(event) =>
                    setAspectForm((prev) => ({ ...prev, process_activity: event.target.value }))
                  }
                />
              </div>
              <div>
                <Label>Origen en fases del ciclo de vida</Label>
                <div className="space-y-2 rounded-2xl border border-emerald-100 p-3">
                  {lifecycleRows.map((phase) => (
                    <label key={phase.id} className="flex items-center gap-2 text-sm text-gray-600">
                      <Checkbox
                        checked={aspectLifecycleSelection.includes(phase.id)}
                        onCheckedChange={(checked) => {
                          setAspectLifecycleSelection((prev) =>
                            checked
                              ? [...prev, phase.id]
                              : prev.filter((id) => id !== phase.id),
                          );
                        }}
                      />
                      {phase.lifecycle_phase ?? "Fase"}
                    </label>
                  ))}
                  {!lifecycleRows.length && (
                    <p className="text-xs text-gray-500">No hay fases registradas.</p>
                  )}
                </div>
              </div>
              <div>
                <Label>Impacto normal</Label>
                <Textarea
                  value={aspectForm.impact_normal}
                  onChange={(event) => setAspectForm((prev) => ({ ...prev, impact_normal: event.target.value }))}
                />
              </div>
              <div>
                <Label>Impacto anormal</Label>
                <Textarea
                  value={aspectForm.impact_abnormal}
                  onChange={(event) =>
                    setAspectForm((prev) => ({ ...prev, impact_abnormal: event.target.value }))
                  }
                />
              </div>
              <div>
                <Label>Impacto en emergencia</Label>
                <Textarea
                  value={aspectForm.impact_emergency}
                  onChange={(event) =>
                    setAspectForm((prev) => ({ ...prev, impact_emergency: event.target.value }))
                  }
                />
              </div>
              <div>
                <Label>Medida de control operacional</Label>
                <Textarea
                  value={aspectForm.operational_control}
                  onChange={(event) =>
                    setAspectForm((prev) => ({ ...prev, operational_control: event.target.value }))
                  }
                />
              </div>
              <div>
                <Label>Área responsable de controles</Label>
                <Input
                  value={aspectForm.control_responsible_area}
                  onChange={(event) =>
                    setAspectForm((prev) => ({ ...prev, control_responsible_area: event.target.value }))
                  }
                />
              </div>
              <div>
                <Label>Plan de respuesta a emergencias (link)</Label>
                <Input
                  value={aspectForm.emergency_plan_link}
                  onChange={(event) =>
                    setAspectForm((prev) => ({ ...prev, emergency_plan_link: event.target.value }))
                  }
                />
              </div>
              <div>
                <Label>Instrucciones técnicas (link)</Label>
                <Input
                  value={aspectForm.technical_instructions_link}
                  onChange={(event) =>
                    setAspectForm((prev) => ({ ...prev, technical_instructions_link: event.target.value }))
                  }
                />
              </div>
            </div>
            <div className="grid gap-4 rounded-2xl border border-emerald-100 bg-emerald-50/50 p-4 md:grid-cols-5">
              <div>
                <Label className="text-xs">Magnitud (M)</Label>
                <Select
                  value={aspectForm.magnitude_m}
                  onValueChange={(value) => setAspectForm((prev) => ({ ...prev, magnitude_m: value }))}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="1" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="1">1</SelectItem>
                    <SelectItem value="2">2</SelectItem>
                    <SelectItem value="3">3</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label className="text-xs">FCI</Label>
                <Select
                  value={aspectForm.control_influence_fci}
                  onValueChange={(value) =>
                    setAspectForm((prev) => ({ ...prev, control_influence_fci: value }))
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="1" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="1">1</SelectItem>
                    <SelectItem value="2">2</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label className="text-xs">RL</Label>
                <Select
                  value={aspectForm.legal_requirement_rl}
                  onValueChange={(value) => setAspectForm((prev) => ({ ...prev, legal_requirement_rl: value }))}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="1" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="1">1</SelectItem>
                    <SelectItem value="2">2</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label className="text-xs">Frecuencia (F)</Label>
                <Select
                  value={aspectForm.frequency_f}
                  onValueChange={(value) => setAspectForm((prev) => ({ ...prev, frequency_f: value }))}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="1" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="1">1</SelectItem>
                    <SelectItem value="2">2</SelectItem>
                    <SelectItem value="3">3</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label className="text-xs">Extensión (E)</Label>
                <Select
                  value={aspectForm.extension_e}
                  onValueChange={(value) => setAspectForm((prev) => ({ ...prev, extension_e: value }))}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="1" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="1">1</SelectItem>
                    <SelectItem value="2">2</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
            <div className="flex items-center justify-between rounded-2xl border border-emerald-100 bg-white p-4">
              <div>
                <p className="text-sm text-gray-500">Valor esperado (VE)</p>
                <p className="text-2xl font-semibold text-gray-900">{computeExpectedValue(aspectForm)}</p>
              </div>
              <Badge
                className={
                  computeExpectedValue(aspectForm) >= 10
                    ? "bg-amber-100 text-amber-800"
                    : "bg-emerald-100 text-emerald-700"
                }
              >
                {computeExpectedValue(aspectForm) >= 10
                  ? significanceLabels.significant
                  : significanceLabels.not_significant}
              </Badge>
            </div>
            <div className="flex justify-end gap-2">
              <Button type="button" variant="secondary" onClick={() => setAspectDialogOpen(false)}>
                Cancelar
              </Button>
              <Button type="submit">Guardar</Button>
            </div>
          </form>
        </DialogContent>
      </Dialog>

      <Dialog open={linkDialogOpen} onOpenChange={setLinkDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Vincular aspectos a fase</DialogTitle>
          </DialogHeader>
          <form className="space-y-4" onSubmit={handleLinkSubmit}>
            <p className="text-sm text-gray-600">
              {linkLifecycle?.lifecycle_phase ?? "Fase seleccionada"}
            </p>
            <div className="space-y-2">
              {aspects.map((aspect) => (
                <label key={aspect.id} className="flex items-center gap-2 text-sm text-gray-600">
                  <Checkbox
                    checked={linkAspectSelection.includes(aspect.id)}
                    onCheckedChange={(checked) => {
                      setLinkAspectSelection((prev) =>
                        checked ? [...prev, aspect.id] : prev.filter((id) => id !== aspect.id),
                      );
                    }}
                  />
                  {aspect.aspect ?? "Aspecto"}
                </label>
              ))}
              {!aspects.length && <p className="text-xs text-gray-500">No hay aspectos disponibles.</p>}
            </div>
            <div className="flex justify-end gap-2">
              <Button type="button" variant="secondary" onClick={() => setLinkDialogOpen(false)}>
                Cancelar
              </Button>
              <Button type="submit">Guardar</Button>
            </div>
          </form>
        </DialogContent>
      </Dialog>
    </SGIDashboardLayout>
  );
}