"use client";

import { useCallback, useEffect, useMemo, useState, type FormEvent } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import Link from "next/link";
import { supabase } from "@/lib/supabase";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Textarea } from "@/components/ui/textarea";
import {
  Award,
  Eye,
  FileDown,
  Leaf,
  Pencil,
  Plus,
  RefreshCw,
  Search,
  ShieldCheck,
  Trash2,
} from "lucide-react";
import { toast } from "sonner";
import type { User } from "@supabase/supabase-js";

type Objective = {
  id: string;
  objective_type: string | null;
  policy_excerpt_id: string | null;
  policy_excerpt: string | null;
  objective_id: string | null;
  objective_title: string | null;
  objective_elements: string | null;
  indicator: string | null;
  target: string | null;
  target_indicator_value: number | null;
  current_indicator_value: number | null;
  target_date: string | null;
  responsible: string | null;
  review_frequency: string | null;
  action_plans: string | null;
  resources: string | null;
  status: string | null;
  countermeasures: string | null;
  last_updated_at: string | null;
  next_review_date: string | null;
};

type IndicatorRecord = {
  id: string;
  jan: number | null;
  feb: number | null;
  mar: number | null;
  apr: number | null;
  may: number | null;
  jun: number | null;
  jul: number | null;
  aug: number | null;
  sep: number | null;
  oct: number | null;
  nov: number | null;
  dec: number | null;
  annual_sum_target: number | null;
  annual_average_target: number | null;
  compliance: boolean | null;
};

type IndicatorLinkSource = {
  id: string;
  objective_id: string | null;
};

type ObjectiveIndicatorLink = {
  objective_id: string;
  indicator_id: string;
  weight: number | null;
  polarity: "higher_better" | "lower_better" | null;
  target_value: number | null;
  target_type: string | null;
};

type ObjectiveFormState = {
  objective_type: string;
  policy_excerpt_id: string;
  policy_excerpt: string;
  objective_id: string;
  objective_title: string;
  objective_elements: string;
  indicator: string;
  target: string;
  target_indicator_value: string;
  current_indicator_value: string;
  target_date: string;
  responsible: string;
  review_frequency: string;
  action_plans: string;
  resources: string;
  status: string;
  countermeasures: string;
  next_review_date: string;
};

const objectiveTypes = [
  { value: "environment", label: "Ambiental" },
  { value: "quality", label: "Calidad" },
  { value: "safety", label: "Seguridad" },
];

const categories = [
  {
    id: "environment",
    label: "Medio Ambiente",
    description: "Gestión ambiental y objetivos de sostenibilidad.",
    accent: "from-emerald-500 via-emerald-400 to-lime-400",
    icon: Leaf,
  },
  {
    id: "quality",
    label: "Calidad",
    description: "Metas de calidad y mejora continua.",
    accent: "from-sky-500 via-indigo-500 to-violet-500",
    icon: Award,
  },
  {
    id: "safety",
    label: "Seguridad",
    description: "Objetivos de seguridad y prevención.",
    accent: "from-rose-500 via-orange-500 to-amber-400",
    icon: ShieldCheck,
  },
];

const statusOptions = [
  "No iniciado",
  "En progreso",
  "Cumplido",
  "Retrasado",
  "Cancelado",
];

const reviewFrequencies = [
  "Mensual",
  "Trimestral",
  "Semestral",
  "Anual",
  "Eventual",
];

const monthOptions = [
  "jan",
  "feb",
  "mar",
  "apr",
  "may",
  "jun",
  "jul",
  "aug",
  "sep",
  "oct",
  "nov",
  "dec",
] as const;

const emptyFormState: ObjectiveFormState = {
  objective_type: "",
  policy_excerpt_id: "",
  policy_excerpt: "",
  objective_id: "",
  objective_title: "",
  objective_elements: "",
  indicator: "",
  target: "",
  target_indicator_value: "",
  current_indicator_value: "",
  target_date: "",
  responsible: "",
  review_frequency: "",
  action_plans: "",
  resources: "",
  status: "",
  countermeasures: "",
  next_review_date: "",
};

const actionSourceLabel = "Objetivos ambientales";

const buildActionPlanHref = (objective: Objective) => {
  const params = new URLSearchParams({
    source: actionSourceLabel,
    description: objective.objective_title ?? "",
    comments: objective.objective_elements ?? "",
    rootCause: objective.countermeasures ?? "",
    actionPlan: objective.action_plans ?? "",
    owner: objective.responsible ?? "",
    targetDate: objective.target_date ?? "",
  });
  return `/meiwa-core/meiwa-sgi/planning/action-plan?${params.toString()}`;
};

const parseNumber = (value: string) => {
  if (!value) return null;
  const parsed = Number(value);
  return Number.isNaN(parsed) ? null : parsed;
};

const formatDate = (value: string | null) =>
  value ? new Date(value).toLocaleDateString("es-MX") : "-";

const formatNumber = (value: number | null) => (value === null ? "-" : value.toString());

const computeAnnualSummary = (record: IndicatorRecord) => {
  const values = monthOptions
    .map((month) => record[month])
    .filter((value): value is number => value !== null && value !== undefined);
  if (!values.length) {
    return { sum: null, average: null };
  }
  const sum = values.reduce((total, current) => total + current, 0);
  const average = values.length ? sum / values.length : null;
  return { sum, average };
};

const formatAnnualTargetMatch = (
  summary: { sum: number | null; average: number | null },
  record: IndicatorRecord,
) => {
  if (
    summary.sum === null ||
    summary.average === null ||
    record.annual_sum_target == null ||
    record.annual_average_target == null
  ) {
    return null;
  }
  const averageRounded = Number(summary.average.toFixed(2));
  const sumMatches = summary.sum === record.annual_sum_target;
  const averageMatches = averageRounded === record.annual_average_target;
  return sumMatches && averageMatches;
};

const resolveIndicatorCompliance = (record: IndicatorRecord) => {
  if (record.compliance !== null) return record.compliance;
  const summary = computeAnnualSummary(record);
  return formatAnnualTargetMatch(summary, record);
};

const computeObjectiveComplianceRate = (
  links: ObjectiveIndicatorLink[],
  indicatorMap: Map<string, IndicatorRecord>,
) => {
  if (!links.length) return null;
  const compliantCount = links.reduce((total, link) => {
    const indicator = indicatorMap.get(link.indicator_id);
    if (!indicator) return total;
    const compliance = resolveIndicatorCompliance(indicator);
    return compliance ? total + 1 : total;
  }, 0);

  return (compliantCount / links.length) * 100;
};

const computeObjectiveStatus = (
  currentValue: number | null,
  targetValue: number | null,
  targetDate: string | null,
) => {
  if (currentValue !== null && targetValue !== null && currentValue >= targetValue) {
    return "Cumplido";
  }
  if (targetDate) {
    const target = new Date(targetDate);
    if (target < new Date()) {
      return "Retrasado";
    }
  }
  return "En progreso";
};

const getStatusBadge = (status: string | null) => {
  if (!status) return "bg-slate-100 text-slate-700";
  if (status === "Cumplido") return "bg-emerald-100 text-emerald-700";
  if (status === "En progreso") return "bg-sky-100 text-sky-700";
  if (status === "Retrasado") return "bg-rose-100 text-rose-700";
  if (status === "Cancelado") return "bg-slate-200 text-slate-600";
  return "bg-amber-100 text-amber-700";
};

const getObjectiveTypeLabel = (value: string | null) =>
  objectiveTypes.find((type) => type.value === value)?.label ?? "-";

const computeNextReviewDate = (frequency: string, baseDate: Date) => {
  const next = new Date(baseDate);
  switch (frequency) {
    case "Mensual":
      next.setMonth(next.getMonth() + 1);
      break;
    case "Trimestral":
      next.setMonth(next.getMonth() + 3);
      break;
    case "Semestral":
      next.setMonth(next.getMonth() + 6);
      break;
    case "Anual":
      next.setFullYear(next.getFullYear() + 1);
      break;
    default:
      return null;
  }
  return next.toISOString().split("T")[0];
};

const toCsvValue = (value: string | number | null) => {
  if (value === null || value === undefined) return "";
  const stringValue = `${value}`;
  if (stringValue.includes(",") || stringValue.includes("\n") || stringValue.includes('"')) {
    return `"${stringValue.replace(/"/g, '""')}"`;
  }
  return stringValue;
};

export default function EnvironmentalObjectivesPage() {
  const router = useRouter();
  const searchParams = useSearchParams();  
  const [user, setUser] = useState<User | null>(null);
  const [objectives, setObjectives] = useState<Objective[]>([]);
  const [loading, setLoading] = useState(true);
  const [search, setSearch] = useState("");
  const [activeCategory, setActiveCategory] = useState(categories[0].id);
  const [selectedObjectiveId, setSelectedObjectiveId] = useState<string | null>(null);
  const [filters, setFilters] = useState({
    status: "all",
    responsible: "all",
    reviewFrequency: "all",
    dateFrom: "",
    dateTo: "",
  });
  const [dialogOpen, setDialogOpen] = useState(false);
  const [detailOpen, setDetailOpen] = useState(false);
  const [editingObjective, setEditingObjective] = useState<Objective | null>(null);
  const [formState, setFormState] = useState<ObjectiveFormState>(emptyFormState);
  const [isSaving, setIsSaving] = useState(false);
  const [objectiveLinkCounts, setObjectiveLinkCounts] = useState<Record<string, number>>({});

  const ensureObjectiveIndicatorLinks = useCallback(async (items: Objective[]) => {
    if (!items.length) return;

    const objectivesByCode = new Map(
      items
        .filter((objective) => Boolean(objective.objective_id))
        .map((objective) => [objective.objective_id ?? "", objective.id]),
    );

    if (!objectivesByCode.size) return;

    const { data: indicators, error: indicatorsError } = await supabase
      .from("indicators_yearly")
      .select("id, objective_id")
      .not("objective_id", "is", null);

    if (indicatorsError || !indicators?.length) {
      if (indicatorsError) console.error(indicatorsError);
      return;
    }

    const indicatorsData = indicators as IndicatorLinkSource[];
    const candidateLinks = indicatorsData
      .map((indicator) => {
        const objectiveId = objectivesByCode.get(indicator.objective_id ?? "");
        if (!objectiveId) return null;
        return { objective_id: objectiveId, indicator_id: indicator.id };
      })
      .filter((link): link is { objective_id: string; indicator_id: string } => Boolean(link));

    if (!candidateLinks.length) return;

    const objectiveIds = Array.from(new Set(candidateLinks.map((link) => link.objective_id)));
    const { data: existingLinks, error: existingLinksError } = await supabase
      .from("objective_indicator_links")
      .select("objective_id, indicator_id")
      .in("objective_id", objectiveIds);

    if (existingLinksError) {
      console.error(existingLinksError);
      return;
    }

    const existingKeys = new Set(
      (existingLinks ?? []).map((link) => `${link.objective_id}:${link.indicator_id}`),
    );
    const newLinks = candidateLinks.filter(
      (link) => !existingKeys.has(`${link.objective_id}:${link.indicator_id}`),
    );

    if (!newLinks.length) return;

    const { error: insertError } = await supabase
      .from("objective_indicator_links")
      .insert(newLinks);

    if (insertError) {
      console.error(insertError);
    }
  }, []);

  const syncObjectiveStatusWithIndicators = useCallback(async (items: Objective[]) => {
    if (!items.length) return;

    const objectiveIds = items.map((objective) => objective.id);
    const { data: links, error: linksError } = await supabase
      .from("objective_indicator_links")
      .select("objective_id, indicator_id, weight, polarity, target_value, target_type")
      .in("objective_id", objectiveIds)
      .or("active.eq.true,active.is.null");

    if (linksError || !links?.length) {
      if (linksError) console.error(linksError);
      return;
    }

    const linkData = links as ObjectiveIndicatorLink[];
    const indicatorIds = Array.from(new Set(linkData.map((link) => link.indicator_id)));

    const { data: indicators, error: indicatorsError } = await supabase
      .from("indicators_yearly")
      .select(
        "id, jan, feb, mar, apr, may, jun, jul, aug, sep, oct, nov, dec, annual_sum_target, annual_average_target, compliance",
      )
      .in("id", indicatorIds);

    if (indicatorsError || !indicators?.length) {
      if (indicatorsError) console.error(indicatorsError);
      return;
    }

    const indicatorMap = new Map<string, IndicatorRecord>(
      indicators.map((indicator) => [indicator.id, indicator]),
    );
    const now = new Date().toISOString();
    const computedUpdates = new Map<
      string,
      { current_indicator_value: number | null; status: string; next_review_date: string | null; last_updated_at?: string }
    >();

    const updates = items
      .map((objective) => {
        const linksForObjective = linkData.filter((link) => link.objective_id === objective.id);
        if (!linksForObjective.length) return null;
        const complianceRate = computeObjectiveComplianceRate(linksForObjective, indicatorMap);
        const roundedProgress =
          complianceRate === null ? null : Math.round(complianceRate * 100) / 100;
        const status = computeObjectiveStatus(
          roundedProgress,
          objective.target_indicator_value ?? null,
          objective.target_date ?? null,
        );
        const nextReviewDate = objective.review_frequency
          ? computeNextReviewDate(objective.review_frequency, new Date())
          : null;

        const shouldUpdate =
          objective.current_indicator_value !== roundedProgress ||
          objective.status !== status ||
          objective.next_review_date !== nextReviewDate;

        const payload = {
          current_indicator_value: roundedProgress,
          status,
          next_review_date: nextReviewDate,
          ...(shouldUpdate ? { last_updated_at: now } : {}),
        };

        computedUpdates.set(objective.id, {
          current_indicator_value: roundedProgress,
          status,
          next_review_date: nextReviewDate,
          ...(shouldUpdate ? { last_updated_at: now } : {}),
        });

        if (!shouldUpdate) return null;
        return supabase.from("objectives").update(payload).eq("id", objective.id);
      })
      .filter(Boolean);

    if (updates.length) {
      const results = await Promise.all(updates);
      results.forEach((result) => {
        if (result?.error) {
          console.error(result.error);
        }
      });
    }

    if (computedUpdates.size) {
      setObjectives((prev) =>
        prev.map((objective) => {
          const update = computedUpdates.get(objective.id);
          return update ? { ...objective, ...update } : objective;
        }),
      );
    }
  }, []);
  
  const fetchObjectives = useCallback(async () => {
    setLoading(true);
    const { data: { session } } = await supabase.auth.getSession();

    if (!session?.user) {
      router.push("/");
      return;
    }

    setUser(session.user);

    const { data, error } = await supabase
      .from("objectives")
      .select("*")
      .order("last_updated_at", { ascending: false });

    if (error) {
      toast.error("No se pudieron cargar los objetivos.");
      console.error(error);
      setObjectives([]);
      setObjectiveLinkCounts({});
    } else {
      setObjectives(data ?? []);
      await ensureObjectiveIndicatorLinks(data ?? []);
      const objectiveIds = (data ?? []).map((objective) => objective.id);
      if (objectiveIds.length) {
        const { data: links, error: linksError } = await supabase
          .from("objective_indicator_links")
          .select("objective_id")
          .in("objective_id", objectiveIds)
          .or("active.eq.true,active.is.null");

        if (linksError) {
          console.error(linksError);
          setObjectiveLinkCounts({});
        } else {
          const counts = (links ?? []).reduce<Record<string, number>>((acc, link) => {
            acc[link.objective_id] = (acc[link.objective_id] ?? 0) + 1;
            return acc;
          }, {});
          setObjectiveLinkCounts(counts);
        }
      } else {
        setObjectiveLinkCounts({});
      }

      await syncObjectiveStatusWithIndicators(data ?? []);      
    }

    setLoading(false);
  }, [router, syncObjectiveStatusWithIndicators]);

  useEffect(() => {
    fetchObjectives();
  }, [fetchObjectives]);

  useEffect(() => {
    const objectiveIdFilter = searchParams.get("objectiveId");
    if (objectiveIdFilter) {
      setSearch(objectiveIdFilter);
    }
  }, [searchParams]);

  const selectedObjective = useMemo(
    () => objectives.find((objective) => objective.id === selectedObjectiveId) ?? null,
    [objectives, selectedObjectiveId]
  );

  const responsibleOptions = useMemo(() => {
    const values = new Set<string>();
    objectives.forEach((objective) => {
      if (objective.responsible) values.add(objective.responsible);
    });
    return Array.from(values);
  }, [objectives]);

  const reviewFrequencyOptions = useMemo(() => {
    const values = new Set<string>();
    objectives.forEach((objective) => {
      if (objective.review_frequency) values.add(objective.review_frequency);
    });
    return Array.from(values);
  }, [objectives]);

  const statusOptionsList = useMemo(() => {
    const values = new Set<string>();
    statusOptions.forEach((status) => values.add(status));
    objectives.forEach((objective) => {
      if (objective.status) values.add(objective.status);
    });
    return Array.from(values);
  }, [objectives]);

  const baseFilteredObjectives = useMemo(() => {
    const term = search.trim().toLowerCase();
    return objectives.filter((objective) => {
      const matchesTerm =
        !term ||
        (objective.objective_id ?? "").toLowerCase().includes(term) ||
        (objective.policy_excerpt_id ?? "").toLowerCase().includes(term) ||
        (objective.objective_title ?? "").toLowerCase().includes(term) ||
        (objective.indicator ?? "").toLowerCase().includes(term) ||
        (objective.policy_excerpt ?? "").toLowerCase().includes(term);

      const matchesStatus = filters.status === "all" || objective.status === filters.status;
      const matchesResponsible =
        filters.responsible === "all" || objective.responsible === filters.responsible;
      const matchesFrequency =
        filters.reviewFrequency === "all" ||
        objective.review_frequency === filters.reviewFrequency;

      const targetDate = objective.target_date ? new Date(objective.target_date) : null;
      const fromDate = filters.dateFrom ? new Date(filters.dateFrom) : null;
      const toDate = filters.dateTo ? new Date(filters.dateTo) : null;

      const matchesFrom = !fromDate || (targetDate && targetDate >= fromDate);
      const matchesTo = !toDate || (targetDate && targetDate <= toDate);

      return (
        matchesTerm &&
        matchesStatus &&
        matchesResponsible &&
        matchesFrequency &&
        matchesFrom &&
        matchesTo
      );
    });
  }, [filters, objectives, search]);

  const filteredObjectives = useMemo(
    () => baseFilteredObjectives.filter((objective) => objective.objective_type === activeCategory),
    [activeCategory, baseFilteredObjectives]
  );

  const kpiData = useMemo(() => {
    const total = filteredObjectives.length;
    const statusCounts: Record<string, number> = {};
    let delayedOrOverdue = 0;
    let dueSoon = 0;
    const now = new Date();
    const soonLimit = new Date();
    soonLimit.setDate(soonLimit.getDate() + 30);

    filteredObjectives.forEach((objective) => {
      if (objective.status) statusCounts[objective.status] = (statusCounts[objective.status] ?? 0) + 1;
      const targetDate = objective.target_date ? new Date(objective.target_date) : null;
      const isCompleted = objective.status === "Cumplido";
      const isOverdue = Boolean(targetDate && !isCompleted && targetDate < now);
      const isDelayedStatus = objective.status === "Retrasado";
      if (isDelayedStatus || isOverdue) delayedOrOverdue += 1;
      if (targetDate && !isCompleted && targetDate >= now && targetDate <= soonLimit) dueSoon += 1;
    });

    const completed = statusCounts["Cumplido"] ?? 0;
    const inProgress = statusCounts["En progreso"] ?? 0;
    const complianceRate = total > 0 ? Math.round((completed / total) * 100) : 0;

    return {
      total,
      completed,
      inProgress,
      delayedOrOverdue,
      complianceRate,
      dueSoon,
    };
  }, [filteredObjectives]);

  const openNewObjective = () => {
    setEditingObjective(null);
    setFormState(emptyFormState);
    setDialogOpen(true);
  };

  const openEditObjective = () => {
    if (!selectedObjective) return;
    setEditingObjective(selectedObjective);
    setFormState({
      objective_type: selectedObjective.objective_type ?? "",
      policy_excerpt_id: selectedObjective.policy_excerpt_id ?? "",
      policy_excerpt: selectedObjective.policy_excerpt ?? "",
      objective_id: selectedObjective.objective_id ?? "",
      objective_title: selectedObjective.objective_title ?? "",
      objective_elements: selectedObjective.objective_elements ?? "",
      indicator: selectedObjective.indicator ?? "",
      target: selectedObjective.target ?? "",
      target_indicator_value: selectedObjective.target_indicator_value?.toString() ?? "",
      current_indicator_value: selectedObjective.current_indicator_value?.toString() ?? "",
      target_date: selectedObjective.target_date ?? "",
      responsible: selectedObjective.responsible ?? "",
      review_frequency: selectedObjective.review_frequency ?? "",
      action_plans: selectedObjective.action_plans ?? "",
      resources: selectedObjective.resources ?? "",
      status: selectedObjective.status ?? "",
      countermeasures: selectedObjective.countermeasures ?? "",
      next_review_date: selectedObjective.next_review_date ?? "",
    });
    setDialogOpen(true);
  };

  const handleSave = async (event: FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    setIsSaving(true);
    const now = new Date();
    const nextReviewDate = formState.review_frequency
      ? computeNextReviewDate(formState.review_frequency, now)
      : null;
    const payload = {
      objective_type: formState.objective_type || null,
      policy_excerpt_id: formState.policy_excerpt_id || null,
      policy_excerpt: formState.policy_excerpt || null,
      objective_id: formState.objective_id || null,
      objective_title: formState.objective_title || null,
      objective_elements: formState.objective_elements || null,
      indicator: formState.indicator || null,
      target: formState.target || null,
      target_indicator_value: parseNumber(formState.target_indicator_value),
      current_indicator_value: parseNumber(formState.current_indicator_value),
      target_date: formState.target_date || null,
      responsible: formState.responsible || null,
      review_frequency: formState.review_frequency || null,
      action_plans: formState.action_plans || null,
      resources: formState.resources || null,
      status: formState.status || null,
      countermeasures: formState.countermeasures || null,
      last_updated_at: now.toISOString(),
      next_review_date: nextReviewDate,
    };

    const { data, error } = editingObjective
      ? await supabase
          .from("objectives")
          .update(payload)
          .eq("id", editingObjective.id)
          .select()
          .single()
      : await supabase
          .from("objectives")
          .insert(payload)
          .select()
          .single();

    if (error) {
      toast.error("No se pudo guardar el objetivo.");
      console.error(error);
    } else if (data) {
      toast.success(editingObjective ? "Objetivo actualizado." : "Objetivo creado.");
      setObjectives((prev) => {
        if (editingObjective) {
          return prev.map((item) => (item.id === data.id ? data : item));
        }
        return [data, ...prev];
      });
      setDialogOpen(false);
      setEditingObjective(null);
      setFormState(emptyFormState);
      setSelectedObjectiveId(data.id);
    }

    setIsSaving(false);
  };

  const handleDelete = async () => {
    if (!selectedObjective) return;
    const confirmed = window.confirm("¿Seguro que deseas eliminar este objetivo?");
    if (!confirmed) return;

    const { error } = await supabase.from("objectives").delete().eq("id", selectedObjective.id);

    if (error) {
      toast.error("No se pudo eliminar el objetivo.");
      console.error(error);
      return;
    }

    toast.success("Objetivo eliminado.");
    setObjectives((prev) => prev.filter((item) => item.id !== selectedObjective.id));
    setSelectedObjectiveId(null);
  };

  const exportToCsv = () => {
    if (filteredObjectives.length === 0) {
      toast.info("No hay datos para exportar.");
      return;
    }
    const headers = [
      "ID Extracto de Política",
      "Extracto de Política",
      "ID Objetivo",
      "Objetivo",
      "Elementos del objetivo",
      "Indicador",
      "Meta",
      "Indicador Meta",
      "Indicador Actual",
      "Fecha objetivo",
      "Responsable",
      "Frecuencia de revisión",
      "Planes de acción",
      "Recursos",
      "Estado",
      "Contramedidas",
      "Tipo de objetivo",
      "Última actualización",
      "Siguiente revisión",
    ];

    const rows = filteredObjectives.map((objective) => [
      objective.policy_excerpt_id,
      objective.policy_excerpt,
      objective.objective_id,
      objective.objective_title,
      objective.objective_elements,
      objective.indicator,
      objective.target,
      objective.target_indicator_value,
      objective.current_indicator_value,
      objective.target_date,
      objective.responsible,
      objective.review_frequency,
      objective.action_plans,
      objective.resources,
      objective.status,
      objective.countermeasures,
      getObjectiveTypeLabel(objective.objective_type),
      objective.last_updated_at,
      objective.next_review_date,
    ]);

    const csvContent = [
      headers.map(toCsvValue).join(","),
      ...rows.map((row) => row.map(toCsvValue).join(",")),
    ].join("\n");

    const blob = new Blob(["\ufeff", csvContent], { type: "text/csv;charset=utf-8;" });
    const url = URL.createObjectURL(blob);
    const link = document.createElement("a");
    link.href = url;
    link.download = "objetivos-ambientales.csv";
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);
  };

  const activeCategoryData = categories.find((category) => category.id === activeCategory);
  const selectedObjectiveIndicatorCount = selectedObjective
    ? objectiveLinkCounts[selectedObjective.id] ?? 0
    : 0;

  const handleViewIndicators = () => {
    if (!selectedObjective?.objective_id) return;
    router.push(
      `/meiwa-core/meiwa-sgi/planning/environmental-indicators?objectiveId=${encodeURIComponent(
        selectedObjective.objective_id,
      )}`,
    );
  };

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-8 bg-gradient-to-br from-slate-50 via-white to-emerald-50 pb-20">
        <section className="rounded-3xl border border-emerald-100 bg-white p-8 shadow-sm">
          <div className="space-y-2">
            <p className="text-xs font-semibold uppercase tracking-[0.4em] text-emerald-600">
              Objetivos SGI
            </p>
            <h1 className="text-3xl font-bold text-gray-900 sm:text-4xl">
              Objetivos ambientales, calidad y seguridad
            </h1>
            <p className="text-sm text-gray-600 sm:text-base">
              Centraliza los objetivos vinculados a políticas, indicadores y planes de acción.
            </p>
          </div>
        </section>

        <section className="flex flex-wrap gap-3">
          {categories.map((category) => {
            const Icon = category.icon;
            const isActive = activeCategory === category.id;

            return (
              <button
                key={category.id}
                type="button"
                onClick={() => setActiveCategory(category.id)}
                className={`flex items-center gap-2 rounded-full border px-4 py-2 text-sm font-semibold transition-all duration-200 ${
                  isActive
                    ? "border-transparent bg-gradient-to-r from-emerald-500 to-lime-500 text-white shadow-lg"
                    : "border-gray-200 bg-white text-gray-600 hover:border-emerald-200 hover:text-emerald-700"
                }`}
              >
                <Icon className={`h-4 w-4 ${isActive ? "text-white" : "text-emerald-500"}`} />
                {category.label}
              </button>
            );
          })}
        </section>

        <section className="flex flex-wrap gap-3 rounded-3xl border border-emerald-100 bg-white p-6 shadow-sm">
          <Button
            type="button"
            onClick={openNewObjective}
            className="rounded-full bg-gradient-to-r from-emerald-500 to-lime-500 text-white shadow-lg"
          >
            <Plus className="mr-2 h-4 w-4" /> Nuevo objetivo
          </Button>
          <Button
            type="button"
            onClick={openEditObjective}
            disabled={!selectedObjective}
            className="rounded-full border border-gray-200 bg-white text-gray-600 hover:border-emerald-200 hover:text-emerald-700"
            variant="outline"
          >
            <Pencil className="mr-2 h-4 w-4" /> Editar
          </Button>
          <Button
            type="button"
            onClick={handleDelete}
            disabled={!selectedObjective}
            className="rounded-full border border-gray-200 bg-white text-gray-600 hover:border-rose-200 hover:text-rose-600"
            variant="outline"
          >
            <Trash2 className="mr-2 h-4 w-4" /> Eliminar
          </Button>
          <Button
            type="button"
            onClick={() => setDetailOpen(true)}
            disabled={!selectedObjective}
            className="rounded-full border border-gray-200 bg-white text-gray-600 hover:border-emerald-200 hover:text-emerald-700"
            variant="outline"
          >
            <Eye className="mr-2 h-4 w-4" /> Ver detalle
          </Button>
          <Button
            type="button"
            onClick={handleViewIndicators}
            disabled={!selectedObjective?.objective_id}
            className="rounded-full border border-gray-200 bg-white text-gray-600 hover:border-emerald-200 hover:text-emerald-700"
            variant="outline"
          >
            Ver indicadores
            <Badge className="ml-2 bg-emerald-100 text-emerald-700">
              {selectedObjectiveIndicatorCount}
            </Badge>
          </Button>
          <Button
            type="button"
            onClick={exportToCsv}
            className="rounded-full border border-gray-200 bg-white text-gray-600 hover:border-emerald-200 hover:text-emerald-700"
            variant="outline"
          >
            <FileDown className="mr-2 h-4 w-4" /> Exportar
          </Button>
          <Button
            type="button"
            onClick={fetchObjectives}
            className="rounded-full border border-gray-200 bg-white text-gray-600 hover:border-emerald-200 hover:text-emerald-700"
            variant="outline"
          >
            <RefreshCw className="mr-2 h-4 w-4" /> Actualizar
          </Button>
        </section>

        <section className="grid gap-4 md:grid-cols-2 xl:grid-cols-6">
          <Card className="border-emerald-100 shadow-sm">
            <CardHeader>
              <CardTitle className="text-sm text-gray-500">Total objetivos</CardTitle>
            </CardHeader>
            <CardContent className="text-3xl font-semibold text-gray-900">
              {kpiData.total}
            </CardContent>
          </Card>
          <Card className="border-emerald-100 shadow-sm">
            <CardHeader>
              <CardTitle className="text-sm text-gray-500">Cumplidos</CardTitle>
            </CardHeader>
            <CardContent className="text-3xl font-semibold text-emerald-600">
              {kpiData.completed}
            </CardContent>
          </Card>
          <Card className="border-emerald-100 shadow-sm">
            <CardHeader>
              <CardTitle className="text-sm text-gray-500">En progreso</CardTitle>
            </CardHeader>
            <CardContent className="text-3xl font-semibold text-sky-600">
              {kpiData.inProgress}
            </CardContent>
          </Card>
          <Card className="border-emerald-100 shadow-sm">
            <CardHeader>
              <CardTitle className="text-sm text-gray-500">Retrasados / fuera de fecha</CardTitle>
            </CardHeader>
            <CardContent className="text-3xl font-semibold text-rose-600">
              {kpiData.delayedOrOverdue}
            </CardContent>
          </Card>
          <Card className="border-emerald-100 shadow-sm">
            <CardHeader>
              <CardTitle className="text-sm text-gray-500">Próximos 30 días</CardTitle>
            </CardHeader>
            <CardContent className="text-3xl font-semibold text-amber-600">
              {kpiData.dueSoon}
            </CardContent>
          </Card>
          <Card className="border-emerald-100 shadow-sm">
            <CardHeader>
              <CardTitle className="text-sm text-gray-500">% cumplimiento</CardTitle>
            </CardHeader>
            <CardContent className="text-3xl font-semibold text-emerald-600">
              {kpiData.complianceRate}%
            </CardContent>
          </Card>
        </section>

        <Card className="border-emerald-100 shadow-sm">
          <CardHeader className="flex flex-col gap-4">
            <div className="flex flex-col gap-2 sm:flex-row sm:items-center sm:justify-between">
              <div className="space-y-1">
                <CardTitle className="text-lg text-gray-900">Registro de objetivos</CardTitle>
                <p className="text-sm text-gray-500">
                  {activeCategoryData?.label ?? "Categoría"} ·{" "}
                  {activeCategoryData?.description ?? ""}
                </p>
              </div>
              <div className="relative w-full sm:w-72">
                <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-gray-400" />
                <Input
                  value={search}
                  onChange={(event) => setSearch(event.target.value)}
                  placeholder="Buscar ID de objetivo, ID de política, indicador o título"
                  className="pl-9"
                />
              </div>
            </div>

            <div className="grid gap-4 md:grid-cols-2 xl:grid-cols-5">
              <div className="space-y-2">
                <Label>Estado</Label>
                <Select
                  value={filters.status}
                  onValueChange={(value) => setFilters((prev) => ({ ...prev, status: value }))}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Todos" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    {statusOptionsList.map((status) => (
                      <SelectItem key={status} value={status}>
                        {status}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label>Responsable</Label>
                <Select
                  value={filters.responsible}
                  onValueChange={(value) => setFilters((prev) => ({ ...prev, responsible: value }))}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Todos" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    {responsibleOptions.map((responsible) => (
                      <SelectItem key={responsible} value={responsible}>
                        {responsible}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label>Frecuencia revisión</Label>
                <Select
                  value={filters.reviewFrequency}
                  onValueChange={(value) =>
                    setFilters((prev) => ({ ...prev, reviewFrequency: value }))
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Todas" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todas</SelectItem>
                    {reviewFrequencies.map((frequency) => (
                      <SelectItem key={frequency} value={frequency}>
                        {frequency}
                      </SelectItem>
                    ))}
                    {reviewFrequencyOptions
                      .filter((frequency) => !reviewFrequencies.includes(frequency))
                      .map((frequency) => (
                        <SelectItem key={frequency} value={frequency}>
                          {frequency}
                        </SelectItem>
                      ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <Label>Fecha objetivo (desde)</Label>
                <Input
                  type="date"
                  value={filters.dateFrom}
                  onChange={(event) =>
                    setFilters((prev) => ({ ...prev, dateFrom: event.target.value }))
                  }
                />
              </div>

              <div className="space-y-2">
                <Label>Fecha objetivo (hasta)</Label>
                <Input
                  type="date"
                  value={filters.dateTo}
                  onChange={(event) =>
                    setFilters((prev) => ({ ...prev, dateTo: event.target.value }))
                  }
                />
              </div>
            </div>
          </CardHeader>
          <CardContent>
            {loading ? (
              <div className="py-12 text-center text-sm text-gray-500">Cargando objetivos...</div>
            ) : filteredObjectives.length === 0 ? (
              <div className="py-12 text-center text-sm text-gray-500">No hay objetivos registrados.</div>
            ) : (
              <div className="overflow-x-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Tipo</TableHead>
                      <TableHead>ID Extracto</TableHead>
                      <TableHead>Extracto de política</TableHead>
                      <TableHead>ID objetivo</TableHead>
                      <TableHead>Objetivo</TableHead>
                      <TableHead>Elementos</TableHead>
                      <TableHead>Indicador</TableHead>
                      <TableHead>Meta</TableHead>
                      <TableHead>Indicador meta</TableHead>
                      <TableHead>Indicador actual</TableHead>
                      <TableHead>Indicadores ligados</TableHead>
                      <TableHead>Fecha objetivo</TableHead>
                      <TableHead>Responsable</TableHead>
                      <TableHead>Frecuencia</TableHead>
                      <TableHead>Planes de acción</TableHead>
                      <TableHead>Recursos</TableHead>
                      <TableHead>Estado</TableHead>
                      <TableHead>Contramedidas</TableHead>
                      <TableHead>Última actualización</TableHead>
                      <TableHead>Siguiente revisión</TableHead>
                      <TableHead>Crear acción</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {filteredObjectives.map((objective) => {
                      const isSelected = objective.id === selectedObjectiveId;
                      const targetValue = objective.target_indicator_value ?? 0;
                      const currentValue = objective.current_indicator_value ?? 0;
                      const canComputeProgress = targetValue > 0;
                      const progress = canComputeProgress
                        ? Math.min(100, Math.round((currentValue / targetValue) * 100))
                        : null;

                      return (
                        <TableRow
                          key={objective.id}
                          onClick={() => setSelectedObjectiveId(objective.id)}
                          className={
                            isSelected
                              ? "bg-emerald-50/70"
                              : "hover:bg-emerald-50/40"
                          }
                        >
                          <TableCell>{getObjectiveTypeLabel(objective.objective_type)}</TableCell>
                          <TableCell>{objective.policy_excerpt_id ?? "-"}</TableCell>
                          <TableCell className="max-w-[220px]">{objective.policy_excerpt ?? "-"}</TableCell>
                          <TableCell>{objective.objective_id ?? "-"}</TableCell>
                          <TableCell className="max-w-[260px] font-semibold text-gray-900">
                            {objective.objective_title ?? "-"}
                          </TableCell>
                          <TableCell className="max-w-[240px]">{objective.objective_elements ?? "-"}</TableCell>
                          <TableCell className="max-w-[200px]">{objective.indicator ?? "-"}</TableCell>
                          <TableCell className="max-w-[160px]">{objective.target ?? "-"}</TableCell>
                          <TableCell>{formatNumber(objective.target_indicator_value)}</TableCell>
                          <TableCell>
                            <div className="space-y-1">
                              <span>{formatNumber(objective.current_indicator_value)}</span>
                              {progress !== null ? (
                                <div className="h-2 w-24 rounded-full bg-emerald-100">
                                  <div
                                    className="h-2 rounded-full bg-emerald-500"
                                    style={{ width: `${progress}%` }}
                                  />
                                </div>
                              ) : null}
                              {progress !== null ? (
                                <span className="text-xs text-emerald-700">{progress}%</span>
                              ) : null}
                            </div>
                          </TableCell>
                          <TableCell>
                            <Badge className="bg-emerald-50 text-emerald-700">
                              {objectiveLinkCounts[objective.id] ?? 0}
                            </Badge>
                          </TableCell>
                          <TableCell>{formatDate(objective.target_date)}</TableCell>
                          <TableCell>{objective.responsible ?? "-"}</TableCell>
                          <TableCell>{objective.review_frequency ?? "-"}</TableCell>
                          <TableCell className="max-w-[200px]">{objective.action_plans ?? "-"}</TableCell>
                          <TableCell className="max-w-[200px]">{objective.resources ?? "-"}</TableCell>
                          <TableCell>
                            <Badge className={getStatusBadge(objective.status)}>
                              {objective.status ?? "Sin estado"}
                            </Badge>
                          </TableCell>
                          <TableCell className="max-w-[200px]">{objective.countermeasures ?? "-"}</TableCell>
                          <TableCell>{formatDate(objective.last_updated_at)}</TableCell>
                          <TableCell>{formatDate(objective.next_review_date)}</TableCell>
                          <TableCell>
                            <Button
                              asChild
                              size="sm"
                              className="bg-sky-500 text-white hover:bg-sky-600"
                            >
                              <Link href={buildActionPlanHref(objective)}>
                                Crear acción
                              </Link>
                            </Button>
                          </TableCell>
                        </TableRow>
                      );
                    })}
                  </TableBody>
                </Table>
              </div>
            )}
          </CardContent>
        </Card>
      </div>

      <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
        <DialogContent className="max-h-[90vh] overflow-y-auto sm:max-w-4xl">
          <DialogHeader>
            <DialogTitle>
              {editingObjective ? "Editar objetivo" : "Nuevo objetivo"}
            </DialogTitle>
          </DialogHeader>

          <form onSubmit={handleSave} className="space-y-6">
            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <Label>Tipo de objetivo</Label>
                <Select
                  value={formState.objective_type}
                  onValueChange={(value) =>
                    setFormState((prev) => ({ ...prev, objective_type: value }))
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona" />
                  </SelectTrigger>
                  <SelectContent>
                    {objectiveTypes.map((type) => (
                      <SelectItem key={type.value} value={type.value}>
                        {type.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>ID Extracto de política</Label>
                <Input
                  value={formState.policy_excerpt_id}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, policy_excerpt_id: event.target.value }))
                  }
                  placeholder="POL-001"
                />
              </div>
              <div className="space-y-2 md:col-span-2">
                <Label>Extracto de política</Label>
                <Textarea
                  value={formState.policy_excerpt}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, policy_excerpt: event.target.value }))
                  }
                  placeholder="Resumen de política vinculada"
                />
              </div>
              <div className="space-y-2">
                <Label>ID Objetivo</Label>
                <Input
                  value={formState.objective_id}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, objective_id: event.target.value }))
                  }
                  placeholder="OBJ-001"
                />
              </div>
              <div className="space-y-2">
                <Label>Objetivo</Label>
                <Input
                  value={formState.objective_title}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, objective_title: event.target.value }))
                  }
                  placeholder="Reducir emisiones en 10%"
                />
              </div>
              <div className="space-y-2 md:col-span-2">
                <Label>Elementos del objetivo</Label>
                <Textarea
                  value={formState.objective_elements}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, objective_elements: event.target.value }))
                  }
                  placeholder="Procesos, áreas o factores clave"
                />
              </div>
              <div className="space-y-2">
                <Label>Indicador</Label>
                <Input
                  value={formState.indicator}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, indicator: event.target.value }))
                  }
                  placeholder="Kg CO₂ / unidad"
                />
              </div>
              <div className="space-y-2">
                <Label>Meta</Label>
                <Input
                  value={formState.target}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, target: event.target.value }))
                  }
                  placeholder="Meta cualitativa o cuantitativa"
                />
              </div>
              <div className="space-y-2">
                <Label>Indicador meta</Label>
                <Input
                  type="number"
                  value={formState.target_indicator_value}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, target_indicator_value: event.target.value }))
                  }
                  placeholder="100"
                />
              </div>
              <div className="space-y-2">
                <Label>Indicador actual</Label>
                <Input
                  type="number"
                  value={formState.current_indicator_value}
                  readOnly
                  className="bg-slate-50"
                />
              </div>
              <div className="space-y-2">
                <Label>Fecha objetivo</Label>
                <Input
                  type="date"
                  value={formState.target_date}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, target_date: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label>Responsable</Label>
                <Input
                  value={formState.responsible}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, responsible: event.target.value }))
                  }
                  placeholder="Nombre o área"
                />
              </div>
              <div className="space-y-2">
                <Label>Frecuencia de revisión</Label>
                <Select
                  value={formState.review_frequency}
                  onValueChange={(value) =>
                    setFormState((prev) => ({
                      ...prev,
                      review_frequency: value,
                      next_review_date: value
                        ? computeNextReviewDate(value, new Date()) ?? ""
                        : "",
                    }))
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona" />
                  </SelectTrigger>
                  <SelectContent>
                    {reviewFrequencies.map((frequency) => (
                      <SelectItem key={frequency} value={frequency}>
                        {frequency}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Estado</Label>
                <Select
                  value={formState.status}
                  onValueChange={(value) =>
                    setFormState((prev) => ({ ...prev, status: value }))
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona" />
                  </SelectTrigger>
                  <SelectContent>
                    {statusOptions.map((status) => (
                      <SelectItem key={status} value={status}>
                        {status}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Recursos</Label>
                <Input
                  value={formState.resources}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, resources: event.target.value }))
                  }
                  placeholder="Presupuesto, equipos"
                />
              </div>
              <div className="space-y-2">
                <Label>Planes de acción</Label>
                <Textarea
                  value={formState.action_plans}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, action_plans: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label>Contramedidas</Label>
                <Textarea
                  value={formState.countermeasures}
                  onChange={(event) =>
                    setFormState((prev) => ({ ...prev, countermeasures: event.target.value }))
                  }
                />
              </div>
              <div className="space-y-2">
                <Label>Siguiente revisión</Label>
                <Input
                  type="date"
                  value={formState.next_review_date}
                  readOnly
                  className="bg-slate-50"
                />
              </div>
            </div>

            <div className="flex flex-wrap justify-end gap-3">
              <Button
                type="button"
                variant="outline"
                onClick={() => setDialogOpen(false)}
              >
                Cancelar
              </Button>
              <Button type="submit" disabled={isSaving}>
                {isSaving ? "Guardando..." : "Guardar objetivo"}
              </Button>
            </div>
          </form>
        </DialogContent>
      </Dialog>

      <Dialog open={detailOpen} onOpenChange={setDetailOpen}>
        <DialogContent className="max-h-[90vh] overflow-y-auto sm:max-w-3xl">
          <DialogHeader>
            <DialogTitle>Detalle del objetivo</DialogTitle>
          </DialogHeader>
          {selectedObjective ? (
            <div className="grid gap-4 text-sm text-gray-700">
              <div className="grid gap-4 md:grid-cols-2">
                <div>
                  <p className="text-xs uppercase text-gray-400">Tipo</p>
                  <p className="font-semibold text-gray-900">
                    {getObjectiveTypeLabel(selectedObjective.objective_type)}
                  </p>
                </div>
                <div>
                  <p className="text-xs uppercase text-gray-400">Estado</p>
                  <Badge className={getStatusBadge(selectedObjective.status)}>
                    {selectedObjective.status ?? "Sin estado"}
                  </Badge>
                </div>
                <div>
                  <p className="text-xs uppercase text-gray-400">ID Objetivo</p>
                  <p className="font-semibold text-gray-900">
                    {selectedObjective.objective_id ?? "-"}
                  </p>
                </div>
                <div>
                  <p className="text-xs uppercase text-gray-400">Responsable</p>
                  <p className="font-semibold text-gray-900">
                    {selectedObjective.responsible ?? "-"}
                  </p>
                </div>
                <div>
                  <p className="text-xs uppercase text-gray-400">Indicadores ligados</p>
                  <p className="font-semibold text-gray-900">
                    {objectiveLinkCounts[selectedObjective.id] ?? 0}
                  </p>
                </div>
              </div>

              <div>
                <p className="text-xs uppercase text-gray-400">Objetivo</p>
                <p className="font-semibold text-gray-900">
                  {selectedObjective.objective_title ?? "-"}
                </p>
              </div>

              <div className="rounded-xl border border-emerald-100 bg-emerald-50 p-4">
                <p className="text-xs uppercase text-emerald-700">Avance</p>
                <div className="mt-3 grid gap-4 sm:grid-cols-3">
                  <div>
                    <p className="text-xs uppercase text-emerald-600">Indicador meta</p>
                    <p className="font-semibold text-emerald-900">
                      {formatNumber(selectedObjective.target_indicator_value)}
                    </p>
                  </div>
                  <div>
                    <p className="text-xs uppercase text-emerald-600">Indicador actual</p>
                    <p className="font-semibold text-emerald-900">
                      {formatNumber(selectedObjective.current_indicator_value)}
                    </p>
                  </div>
                  <div>
                    <p className="text-xs uppercase text-emerald-600">Estado</p>
                    <Badge className={getStatusBadge(selectedObjective.status)}>
                      {selectedObjective.status ?? "Sin estado"}
                    </Badge>
                  </div>
                </div>
              </div>

              <div>
                <p className="text-xs uppercase text-gray-400">Extracto de política</p>
                <p>{selectedObjective.policy_excerpt ?? "-"}</p>
              </div>

              <div className="grid gap-4 md:grid-cols-2">
                <div>
                  <p className="text-xs uppercase text-gray-400">Indicador</p>
                  <p>{selectedObjective.indicator ?? "-"}</p>
                </div>
                <div>
                  <p className="text-xs uppercase text-gray-400">Meta</p>
                  <p>{selectedObjective.target ?? "-"}</p>
                </div>
                <div>
                  <p className="text-xs uppercase text-gray-400">Indicador meta</p>
                  <p>{formatNumber(selectedObjective.target_indicator_value)}</p>
                </div>
                <div>
                  <p className="text-xs uppercase text-gray-400">Indicador actual</p>
                  <p>{formatNumber(selectedObjective.current_indicator_value)}</p>
                </div>
                <div>
                  <p className="text-xs uppercase text-gray-400">Fecha objetivo</p>
                  <p>{formatDate(selectedObjective.target_date)}</p>
                </div>
                <div>
                  <p className="text-xs uppercase text-gray-400">Frecuencia revisión</p>
                  <p>{selectedObjective.review_frequency ?? "-"}</p>
                </div>
              </div>

              <div>
                <p className="text-xs uppercase text-gray-400">Planes de acción</p>
                <p>{selectedObjective.action_plans ?? "-"}</p>
              </div>
              <div>
                <p className="text-xs uppercase text-gray-400">Contramedidas</p>
                <p>{selectedObjective.countermeasures ?? "-"}</p>
              </div>
              <div>
                <p className="text-xs uppercase text-gray-400">Recursos</p>
                <p>{selectedObjective.resources ?? "-"}</p>
              </div>
              <div className="grid gap-4 md:grid-cols-2">
                <div>
                  <p className="text-xs uppercase text-gray-400">Última actualización</p>
                  <p>{formatDate(selectedObjective.last_updated_at)}</p>
                </div>
                <div>
                  <p className="text-xs uppercase text-gray-400">Siguiente revisión</p>
                  <p>{formatDate(selectedObjective.next_review_date)}</p>
                </div>
              </div>
            </div>
          ) : (
            <p className="text-sm text-gray-500">Selecciona un objetivo para ver el detalle.</p>
          )}
        </DialogContent>
      </Dialog>
    </SGIDashboardLayout>
  );
}