"use client";

import Link from "next/link";
import { useState } from "react";
import {
  ArrowUpRight,
  ClipboardList,
  FileText,
  Leaf,
  LineChart,  
  Scale,
  ShieldCheck,
  Target,
  Trophy,  
  Award,
} from "lucide-react";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";

const categories = [
  {
    id: "medio-ambiente",
    label: "Medio Ambiente",
    description: "Planificación ambiental y gestión de impactos.",
    accent: "from-emerald-500 via-emerald-400 to-lime-400",
    icon: Leaf,
  },
  {
    id: "calidad",
    label: "Calidad",
    description: "Planeación enfocada en mejora continua.",
    accent: "from-sky-500 via-indigo-500 to-violet-500",
    icon: Award,
  },
  {
    id: "seguridad",
    label: "Seguridad",
    description: "Prevención y control de riesgos laborales.",
    accent: "from-rose-500 via-orange-500 to-amber-400",
    icon: ShieldCheck,
  },
];

const planningOptions = [
  {
    id: "riesgos-oportunidades",
    title: "Riesgos y oportunidades",
    description: "Metodología aplicada y resultados clave.",
    href: "/meiwa-core/meiwa-sgi/planning/risk-and-opportunities",
    icon: Target,
    badge: "Metodología + resultado",
  },
  {
    id: "aspectos-impactos",
    title: "Matriz de aspectos e impactos",
    description: "Criterios, evaluación y significancia ambiental.",
    href: "/meiwa-core/meiwa-sgi/planning/aspects-impacts",
    icon: Leaf,
    badge: "Criterios y significancia",
  },
  {
    id: "requisitos-legales",
    title: "Matriz de requisitos legales",
    description: "Otros requisitos y su aplicabilidad.",
    href: "/meiwa-core/meiwa-sgi/planning/legal-requirements",
    icon: Scale,
    badge: "Aplicabilidad",
  },
  {
    id: "plan-acciones",
    title: "Plan de acciones",
    description: "Qué haremos, responsables y plazos.",
    href: "/meiwa-core/meiwa-sgi/planning/action-plan",
    icon: ClipboardList,
    badge: "Seguimiento",
  },
  {
    id: "objetivos-ambientales",
    title: "Objetivos ambientales",
    description: "Metas estratégicas, indicadores y seguimiento anual.",
    href: "/meiwa-core/meiwa-sgi/planning/environmental-objectives",
    icon: Trophy,
    badge: "Metas y avances",
  },
  {
    id: "indicadores-ambientales",
    title: "Indicadores ambientales",
    description: "KPIs, tendencias y desempeño ambiental.",
    href: "/meiwa-core/meiwa-sgi/planning/environmental-indicators",
    icon: LineChart,
    badge: "KPIs clave",
  },
];

export default function PlanningPage() {
  const [activeCategory, setActiveCategory] = useState(categories[0].id);
  const activeCategoryData = categories.find((category) => category.id === activeCategory);

  return (
    <SGIDashboardLayout>
      <div className="min-h-screen space-y-10 bg-gradient-to-br from-slate-50 via-white to-emerald-50 pb-20">
        <section className="relative overflow-hidden rounded-3xl border border-emerald-100 bg-white px-6 py-10 shadow-sm sm:px-10">
          <div className="absolute inset-0 bg-gradient-to-r from-emerald-500/10 to-lime-400/10" />
          <div className="relative space-y-4">
            <p className="text-xs font-semibold uppercase tracking-[0.5em] text-emerald-600">
              Sistema de Gestión Integral
            </p>
            <h1 className="text-3xl font-bold text-gray-900 sm:text-4xl">Planificación</h1>
            <p className="max-w-3xl text-base text-gray-600 sm:text-lg">
              Selecciona la disciplina para acceder a los elementos clave de la planificación.
              Las secciones disponibles de Medio Ambiente están listas para consulta.
            </p>
          </div>
        </section>

        <div className="flex flex-wrap gap-3">
          {categories.map((category) => {
            const Icon = category.icon;
            const isActive = activeCategory === category.id;

            return (
              <button
                key={category.id}
                type="button"
                onClick={() => setActiveCategory(category.id)}
                className={`flex items-center gap-2 rounded-full border px-4 py-2 text-sm font-semibold transition-all duration-200 ${
                  isActive
                    ? "border-transparent bg-gradient-to-r from-emerald-500 to-lime-500 text-white shadow-lg"
                    : "border-gray-200 bg-white text-gray-600 hover:border-emerald-200 hover:text-emerald-700"
                }`}
              >
                <Icon className={`h-4 w-4 ${isActive ? "text-white" : "text-emerald-500"}`} />
                {category.label}
              </button>
            );
          })}
        </div>

        {activeCategory === "medio-ambiente" ? (
          <section className="grid gap-6 lg:grid-cols-2">
            {planningOptions.map((option) => {
              const Icon = option.icon;

              return (
                <Link
                  key={option.id}
                  href={option.href}
                  className="group flex h-full flex-col justify-between rounded-3xl border border-emerald-100 bg-white p-6 shadow-sm transition duration-200 hover:-translate-y-1 hover:border-emerald-300 hover:shadow-lg"
                >
                  <div className="flex items-start justify-between gap-4">
                    <div className="rounded-2xl bg-emerald-50 p-3">
                      <Icon className="h-6 w-6 text-emerald-600" />
                    </div>
                    <span className="rounded-full bg-emerald-100 px-3 py-1 text-xs font-semibold uppercase tracking-wide text-emerald-700">
                      {option.badge}
                    </span>
                  </div>
                  <div className="mt-6 space-y-3">
                    <h2 className="text-xl font-semibold text-gray-900">{option.title}</h2>
                    <p className="text-sm text-gray-600">{option.description}</p>
                  </div>
                  <div className="mt-6 flex items-center gap-2 text-sm font-semibold text-emerald-600">
                    <span>Ir a la sección</span>
                    <ArrowUpRight className="h-4 w-4 transition-transform duration-200 group-hover:translate-x-1" />
                  </div>
                </Link>
              );
            })}
          </section>
        ) : (
          <section className="rounded-3xl border border-slate-200 bg-white p-10 text-center shadow-sm">
            <div className="mx-auto flex h-14 w-14 items-center justify-center rounded-2xl bg-slate-50">
              <FileText className="h-6 w-6 text-slate-400" />
            </div>
            <p className="mt-4 text-xs font-semibold uppercase tracking-[0.3em] text-slate-400">
              Próximamente
            </p>
            <h2 className="mt-3 text-2xl font-semibold text-gray-900">
              Planificación de {activeCategoryData?.label} en preparación
            </h2>
            <p className="mt-2 text-sm text-gray-600">
              Estas secciones estarán disponibles próximamente. Te avisaremos cuando el contenido
              esté listo para consulta.
            </p>
          </section>
        )}
      </div>
    </SGIDashboardLayout>
  );
}