"use client";

import { useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import { BookCheck, ChartBar, ClipboardList, Medal, RefreshCcw, UserCircle } from "lucide-react";
import { type User } from "@supabase/supabase-js";
import { toast } from "sonner";

import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { supabase } from "@/lib/supabase";

const allowedManagers = new Set([
  "ehs.mexico@meiwa.com.mx",
  "recursoshumanos@meiwa.com.mx",
]);

const statusLabels: Record<string, string> = {
  not_started: "No iniciado",
  in_progress: "En progreso",
  completed: "Completado",
};

const statusBadgeClasses: Record<string, string> = {
  not_started: "bg-slate-100 text-slate-700",
  in_progress: "bg-amber-100 text-amber-700",
  completed: "bg-emerald-100 text-emerald-700",
};

const categoryLabels: Record<string, string> = {
  environment: "Medio Ambiente",
  quality: "Calidad",
  safety: "Seguridad",
  general: "General",
};

const formatDate = (value?: string | null) => {
  if (!value) return "Sin fecha";
  return new Date(value).toLocaleDateString("es-MX");
};

const formatNumber = (value?: number | null, fallback = 0) => (value == null ? fallback : value);

const formatPercent = (value: number) => `${Math.round(value)}%`;

type ModuleRecord = {
  id: string;
  title: string;
  category: string;
  points_available: number;
  is_active?: boolean | null;
};

type ProgressRecord = {
  id: string;
  user_id: string;
  module_id: string;
  status: "not_started" | "in_progress" | "completed";
  progress_percent: number | null;
  final_score: number | null;
  attempts_count: number | null;
  points_obtained: number | null;
  start_date: string | null;
  end_date: string | null;
  last_updated_at: string | null;
};

type ProfileRecord = {
  id: string;
  full_name: string | null;
  email: string | null;
};

export default function AwarenessGradesPage() {
  const router = useRouter();
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [progress, setProgress] = useState<ProgressRecord[]>([]);
  const [modules, setModules] = useState<ModuleRecord[]>([]);
  const [profiles, setProfiles] = useState<ProfileRecord[]>([]);
  const [selectedUserId, setSelectedUserId] = useState<string>("");
  const [userSearch, setUserSearch] = useState("");
  const [categoryFilter, setCategoryFilter] = useState("all");
  const [statusFilter, setStatusFilter] = useState("all");
  const [moduleSearch, setModuleSearch] = useState("");
  const [startDateFilter, setStartDateFilter] = useState("");
  const [endDateFilter, setEndDateFilter] = useState("");

  const canManage = allowedManagers.has(user?.email ?? "");

  useEffect(() => {
    const fetchSession = async () => {
      setLoading(true);
      const { data: { session } } = await supabase.auth.getSession();

      if (!session?.user) {
        router.push("/");
        return;
      }

      setUser(session.user);

      if (allowedManagers.has(session.user.email ?? "")) {
        const { data: profileData, error } = await supabase
          .from("profiles")
          .select("id, full_name, email")
          .order("email", { ascending: true });

        if (error) {
          toast.error("No se pudieron cargar los usuarios.");
          console.error(error);
        }

        setProfiles(profileData ?? []);
        setSelectedUserId("all");
      } else {
        setSelectedUserId(session.user.id);
      }
    };

    void fetchSession();
  }, [router]);

  useEffect(() => {
    const fetchProgress = async () => {
      if (!user) return;
      if (!selectedUserId) return;

      setLoading(true);

      let progressQuery = supabase
        .from("awareness_user_module_progress")
        .select("*");

      if (!canManage) {
        progressQuery = progressQuery.eq("user_id", user.id);
      } else if (selectedUserId !== "all") {
        progressQuery = progressQuery.eq("user_id", selectedUserId);
      }

      const { data: progressData, error: progressError } = await progressQuery;

      if (progressError) {
        toast.error("No se pudieron cargar los resultados.");
        console.error(progressError);
        setLoading(false);
        return;
      }

      const progressRows = (progressData ?? []) as ProgressRecord[];
      setProgress(progressRows);

      const moduleIds = progressRows.map((record) => record.module_id);

      if (!moduleIds.length) {
        setModules([]);
        setLoading(false);
        return;
      }

      const { data: moduleData, error: moduleError } = await supabase
        .from("awareness_modules")
        .select("id, title, category, points_available, is_active")
        .in("id", moduleIds)
        .order("order_index", { ascending: true, nullsFirst: true });

      if (moduleError) {
        toast.error("No se pudieron cargar los módulos.");
        console.error(moduleError);
      }

      setModules((moduleData ?? []) as ModuleRecord[]);
      setLoading(false);
    };

    void fetchProgress();
  }, [user, selectedUserId, canManage]);

  const profileLookup = useMemo(() => {
    return profiles.reduce<Record<string, ProfileRecord>>((acc, profile) => {
      acc[profile.id] = profile;
      return acc;
    }, {});
  }, [profiles]);

  const moduleLookup = useMemo(() => {
    return modules.reduce<Record<string, ModuleRecord>>((acc, module) => {
      acc[module.id] = module;
      return acc;
    }, {});
  }, [modules]);

  const stats = useMemo(() => {
    const totalRecords = progress.length;
    const completedRecords = progress.filter((record) => record.status === "completed");
    const inProgressRecords = progress.filter((record) => record.status === "in_progress");

    const totalProgress = progress.reduce((sum, record) => {
      return sum + formatNumber(record.progress_percent, 0);
    }, 0);

    const totalScore = completedRecords.reduce((sum, record) => {
      return sum + formatNumber(record.final_score, 0);
    }, 0);

    const totalPoints = progress.reduce((sum, record) => {
      return sum + formatNumber(record.points_obtained, 0);
    }, 0);

    const totalAttempts = progress.reduce((sum, record) => {
      return sum + formatNumber(record.attempts_count, 0);
    }, 0);

    const lastCompleted = completedRecords
      .filter((record) => record.end_date)
      .sort((a, b) => new Date(b.end_date ?? 0).getTime() - new Date(a.end_date ?? 0).getTime())[0];

    return {
      totalRecords,
      completedCount: completedRecords.length,
      inProgressCount: inProgressRecords.length,
      avgProgress: totalRecords ? totalProgress / totalRecords : 0,
      globalScore: completedRecords.length ? totalScore / completedRecords.length : 0,
      totalPoints,
      totalAttempts,
      lastCompletedDate: lastCompleted?.end_date ?? null,
    };
  }, [progress]);

  const filteredRows = useMemo(() => {
    return progress
      .map((record) => {
        const module = moduleLookup[record.module_id];
        return {
          ...record,
          module,
        };
      })
      .filter((record) => {
        if (categoryFilter !== "all" && record.module?.category !== categoryFilter) return false;
        if (statusFilter !== "all" && record.status !== statusFilter) return false;
        if (moduleSearch.trim()) {
          const target = record.module?.title ?? "";
          if (!target.toLowerCase().includes(moduleSearch.trim().toLowerCase())) return false;
        }
        if (startDateFilter) {
          if (!record.start_date) return false;
          if (new Date(record.start_date) < new Date(startDateFilter)) return false;
        }
        if (endDateFilter) {
          if (!record.end_date) return false;
          if (new Date(record.end_date) > new Date(endDateFilter)) return false;
        }
        return true;
      })
      .sort((a, b) => new Date(b.last_updated_at ?? 0).getTime() - new Date(a.last_updated_at ?? 0).getTime());
  }, [progress, moduleLookup, categoryFilter, statusFilter, moduleSearch, startDateFilter, endDateFilter]);

  const userOptions = useMemo(() => {
    return profiles.filter((profile) => {
      if (!userSearch.trim()) return true;
      const query = userSearch.trim().toLowerCase();
      return (
        profile.email?.toLowerCase().includes(query) ||
        profile.full_name?.toLowerCase().includes(query) ||
        profile.id.toLowerCase().includes(query)
      );
    });
  }, [profiles, userSearch]);

  const selectedUserLabel = useMemo(() => {
    if (canManage) {
      if (selectedUserId === "all") return "Todos los usuarios";
      const profile = profileLookup[selectedUserId];
      return profile?.email ?? profile?.full_name ?? selectedUserId;
    }
    return user?.email ?? user?.id ?? "";
  }, [canManage, selectedUserId, profileLookup, user]);

  return (
    <SGIDashboardLayout>
      <div className="space-y-8">
        <div className="flex flex-wrap items-center justify-between gap-4">
          <div>
            <p className="text-sm font-semibold uppercase tracking-[0.2em] text-muted-foreground">
              Toma de Conciencia · Calificaciones
            </p>
            <h1 className="text-2xl font-bold text-gray-900">Dashboard de resultados</h1>
            <p className="text-sm text-muted-foreground">
              Consulta el avance y los resultados de los módulos de Awareness.
            </p>
          </div>
          <Badge variant="outline" className="text-xs">
            {selectedUserLabel ? `Usuario: ${selectedUserLabel}` : "Usuario"}
          </Badge>
        </div>

        <div className="grid gap-4 lg:grid-cols-3">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-sm">Módulos en progreso</CardTitle>
              <RefreshCcw className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <p className="text-2xl font-semibold">{stats.inProgressCount}</p>
              <p className="text-xs text-muted-foreground">En avance actualmente</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-sm">Módulos completados</CardTitle>
              <BookCheck className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <p className="text-2xl font-semibold">{stats.completedCount}</p>
              <p className="text-xs text-muted-foreground">Con evaluación final</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-sm">Avance promedio</CardTitle>
              <ChartBar className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <p className="text-2xl font-semibold">{formatPercent(stats.avgProgress)}</p>
              <p className="text-xs text-muted-foreground">Promedio de progreso</p>
            </CardContent>
          </Card>
        </div>

        <div className="grid gap-4 lg:grid-cols-3">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-sm">Calificación global</CardTitle>
              <Medal className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <p className="text-2xl font-semibold">{Math.round(stats.globalScore)}</p>
              <p className="text-xs text-muted-foreground">Promedio en módulos completados</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-sm">Puntos obtenidos</CardTitle>
              <ClipboardList className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <p className="text-2xl font-semibold">{stats.totalPoints}</p>
              <p className="text-xs text-muted-foreground">Suma de puntos</p>
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle className="text-sm">Intentos totales</CardTitle>
              <UserCircle className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <p className="text-2xl font-semibold">{stats.totalAttempts}</p>
              <p className="text-xs text-muted-foreground">
                Último completado: {formatDate(stats.lastCompletedDate)}
              </p>
            </CardContent>
          </Card>
        </div>

        <Card>
          <CardHeader>
            <CardTitle className="text-sm">Filtros</CardTitle>
          </CardHeader>
          <CardContent className="grid gap-4 md:grid-cols-2 xl:grid-cols-4">
            {canManage && (
              <div className="space-y-2">
                <label className="text-xs font-semibold text-muted-foreground">Usuario</label>
                <Input
                  value={userSearch}
                  onChange={(event) => setUserSearch(event.target.value)}
                  placeholder="Buscar por correo o UID"
                />
                <Select value={selectedUserId} onValueChange={setSelectedUserId}>
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona un usuario" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Todos</SelectItem>
                    {userOptions.map((profile) => (
                      <SelectItem key={profile.id} value={profile.id}>
                        {profile.email ?? profile.full_name ?? profile.id}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            )}
            <div className="space-y-2">
              <label className="text-xs font-semibold text-muted-foreground">Categoría</label>
              <Select value={categoryFilter} onValueChange={setCategoryFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Categoría" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todas las categorías</SelectItem>
                  <SelectItem value="environment">Medio Ambiente</SelectItem>
                  <SelectItem value="quality">Calidad</SelectItem>
                  <SelectItem value="safety">Seguridad</SelectItem>
                  <SelectItem value="general">General</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <label className="text-xs font-semibold text-muted-foreground">Estado</label>
              <Select value={statusFilter} onValueChange={setStatusFilter}>
                <SelectTrigger>
                  <SelectValue placeholder="Estado" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Todos los estados</SelectItem>
                  <SelectItem value="not_started">No iniciado</SelectItem>
                  <SelectItem value="in_progress">En progreso</SelectItem>
                  <SelectItem value="completed">Completado</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <div className="space-y-2">
              <label className="text-xs font-semibold text-muted-foreground">Módulo</label>
              <Input
                value={moduleSearch}
                onChange={(event) => setModuleSearch(event.target.value)}
                placeholder="Buscar por nombre"
              />
            </div>
            <div className="space-y-2">
              <label className="text-xs font-semibold text-muted-foreground">Fecha inicio (desde)</label>
              <Input
                type="date"
                value={startDateFilter}
                onChange={(event) => setStartDateFilter(event.target.value)}
              />
            </div>
            <div className="space-y-2">
              <label className="text-xs font-semibold text-muted-foreground">Fecha término (hasta)</label>
              <Input
                type="date"
                value={endDateFilter}
                onChange={(event) => setEndDateFilter(event.target.value)}
              />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="text-sm">Resultados</CardTitle>
          </CardHeader>
          <CardContent>
            {loading ? (
              <div className="py-6 text-center text-sm text-muted-foreground">Cargando resultados...</div>
            ) : filteredRows.length === 0 ? (
              <div className="py-6 text-center text-sm text-muted-foreground">
                No hay resultados con los filtros seleccionados.
              </div>
            ) : (
              <div className="overflow-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      {canManage && selectedUserId === "all" && (
                        <TableHead className="min-w-[180px]">Usuario</TableHead>
                      )}
                      <TableHead className="min-w-[140px]">Categoría</TableHead>
                      <TableHead className="min-w-[220px]">Módulo</TableHead>
                      <TableHead className="min-w-[120px]">Estado</TableHead>
                      <TableHead className="min-w-[120px]">Progreso</TableHead>
                      <TableHead className="min-w-[140px]">Calificación final</TableHead>
                      <TableHead className="min-w-[140px]">Puntos obtenidos</TableHead>
                      <TableHead className="min-w-[140px]">Puntos disponibles</TableHead>
                      <TableHead className="min-w-[140px]">Fecha inicio</TableHead>
                      <TableHead className="min-w-[140px]">Fecha término</TableHead>
                      <TableHead className="min-w-[120px]">Intentos</TableHead>
                      <TableHead className="min-w-[160px]">Última actualización</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {filteredRows.map((record) => (
                      <TableRow key={record.id}>
                        {canManage && selectedUserId === "all" && (
                          <TableCell className="font-medium">
                            {profileLookup[record.user_id]?.email ?? record.user_id}
                          </TableCell>
                        )}
                        <TableCell>
                          {categoryLabels[record.module?.category ?? ""] ?? record.module?.category ?? "—"}
                        </TableCell>
                        <TableCell className="font-medium">
                          {record.module?.title ?? "Módulo sin título"}
                        </TableCell>
                        <TableCell>
                          <Badge className={statusBadgeClasses[record.status] ?? "bg-muted text-muted-foreground"}>
                            {statusLabels[record.status] ?? record.status}
                          </Badge>
                        </TableCell>
                        <TableCell>{formatPercent(formatNumber(record.progress_percent, 0))}</TableCell>
                        <TableCell>{Math.round(formatNumber(record.final_score, 0))}</TableCell>
                        <TableCell>{formatNumber(record.points_obtained, 0)}</TableCell>
                        <TableCell>{formatNumber(record.module?.points_available, 0)}</TableCell>
                        <TableCell>{formatDate(record.start_date)}</TableCell>
                        <TableCell>{formatDate(record.end_date)}</TableCell>
                        <TableCell>{formatNumber(record.attempts_count, 0)}</TableCell>
                        <TableCell>{formatDate(record.last_updated_at)}</TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </SGIDashboardLayout>
  );
}