"use client";

import { useEffect, useMemo, useState } from "react";
import { useRouter } from "next/navigation";
import { toast } from "sonner";

import { AwarenessModuleForm } from "@/components/awareness/awareness-module-form";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { supabase } from "@/lib/supabase";

const allowedManagers = new Set([
  "ehs.mexico@meiwa.com.mx",
  "admin2.mexico@meiwa.com.mx",
  "auxiliar.ehs@meiwa.com.mx",
]);

export default function AwarenessModuleNewPage() {
  const router = useRouter();
  const [checkingAccess, setCheckingAccess] = useState(true);
  const [authorized, setAuthorized] = useState(false);
  const [modules, setModules] = useState<any[]>([]);
  const [modulesLoading, setModulesLoading] = useState(false);
  const [searchTerm, setSearchTerm] = useState("");

  const filteredModules = useMemo(() => {
    const query = searchTerm.trim().toLowerCase();
    if (!query) return modules;

    return modules.filter((module) => {
      const title = module.title?.toLowerCase() ?? "";
      const description = module.description?.toLowerCase() ?? "";
      return title.includes(query) || description.includes(query);
    });
  }, [modules, searchTerm]);

  useEffect(() => {
    const verifyAccess = async () => {
      const { data: { session } } = await supabase.auth.getSession();

      if (!session?.user) {
        router.push("/");
        return;
      }

      const canManage = allowedManagers.has(session.user.email ?? "");
      if (!canManage) {
        toast.error("No tienes permisos para crear módulos.");
        router.push("/meiwa-core/meiwa-sgi/support/awareness");
        return;
      }

      setAuthorized(true);
      setCheckingAccess(false);
    };

    void verifyAccess();
  }, [router]);

  useEffect(() => {
    if (!authorized) return;

    const fetchModules = async () => {
      setModulesLoading(true);
      const { data, error } = await supabase
        .from("awareness_modules")
        .select("id, title, description, category, is_active")
        .order("title");

      if (error) {
        console.error(error);
        toast.error("No pudimos cargar los módulos existentes.");
      } else {
        setModules(data ?? []);
      }

      setModulesLoading(false);
    };

    void fetchModules();
  }, [authorized]);

  if (checkingAccess || !authorized) {
    return (
      <SGIDashboardLayout>
        <Card>
          <CardContent className="py-10 text-center text-sm text-muted-foreground">
            Verificando permisos...
          </CardContent>
        </Card>
      </SGIDashboardLayout>
    );
  }

  return (
    <SGIDashboardLayout>
      <div className="space-y-6">
        <div>
          <p className="text-sm font-semibold uppercase tracking-[0.2em] text-muted-foreground">
            Awareness · Administración
          </p>
          <h1 className="text-2xl font-bold text-gray-900">Crear módulo</h1>
        </div>
        <Card>
          <CardContent className="space-y-4 py-6">
            <div className="space-y-1">
              <h2 className="text-lg font-semibold text-gray-900">Editar módulos existentes</h2>
              <p className="text-sm text-muted-foreground">
                Busca un módulo y abre la pantalla de edición sin usar el enlace directo.
              </p>
            </div>
            <Input
              value={searchTerm}
              onChange={(event) => setSearchTerm(event.target.value)}
              placeholder="Buscar módulos por nombre o descripción..."
            />
            {modulesLoading ? (
              <p className="text-sm text-muted-foreground">Cargando módulos...</p>
            ) : filteredModules.length === 0 ? (
              <p className="text-sm text-muted-foreground">No encontramos módulos con ese criterio.</p>
            ) : (
              <div className="divide-y rounded-lg border">
                {filteredModules.map((module) => (
                  <div
                    key={module.id}
                    className="flex flex-col gap-3 px-4 py-3 md:flex-row md:items-center md:justify-between"
                  >
                    <div className="space-y-1">
                      <div className="flex flex-wrap items-center gap-2">
                        <p className="font-medium text-gray-900">{module.title}</p>
                        {module.category && <Badge variant="secondary">{module.category}</Badge>}
                        {!module.is_active && <Badge variant="outline">Inactivo</Badge>}
                      </div>
                      <p className="text-sm text-muted-foreground">
                        {module.description ?? "Sin descripción"}
                      </p>
                    </div>
                    <Button
                      type="button"
                      variant="outline"
                      onClick={() =>
                        router.push(`/meiwa-core/meiwa-sgi/support/awareness/modules/edit?moduleId=${module.id}`)
                      }
                    >
                      Editar
                    </Button>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>
        <AwarenessModuleForm
          mode="create"
          onSuccess={() => router.push("/meiwa-core/meiwa-sgi/support/awareness")}
        />
      </div>
    </SGIDashboardLayout>
  );
}