"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { useSearchParams } from "next/navigation";
import { supabase } from "@/lib/supabase";
import { SGIDashboardLayout } from "@/components/dashboard/sgi-dashboard-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Textarea } from "@/components/ui/textarea";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import {
  Calendar,
  FileText,
  Filter,
  Pencil,
  Plus,
  Search,
  Trash2,
  Upload,
} from "lucide-react";
import { toast } from "sonner";
import type { User } from "@supabase/supabase-js";

const allowedManagers = new Set(["ehs.mexico@meiwa.com.mx", "auxiliar.ehs@meiwa.com.mx"]);

type CommunicationMatrixRecord = {
  id: string;
  category: string;
  communication_type: string;
  what_to_communicate: string;
  when_to_communicate: string;
  who_to_communicate: string;
  how_to_communicate: string;
  who_communicates: string;
  followup_responsible: string | null;
  updated_at: string | null;
};

type CommunicationPost = {
  id: string;
  title: string;
  category: string;
  summary: string | null;
  content: string | null;
  publish_date: string | null;
  status: "draft" | "published" | "archived";
  created_by: string | null;
  created_at: string;
  updated_at: string | null;
  valid_from: string | null;
  valid_to: string | null;
};

type CommunicationAttachment = {
  id: string;
  post_id: string;
  file_name: string;
  file_path: string;
  file_type: string | null;
  uploaded_at: string | null;
};

type CommunicationIncident = {
  id: string;
  created_at: string;
  created_by: string | null;
  source_type: string;
  source_other_detail: string | null;
  stakeholder: string;
  classification: "complaint" | "suggestion" | "report" | "other";
  description: string;
  evidence_url: string | null;
  action_plan: string | null;
  plan_owner: string | null;
  target_date: string | null;
  start_date: string | null;
  close_date: string | null;
  result: string | null;
  confirmation_date: string | null;
  status: string | null;
  updated_at: string | null;
  updated_by: string | null;
};

const emptyMatrixForm = {
  category: "",
  communication_type: "",
  what_to_communicate: "",
  when_to_communicate: "",
  who_to_communicate: "",
  how_to_communicate: "",
  who_communicates: "",
  followup_responsible: "",
};

const emptyPostForm = {
  title: "",
  category: "",
  summary: "",
  content: "",
  publish_date: "",
  status: "draft" as CommunicationPost["status"],
  valid_from: "",
  valid_to: "",
};

const emptyIncidentForm = {
  source_type: "",
  source_other_detail: "",
  stakeholder: "",
  classification: "complaint" as CommunicationIncident["classification"],
  description: "",
  evidence_url: "",
  action_plan: "",
  plan_owner: "",
  target_date: "",
  start_date: "",
  close_date: "",
  result: "",
  confirmation_date: "",
};

const incidentSourceOptions = [
  "Encuesta de satisfacción",
  "Correo",
  "Oficio",
  "Citatorio",
  "Llamada",
  "Visita/Inspección",
  "Reunión",
  "Red social",
  "Otro",
];

const incidentClassificationOptions: Array<{ value: CommunicationIncident["classification"]; label: string }> = [
  { value: "complaint", label: "Queja" },
  { value: "suggestion", label: "Sugerencia" },
  { value: "report", label: "Denuncia" },
  { value: "other", label: "Otra comunicación" },
];

const statusOptions: Array<{ value: CommunicationPost["status"]; label: string; badge: string }> = [
  { value: "draft", label: "Borrador", badge: "bg-slate-100 text-slate-700" },
  { value: "published", label: "Publicado", badge: "bg-emerald-100 text-emerald-700" },
  { value: "archived", label: "Archivado", badge: "bg-amber-100 text-amber-700" },
];

const incidentStatusLabels: Record<string, string> = {
  unassigned: "Sin asignar",
  planned: "Planeado",
  in_progress: "En proceso",
  completed: "Completo",
  confirmed: "Confirmado",
  overdue: "Retrasado",
};

const incidentStatusBadge: Record<string, string> = {
  unassigned: "bg-slate-100 text-slate-700",
  planned: "bg-blue-100 text-blue-700",
  in_progress: "bg-amber-100 text-amber-700",
  completed: "bg-emerald-100 text-emerald-700",
  confirmed: "bg-indigo-100 text-indigo-700",
  overdue: "bg-rose-100 text-rose-700",
};

const formatDate = (value?: string | null) => {
  if (!value) return "-";
  const date = new Date(value);
  if (Number.isNaN(date.getTime())) return "-";
  return date.toLocaleDateString("es-MX");
};

const toDateInputValue = (value?: string | null) => {
  if (!value) return "";
  const date = new Date(value);
  if (Number.isNaN(date.getTime())) return "";
  return date.toISOString().split("T")[0];
};

const getPublicFileUrl = (filePath: string) => {
  const { data } = supabase.storage.from("communications").getPublicUrl(filePath);
  return data.publicUrl;
};

const getEvidencePublicUrl = (filePath: string) => {
  const { data } = supabase.storage.from("communication-evidence").getPublicUrl(filePath);
  return data.publicUrl;
};

const imageExtensions = new Set(["png", "jpg", "jpeg", "gif", "webp", "bmp", "svg"]);

const isImageAttachment = (attachment: CommunicationAttachment) => {
  const fileType = attachment.file_type?.toLowerCase();
  if (fileType?.startsWith("image/")) return true;
  const extension = attachment.file_name.split(".").pop()?.toLowerCase();
  return extension ? imageExtensions.has(extension) : false;
};

const resolveIncidentStatus = (incident: Partial<CommunicationIncident>) => {
  if (incident.confirmation_date) return "confirmed";
  if (incident.close_date) return "completed";

  const now = new Date();
  if (incident.target_date) {
    const targetDate = new Date(incident.target_date);
    if (!Number.isNaN(targetDate.getTime()) && now > targetDate) {
      return "overdue";
    }
  }

  if (incident.start_date) return "in_progress";
  if (incident.plan_owner) return "planned";
  return "unassigned";
};

export default function CommunicationPage() {
  const searchParams = useSearchParams();  
  const [user, setUser] = useState<User | null>(null);
  const [canManage, setCanManage] = useState(false);
  const [matrixRecords, setMatrixRecords] = useState<CommunicationMatrixRecord[]>([]);
  const [matrixLoading, setMatrixLoading] = useState(false);
  const [matrixSearch, setMatrixSearch] = useState("");
  const [matrixCategoryFilter, setMatrixCategoryFilter] = useState("all");
  const [matrixTypeFilter, setMatrixTypeFilter] = useState("all");
  const [matrixSort, setMatrixSort] = useState("category");
  const [matrixDialogOpen, setMatrixDialogOpen] = useState(false);
  const [editingMatrix, setEditingMatrix] = useState<CommunicationMatrixRecord | null>(null);
  const [matrixForm, setMatrixForm] = useState(emptyMatrixForm);
  const [matrixDeleteTarget, setMatrixDeleteTarget] = useState<CommunicationMatrixRecord | null>(null);

  const [posts, setPosts] = useState<CommunicationPost[]>([]);
  const [attachments, setAttachments] = useState<CommunicationAttachment[]>([]);
  const [postLoading, setPostLoading] = useState(false);
  const [postSearch, setPostSearch] = useState("");
  const [postCategoryFilter, setPostCategoryFilter] = useState("all");
  const [postStatusFilter, setPostStatusFilter] = useState("all");
  const [postDateFrom, setPostDateFrom] = useState("");
  const [postDateTo, setPostDateTo] = useState("");
  const [postDialogOpen, setPostDialogOpen] = useState(false);
  const [postForm, setPostForm] = useState(emptyPostForm);
  const [editingPost, setEditingPost] = useState<CommunicationPost | null>(null);
  const [postFiles, setPostFiles] = useState<FileList | null>(null);
  const [postDeleteTarget, setPostDeleteTarget] = useState<CommunicationPost | null>(null);
  const [detailPost, setDetailPost] = useState<CommunicationPost | null>(null);

  const [incidents, setIncidents] = useState<CommunicationIncident[]>([]);
  const [incidentLoading, setIncidentLoading] = useState(false);
  const [incidentSearch, setIncidentSearch] = useState("");
  const [incidentClassificationFilter, setIncidentClassificationFilter] = useState("all");
  const [incidentSourceFilter, setIncidentSourceFilter] = useState("all");
  const [incidentStatusFilter, setIncidentStatusFilter] = useState("all");
  const [incidentStakeholderFilter, setIncidentStakeholderFilter] = useState("all");
  const [incidentTargetFrom, setIncidentTargetFrom] = useState("");
  const [incidentTargetTo, setIncidentTargetTo] = useState("");
  const [incidentStartFrom, setIncidentStartFrom] = useState("");
  const [incidentStartTo, setIncidentStartTo] = useState("");
  const [incidentCloseFrom, setIncidentCloseFrom] = useState("");
  const [incidentCloseTo, setIncidentCloseTo] = useState("");
  const [incidentDialogOpen, setIncidentDialogOpen] = useState(false);
  const [editingIncident, setEditingIncident] = useState<CommunicationIncident | null>(null);
  const [incidentForm, setIncidentForm] = useState(emptyIncidentForm);
  const [incidentEvidenceFile, setIncidentEvidenceFile] = useState<File | null>(null);
  const [incidentDeleteTarget, setIncidentDeleteTarget] = useState<CommunicationIncident | null>(null);

  useEffect(() => {
    const loadSession = async () => {
      const { data } = await supabase.auth.getSession();
      const currentUser = data.session?.user ?? null;
      setUser(currentUser);
      setCanManage(allowedManagers.has(currentUser?.email ?? ""));
    };

    loadSession();

    const { data: subscription } = supabase.auth.onAuthStateChange((_event, session) => {
      setUser(session?.user ?? null);
      setCanManage(allowedManagers.has(session?.user?.email ?? ""));
    });

    return () => {
      subscription.subscription.unsubscribe();
    };
  }, []);

  useEffect(() => {
    const postId = searchParams.get("postId");
    if (!postId || postLoading) {
      return;
    }

    const matchingPost = posts.find((post) => post.id === postId);
    if (matchingPost) {
      setDetailPost(matchingPost);
    }
  }, [posts, postLoading, searchParams]);

  const loadMatrix = useCallback(async () => {
    setMatrixLoading(true);
    const { data, error } = await supabase
      .from("communication_matrix")
      .select("*")
      .order("category", { ascending: true });

    if (error) {
      toast.error("No se pudo cargar la matriz de comunicación.");
      setMatrixLoading(false);
      return;
    }

    setMatrixRecords(data ?? []);
    setMatrixLoading(false);
  }, []);

  const loadPosts = useCallback(async () => {
    setPostLoading(true);
    let query = supabase.from("communications_posts").select("*").order("publish_date", { ascending: false });

    if (!canManage) {
      query = query.eq("status", "published");
    }

    const { data, error } = await query;

    if (error) {
      toast.error("No se pudieron cargar los comunicados.");
      setPostLoading(false);
      return;
    }

    setPosts(data ?? []);
    setPostLoading(false);
  }, [canManage]);

  const loadAttachments = useCallback(async (postIds: string[]) => {
    if (postIds.length === 0) {
      setAttachments([]);
      return;
    }

    const { data, error } = await supabase
      .from("communications_attachments")
      .select("*")
      .in("post_id", postIds);

    if (error) {
      return;
    }

    setAttachments(data ?? []);
  }, []);

  const loadIncidents = useCallback(async () => {
    setIncidentLoading(true);
    const { data, error } = await supabase
      .from("communication_incidents")
      .select("*")
      .order("created_at", { ascending: false });

    if (error) {
      toast.error("No se pudieron cargar las incidencias de comunicación.");
      setIncidentLoading(false);
      return;
    }

    setIncidents(data ?? []);
    setIncidentLoading(false);
  }, []);

  useEffect(() => {
    loadMatrix();
  }, [loadMatrix]);

  useEffect(() => {
    loadPosts();
  }, [loadPosts]);

  useEffect(() => {
    loadAttachments(posts.map((post) => post.id));
  }, [posts, loadAttachments]);

  useEffect(() => {
    loadIncidents();
  }, [loadIncidents]);

  const matrixCategories = useMemo(() => {
    const categories = new Set(matrixRecords.map((record) => record.category).filter(Boolean));
    return Array.from(categories).sort();
  }, [matrixRecords]);

  const matrixTypes = useMemo(() => {
    const types = new Set(matrixRecords.map((record) => record.communication_type).filter(Boolean));
    return Array.from(types).sort();
  }, [matrixRecords]);

  const filteredMatrix = useMemo(() => {
    const searchValue = matrixSearch.trim().toLowerCase();
    let filtered = matrixRecords.filter((record) => {
      if (matrixCategoryFilter !== "all" && record.category !== matrixCategoryFilter) return false;
      if (matrixTypeFilter !== "all" && record.communication_type !== matrixTypeFilter) return false;

      if (!searchValue) return true;

      return [
        record.category,
        record.communication_type,
        record.what_to_communicate,
        record.when_to_communicate,
        record.who_to_communicate,
        record.who_communicates,
      ]
        .join(" ")
        .toLowerCase()
        .includes(searchValue);
    });

    if (matrixSort === "category") {
      filtered = [...filtered].sort((a, b) => a.category.localeCompare(b.category));
    }
    if (matrixSort === "when") {
      filtered = [...filtered].sort((a, b) => a.when_to_communicate.localeCompare(b.when_to_communicate));
    }

    return filtered;
  }, [matrixRecords, matrixSearch, matrixCategoryFilter, matrixTypeFilter, matrixSort]);

  const postCategories = useMemo(() => {
    const categories = new Set(posts.map((post) => post.category).filter(Boolean));
    return Array.from(categories).sort();
  }, [posts]);

  const filteredPosts = useMemo(() => {
    const searchValue = postSearch.trim().toLowerCase();
    return posts.filter((post) => {
      if (postCategoryFilter !== "all" && post.category !== postCategoryFilter) return false;
      if (canManage && postStatusFilter !== "all" && post.status !== postStatusFilter) return false;

      if (postDateFrom) {
        const from = new Date(postDateFrom);
        const postDate = post.publish_date ? new Date(post.publish_date) : null;
        if (postDate && postDate < from) return false;
      }

      if (postDateTo) {
        const to = new Date(postDateTo);
        const postDate = post.publish_date ? new Date(post.publish_date) : null;
        if (postDate && postDate > to) return false;
      }

      if (!searchValue) return true;

      return [post.title, post.summary ?? "", post.content ?? ""].join(" ").toLowerCase().includes(searchValue);
    });
  }, [posts, postSearch, postCategoryFilter, postStatusFilter, postDateFrom, postDateTo, canManage]);

  const postAttachments = useMemo(() => {
    return attachments.reduce<Record<string, CommunicationAttachment[]>>((acc, attachment) => {
      if (!acc[attachment.post_id]) {
        acc[attachment.post_id] = [];
      }
      acc[attachment.post_id].push(attachment);
      return acc;
    }, {});
  }, [attachments]);

  const incidentSources = useMemo(() => {
    const sources = new Set(incidents.map((incident) => incident.source_type).filter(Boolean));
    return Array.from(sources).sort();
  }, [incidents]);

  const incidentStakeholders = useMemo(() => {
    const stakeholders = new Set(incidents.map((incident) => incident.stakeholder).filter(Boolean));
    return Array.from(stakeholders).sort();
  }, [incidents]);

  const filteredIncidents = useMemo(() => {
    const searchValue = incidentSearch.trim().toLowerCase();

    return incidents.filter((incident) => {
      const incidentStatus = resolveIncidentStatus(incident);
      if (incidentClassificationFilter !== "all" && incident.classification !== incidentClassificationFilter) {
        return false;
      }
      if (incidentSourceFilter !== "all" && incident.source_type !== incidentSourceFilter) {
        return false;
      }
      if (incidentStatusFilter !== "all" && incidentStatus !== incidentStatusFilter) {
        return false;
      }
      if (incidentStakeholderFilter !== "all" && incident.stakeholder !== incidentStakeholderFilter) {
        return false;
      }

      const targetDate = incident.target_date ? new Date(incident.target_date) : null;
      const startDate = incident.start_date ? new Date(incident.start_date) : null;
      const closeDate = incident.close_date ? new Date(incident.close_date) : null;

      if (incidentTargetFrom && targetDate && targetDate < new Date(incidentTargetFrom)) {
        return false;
      }
      if (incidentTargetTo && targetDate && targetDate > new Date(incidentTargetTo)) {
        return false;
      }
      if (incidentStartFrom && startDate && startDate < new Date(incidentStartFrom)) {
        return false;
      }
      if (incidentStartTo && startDate && startDate > new Date(incidentStartTo)) {
        return false;
      }
      if (incidentCloseFrom && closeDate && closeDate < new Date(incidentCloseFrom)) {
        return false;
      }
      if (incidentCloseTo && closeDate && closeDate > new Date(incidentCloseTo)) {
        return false;
      }

      if (!searchValue) return true;

      return [
        incident.description,
        incident.action_plan ?? "",
        incident.result ?? "",
        incident.stakeholder,
        incident.source_type,
        incident.source_other_detail ?? "",
      ]
        .join(" ")
        .toLowerCase()
        .includes(searchValue);
    });
  }, [
    incidents,
    incidentSearch,
    incidentClassificationFilter,
    incidentSourceFilter,
    incidentStatusFilter,
    incidentStakeholderFilter,
    incidentTargetFrom,
    incidentTargetTo,
    incidentStartFrom,
    incidentStartTo,
    incidentCloseFrom,
    incidentCloseTo,
  ]);

  const openMatrixDialog = (record?: CommunicationMatrixRecord) => {
    if (record) {
      setEditingMatrix(record);
      setMatrixForm({
        category: record.category,
        communication_type: record.communication_type,
        what_to_communicate: record.what_to_communicate,
        when_to_communicate: record.when_to_communicate,
        who_to_communicate: record.who_to_communicate,
        how_to_communicate: record.how_to_communicate,
        who_communicates: record.who_communicates,
        followup_responsible: record.followup_responsible ?? "",
      });
    } else {
      setEditingMatrix(null);
      setMatrixForm(emptyMatrixForm);
    }
    setMatrixDialogOpen(true);
  };

  const saveMatrix = async () => {
    if (!canManage) {
      toast.error("No tienes permisos para modificar la matriz.");
      return;
    }

    if (!matrixForm.category || !matrixForm.communication_type || !matrixForm.what_to_communicate) {
      toast.error("Completa los campos obligatorios de la matriz.");
      return;
    }

    const payload = {
      category: matrixForm.category,
      communication_type: matrixForm.communication_type,
      what_to_communicate: matrixForm.what_to_communicate,
      when_to_communicate: matrixForm.when_to_communicate,
      who_to_communicate: matrixForm.who_to_communicate,
      how_to_communicate: matrixForm.how_to_communicate,
      who_communicates: matrixForm.who_communicates,
      followup_responsible: matrixForm.followup_responsible || null,
      updated_at: new Date().toISOString(),
    };

    const { error } = editingMatrix
      ? await supabase.from("communication_matrix").update(payload).eq("id", editingMatrix.id)
      : await supabase.from("communication_matrix").insert(payload);

    if (error) {
      toast.error("No se pudo guardar el registro.");
      return;
    }

    toast.success(editingMatrix ? "Registro actualizado." : "Registro creado.");
    setMatrixDialogOpen(false);
    await loadMatrix();
  };

  const confirmDeleteMatrix = async () => {
    if (!matrixDeleteTarget) return;
    const { error } = await supabase.from("communication_matrix").delete().eq("id", matrixDeleteTarget.id);

    if (error) {
      toast.error("No se pudo eliminar el registro.");
      return;
    }

    toast.success("Registro eliminado.");
    setMatrixDeleteTarget(null);
    await loadMatrix();
  };

  const openPostDialog = (post?: CommunicationPost) => {
    if (post) {
      setEditingPost(post);
      setPostForm({
        title: post.title,
        category: post.category,
        summary: post.summary ?? "",
        content: post.content ?? "",
        publish_date: toDateInputValue(post.publish_date),
        status: post.status,
        valid_from: toDateInputValue(post.valid_from),
        valid_to: toDateInputValue(post.valid_to),
      });
    } else {
      setEditingPost(null);
      setPostForm(emptyPostForm);
    }
    setPostFiles(null);
    setPostDialogOpen(true);
  };

  const openIncidentDialog = (incident?: CommunicationIncident) => {
    if (incident) {
      setEditingIncident(incident);
      setIncidentForm({
        source_type: incident.source_type,
        source_other_detail: incident.source_other_detail ?? "",
        stakeholder: incident.stakeholder,
        classification: incident.classification,
        description: incident.description,
        evidence_url: incident.evidence_url ?? "",
        action_plan: incident.action_plan ?? "",
        plan_owner: incident.plan_owner ?? "",
        target_date: toDateInputValue(incident.target_date),
        start_date: toDateInputValue(incident.start_date),
        close_date: toDateInputValue(incident.close_date),
        result: incident.result ?? "",
        confirmation_date: toDateInputValue(incident.confirmation_date),
      });
    } else {
      setEditingIncident(null);
      setIncidentForm(emptyIncidentForm);
    }
    setIncidentEvidenceFile(null);
    setIncidentDialogOpen(true);
  };

  const uploadIncidentEvidence = async (incidentId: string, file: File) => {
    const filePath = `${incidentId}/${Date.now()}-${file.name}`;
    const { error } = await supabase.storage
      .from("communication-evidence")
      .upload(filePath, file, { upsert: true });

    if (error) {
      toast.error("No se pudo subir la evidencia.");
      return null;
    }

    return getEvidencePublicUrl(filePath);
  };

  const uploadAttachments = async (postId: string, files: FileList) => {
    const uploads = Array.from(files);
    for (const file of uploads) {
      const filePath = `${postId}/${Date.now()}-${file.name}`;
      const { error: uploadError } = await supabase.storage
        .from("communications")
        .upload(filePath, file, { upsert: true });

      if (uploadError) {
        toast.error(`No se pudo subir ${file.name}.`);
        continue;
      }

      const { error: insertError } = await supabase.from("communications_attachments").insert({
        post_id: postId,
        file_name: file.name,
        file_path: filePath,
        file_type: file.type,
      });

      if (insertError) {
        toast.error(`No se pudo registrar ${file.name}.`);
      }
    }
  };

  const savePost = async () => {
    if (!canManage) {
      toast.error("No tienes permisos para modificar comunicados.");
      return;
    }

    if (!postForm.title || !postForm.category) {
      toast.error("Título y categoría son obligatorios.");
      return;
    }

    const publishDateValue = postForm.publish_date
      ? new Date(postForm.publish_date).toISOString()
      : postForm.status === "published"
        ? new Date().toISOString()
        : null;

    const payload = {
      title: postForm.title,
      category: postForm.category,
      summary: postForm.summary || null,
      content: postForm.content || null,
      publish_date: publishDateValue,
      status: postForm.status,
      created_by: user?.email ?? null,
      updated_at: new Date().toISOString(),
      valid_from: postForm.valid_from ? new Date(postForm.valid_from).toISOString() : null,
      valid_to: postForm.valid_to ? new Date(postForm.valid_to).toISOString() : null,
    };

    const response = editingPost
      ? await supabase
          .from("communications_posts")
          .update(payload)
          .eq("id", editingPost.id)
          .select()
          .single()
      : await supabase.from("communications_posts").insert(payload).select().single();

    if (response.error || !response.data) {
      toast.error("No se pudo guardar el comunicado.");
      return;
    }

    if (postFiles && postFiles.length > 0) {
      await uploadAttachments(response.data.id, postFiles);
    }

    toast.success(editingPost ? "Comunicado actualizado." : "Comunicado creado.");
    setPostDialogOpen(false);
    await loadPosts();
  };

  const saveIncident = async () => {
    if (!user) {
      toast.error("Necesitas iniciar sesión para registrar una incidencia.");
      return;
    }

    if (editingIncident && !canManage) {
      toast.error("No tienes permisos para modificar incidencias.");
      return;
    }

    if (!incidentForm.source_type || !incidentForm.stakeholder || !incidentForm.description) {
      toast.error("Completa los campos obligatorios de la incidencia.");
      return;
    }

    if (incidentForm.source_type === "Otro" && !incidentForm.source_other_detail) {
      toast.error("Especifica la fuente de la comunicación.");
      return;
    }

    const hasEvidence = Boolean(incidentForm.evidence_url) || Boolean(incidentEvidenceFile);
    if (!hasEvidence && !editingIncident?.evidence_url) {
      toast.error("Agrega evidencia con un enlace o archivo.");
      return;
    }

    const payloadBase = {
      source_type: incidentForm.source_type,
      source_other_detail: incidentForm.source_other_detail || null,
      stakeholder: incidentForm.stakeholder,
      classification: incidentForm.classification,
      description: incidentForm.description,
      evidence_url: incidentForm.evidence_url || null,
      action_plan: canManage ? incidentForm.action_plan || null : null,
      plan_owner: canManage ? incidentForm.plan_owner || null : null,
      target_date: canManage && incidentForm.target_date ? incidentForm.target_date : null,
      start_date: canManage && incidentForm.start_date ? incidentForm.start_date : null,
      close_date: canManage && incidentForm.close_date ? incidentForm.close_date : null,
      result: canManage ? incidentForm.result || null : null,
      confirmation_date: canManage && incidentForm.confirmation_date ? incidentForm.confirmation_date : null,
      updated_at: new Date().toISOString(),
      updated_by: user.email ?? null,
    };

    const status = resolveIncidentStatus(payloadBase);
    const payload = { ...payloadBase, status };

    const response = editingIncident
      ? await supabase.from("communication_incidents").update(payload).eq("id", editingIncident.id).select().single()
      : await supabase
          .from("communication_incidents")
          .insert({
            ...payload,
            created_by: user.email ?? null,
          })
          .select()
          .single();

    if (response.error || !response.data) {
      toast.error("No se pudo guardar la incidencia.");
      return;
    }

    if (incidentEvidenceFile) {
      const evidenceUrl = await uploadIncidentEvidence(response.data.id, incidentEvidenceFile);
      if (evidenceUrl) {
        await supabase
          .from("communication_incidents")
          .update({ evidence_url: evidenceUrl, updated_at: new Date().toISOString(), updated_by: user.email ?? null })
          .eq("id", response.data.id);
      }
    }

    toast.success(editingIncident ? "Incidencia actualizada." : "Incidencia registrada.");
    setIncidentDialogOpen(false);
    await loadIncidents();
  };

  const confirmDeletePost = async () => {
    if (!postDeleteTarget) return;

    const { error } = await supabase.from("communications_posts").delete().eq("id", postDeleteTarget.id);

    if (error) {
      toast.error("No se pudo eliminar el comunicado.");
      return;
    }

    toast.success("Comunicado eliminado.");
    setPostDeleteTarget(null);
    await loadPosts();
  };

  const confirmDeleteIncident = async () => {
    if (!incidentDeleteTarget) return;
    if (!canManage) {
      toast.error("No tienes permisos para eliminar incidencias.");
      return;
    }

    const { error } = await supabase.from("communication_incidents").delete().eq("id", incidentDeleteTarget.id);

    if (error) {
      toast.error("No se pudo eliminar la incidencia.");
      return;
    }

    toast.success("Incidencia eliminada.");
    setIncidentDeleteTarget(null);
    await loadIncidents();
  };

  return (
    <SGIDashboardLayout>
      <div className="space-y-8 pb-16">
        <section className="rounded-3xl border border-slate-200 bg-white px-6 py-8 shadow-sm">
          <div className="space-y-2">
            <p className="text-xs font-semibold uppercase tracking-[0.3em] text-emerald-500">Soporte SGI</p>
            <h1 className="text-3xl font-bold text-gray-900">Comunicación</h1>
            <p className="max-w-3xl text-sm text-gray-600">
              Gestiona la matriz de comunicación y publica comunicados internos. Los usuarios con permisos pueden
              crear, editar y actualizar la información; el resto del equipo tendrá acceso de solo lectura.
            </p>
          </div>
        </section>

        <Tabs defaultValue="posts" className="space-y-6">
          <TabsList className="grid w-full max-w-2xl grid-cols-3">
            <TabsTrigger value="posts">Comunicados</TabsTrigger>
            <TabsTrigger value="matrix">Matriz de Comunicación</TabsTrigger>
            <TabsTrigger value="incidents">Incidencias</TabsTrigger>
          </TabsList>

          <TabsContent value="matrix" className="space-y-6">
            <Card>
              <CardHeader className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
                <div>
                  <CardTitle>Matriz de Comunicación</CardTitle>
                  <p className="text-sm text-muted-foreground">
                    Visualiza los canales, responsables y mensajes clave del SGI.
                  </p>
                </div>
                {canManage && (
                  <Button onClick={() => openMatrixDialog()} className="gap-2">
                    <Plus className="h-4 w-4" />
                    Agregar registro
                  </Button>
                )}
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="grid gap-3 lg:grid-cols-[2fr,1fr,1fr,1fr]">
                  <div className="relative">
                    <Search className="absolute left-3 top-3 h-4 w-4 text-muted-foreground" />
                    <Input
                      placeholder="Buscar por qué comunicar, a quién o categoría"
                      value={matrixSearch}
                      onChange={(event) => setMatrixSearch(event.target.value)}
                      className="pl-9"
                    />
                  </div>
                  <Select value={matrixCategoryFilter} onValueChange={setMatrixCategoryFilter}>
                    <SelectTrigger>
                      <SelectValue placeholder="Categoría" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">Todas las categorías</SelectItem>
                      {matrixCategories.map((category) => (
                        <SelectItem key={category} value={category}>
                          {category}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <Select value={matrixTypeFilter} onValueChange={setMatrixTypeFilter}>
                    <SelectTrigger>
                      <SelectValue placeholder="Tipo" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">Todos los tipos</SelectItem>
                      {matrixTypes.map((type) => (
                        <SelectItem key={type} value={type}>
                          {type}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <Select value={matrixSort} onValueChange={setMatrixSort}>
                    <SelectTrigger>
                      <SelectValue placeholder="Ordenar" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="category">Orden por categoría</SelectItem>
                      <SelectItem value="when">Orden por cuándo comunicar</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div className="rounded-xl border">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>Categoría</TableHead>
                        <TableHead>Tipo</TableHead>
                        <TableHead>¿Qué comunicar?</TableHead>
                        <TableHead>¿Cuándo comunicar?</TableHead>
                        <TableHead>¿A quién comunicar?</TableHead>
                        <TableHead>¿Cómo comunicar?</TableHead>
                        <TableHead>¿Quién comunica?</TableHead>
                        <TableHead>Responsable</TableHead>
                        <TableHead>Última actualización</TableHead>
                        {canManage && <TableHead className="text-right">Acciones</TableHead>}
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {matrixLoading ? (
                        <TableRow>
                          <TableCell colSpan={canManage ? 10 : 9} className="text-center text-sm text-muted-foreground">
                            Cargando matriz...
                          </TableCell>
                        </TableRow>
                      ) : filteredMatrix.length === 0 ? (
                        <TableRow>
                          <TableCell colSpan={canManage ? 10 : 9} className="text-center text-sm text-muted-foreground">
                            No hay registros para mostrar.
                          </TableCell>
                        </TableRow>
                      ) : (
                        filteredMatrix.map((record) => (
                          <TableRow key={record.id}>
                            <TableCell className="font-medium">{record.category}</TableCell>
                            <TableCell>{record.communication_type}</TableCell>
                            <TableCell>{record.what_to_communicate}</TableCell>
                            <TableCell>{record.when_to_communicate}</TableCell>
                            <TableCell>{record.who_to_communicate}</TableCell>
                            <TableCell>{record.how_to_communicate}</TableCell>
                            <TableCell>{record.who_communicates}</TableCell>
                            <TableCell>{record.followup_responsible ?? "-"}</TableCell>
                            <TableCell>{formatDate(record.updated_at)}</TableCell>
                            {canManage && (
                              <TableCell className="text-right">
                                <div className="flex justify-end gap-2">
                                  <Button
                                    variant="outline"
                                    size="icon"
                                    onClick={() => openMatrixDialog(record)}
                                  >
                                    <Pencil className="h-4 w-4" />
                                  </Button>
                                  <Button
                                    variant="destructive"
                                    size="icon"
                                    onClick={() => setMatrixDeleteTarget(record)}
                                  >
                                    <Trash2 className="h-4 w-4" />
                                  </Button>
                                </div>
                              </TableCell>
                            )}
                          </TableRow>
                        ))
                      )}
                    </TableBody>
                  </Table>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="posts" className="space-y-6">
            <Card>
              <CardHeader className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
                <div>
                  <CardTitle>Comunicados / Noticias</CardTitle>
                  <p className="text-sm text-muted-foreground">
                    Publicaciones oficiales para informar a todos los usuarios del SGI.
                  </p>
                </div>
                {canManage && (
                  <Button onClick={() => openPostDialog()} className="gap-2">
                    <Plus className="h-4 w-4" />
                    Nuevo comunicado
                  </Button>
                )}
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="grid gap-3 lg:grid-cols-[2fr,1fr,1fr,1fr,1fr]">
                  <div className="relative">
                    <Search className="absolute left-3 top-3 h-4 w-4 text-muted-foreground" />
                    <Input
                      placeholder="Buscar por título o contenido"
                      value={postSearch}
                      onChange={(event) => setPostSearch(event.target.value)}
                      className="pl-9"
                    />
                  </div>
                  <Select value={postCategoryFilter} onValueChange={setPostCategoryFilter}>
                    <SelectTrigger>
                      <SelectValue placeholder="Categoría" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">Todas las categorías</SelectItem>
                      {postCategories.map((category) => (
                        <SelectItem key={category} value={category}>
                          {category}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <Input
                    type="date"
                    value={postDateFrom}
                    onChange={(event) => setPostDateFrom(event.target.value)}
                  />
                  <Input
                    type="date"
                    value={postDateTo}
                    onChange={(event) => setPostDateTo(event.target.value)}
                  />
                  {canManage ? (
                    <Select value={postStatusFilter} onValueChange={setPostStatusFilter}>
                      <SelectTrigger>
                        <SelectValue placeholder="Estatus" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="all">Todos los estatus</SelectItem>
                        {statusOptions.map((status) => (
                          <SelectItem key={status.value} value={status.value}>
                            {status.label}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  ) : (
                    <div className="flex items-center gap-2 text-xs text-muted-foreground">
                      <Filter className="h-4 w-4" />
                      Solo publicados
                    </div>
                  )}
                </div>

                <div className="grid gap-4 lg:grid-cols-2">
                  {postLoading ? (
                    <Card className="lg:col-span-2">
                      <CardContent className="py-12 text-center text-sm text-muted-foreground">
                        Cargando comunicados...
                      </CardContent>
                    </Card>
                  ) : filteredPosts.length === 0 ? (
                    <Card className="lg:col-span-2">
                      <CardContent className="py-12 text-center text-sm text-muted-foreground">
                        No hay comunicados disponibles.
                      </CardContent>
                    </Card>
                  ) : (
                    filteredPosts.map((post) => (
                      <Card key={post.id} className="flex h-full flex-col">
                        <CardHeader className="space-y-3">
                          <div className="flex flex-wrap items-center justify-between gap-2">
                            <Badge variant="secondary" className="bg-emerald-50 text-emerald-600">
                              {post.category}
                            </Badge>
                            <Badge className={statusOptions.find((status) => status.value === post.status)?.badge}>
                              {statusOptions.find((status) => status.value === post.status)?.label}
                            </Badge>
                          </div>
                          <CardTitle className="text-lg">{post.title}</CardTitle>
                          <div className="flex flex-wrap items-center gap-2 text-xs text-muted-foreground">
                            <Calendar className="h-4 w-4" />
                            {formatDate(post.publish_date)}
                            <span className="text-slate-300">•</span>
                            {post.created_by ?? "SGI"}
                          </div>
                        </CardHeader>
                        <CardContent className="flex flex-1 flex-col justify-between gap-6">
                          <p className="text-sm text-slate-600">
                            {post.summary || post.content || "Sin resumen disponible."}
                          </p>
                          <div className="flex flex-wrap gap-2">
                            <Button variant="outline" onClick={() => setDetailPost(post)}>
                              Ver comunicado
                            </Button>
                            {canManage && (
                              <>
                                <Button variant="secondary" onClick={() => openPostDialog(post)}>
                                  <Pencil className="mr-2 h-4 w-4" />
                                  Editar
                                </Button>
                                <Button
                                  variant="destructive"
                                  onClick={() => setPostDeleteTarget(post)}
                                  className="gap-2"
                                >
                                  <Trash2 className="h-4 w-4" />
                                  Eliminar
                                </Button>
                              </>
                            )}
                          </div>
                        </CardContent>
                      </Card>
                    ))
                  )}
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="incidents" className="space-y-6">
            <Card>
              <CardHeader className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
                <div>
                  <CardTitle>Incidencias de Comunicación</CardTitle>
                  <p className="text-sm text-muted-foreground">
                    Registro y seguimiento de quejas, sugerencias, denuncias y otras comunicaciones.
                  </p>
                </div>
                {user && (
                  <Button onClick={() => openIncidentDialog()} className="gap-2">
                    <Plus className="h-4 w-4" />
                    Nueva incidencia
                  </Button>
                )}
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="grid gap-3 lg:grid-cols-[2fr,1fr,1fr,1fr]">
                  <div className="relative">
                    <Search className="absolute left-3 top-3 h-4 w-4 text-muted-foreground" />
                    <Input
                      placeholder="Buscar por descripción, plan o resultado"
                      value={incidentSearch}
                      onChange={(event) => setIncidentSearch(event.target.value)}
                      className="pl-9"
                    />
                  </div>
                  <Select value={incidentClassificationFilter} onValueChange={setIncidentClassificationFilter}>
                    <SelectTrigger>
                      <SelectValue placeholder="Clasificación" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">Todas las clasificaciones</SelectItem>
                      {incidentClassificationOptions.map((option) => (
                        <SelectItem key={option.value} value={option.value}>
                          {option.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <Select value={incidentSourceFilter} onValueChange={setIncidentSourceFilter}>
                    <SelectTrigger>
                      <SelectValue placeholder="Fuente" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">Todas las fuentes</SelectItem>
                      {incidentSources.map((source) => (
                        <SelectItem key={source} value={source}>
                          {source}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <Select value={incidentStatusFilter} onValueChange={setIncidentStatusFilter}>
                    <SelectTrigger>
                      <SelectValue placeholder="Estado" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">Todos los estados</SelectItem>
                      {Object.entries(incidentStatusLabels).map(([value, label]) => (
                        <SelectItem key={value} value={value}>
                          {label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div className="grid gap-3 lg:grid-cols-[1fr,1fr,1fr,1fr]">
                  <Select value={incidentStakeholderFilter} onValueChange={setIncidentStakeholderFilter}>
                    <SelectTrigger>
                      <SelectValue placeholder="Parte interesada" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">Todas las partes interesadas</SelectItem>
                      {incidentStakeholders.map((stakeholder) => (
                        <SelectItem key={stakeholder} value={stakeholder}>
                          {stakeholder}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <Input
                    type="date"
                    value={incidentTargetFrom}
                    onChange={(event) => setIncidentTargetFrom(event.target.value)}
                    placeholder="Objetivo desde"
                  />
                  <Input
                    type="date"
                    value={incidentTargetTo}
                    onChange={(event) => setIncidentTargetTo(event.target.value)}
                    placeholder="Objetivo hasta"
                  />
                  <div className="flex items-center gap-2 text-xs text-muted-foreground">
                    <Calendar className="h-4 w-4" />
                    Rango fecha objetivo
                  </div>
                </div>

                <div className="grid gap-3 lg:grid-cols-[1fr,1fr,1fr,1fr]">
                  <Input
                    type="date"
                    value={incidentStartFrom}
                    onChange={(event) => setIncidentStartFrom(event.target.value)}
                  />
                  <Input
                    type="date"
                    value={incidentStartTo}
                    onChange={(event) => setIncidentStartTo(event.target.value)}
                  />
                  <Input
                    type="date"
                    value={incidentCloseFrom}
                    onChange={(event) => setIncidentCloseFrom(event.target.value)}
                  />
                  <Input
                    type="date"
                    value={incidentCloseTo}
                    onChange={(event) => setIncidentCloseTo(event.target.value)}
                  />
                </div>

                <div className="rounded-xl border">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>Fuente</TableHead>
                        <TableHead>Parte interesada</TableHead>
                        <TableHead>Clasificación</TableHead>
                        <TableHead>Descripción</TableHead>
                        <TableHead>Evidencia</TableHead>
                        <TableHead>Plan de acción</TableHead>
                        <TableHead>Responsable</TableHead>
                        <TableHead>Fecha objetivo</TableHead>
                        <TableHead>Estado</TableHead>
                        <TableHead>Resultado</TableHead>
                        {canManage && <TableHead className="text-right">Acciones</TableHead>}
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {incidentLoading ? (
                        <TableRow>
                          <TableCell colSpan={canManage ? 11 : 10} className="text-center text-sm text-muted-foreground">
                            Cargando incidencias...
                          </TableCell>
                        </TableRow>
                      ) : filteredIncidents.length === 0 ? (
                        <TableRow>
                          <TableCell colSpan={canManage ? 11 : 10} className="text-center text-sm text-muted-foreground">
                            No hay incidencias registradas.
                          </TableCell>
                        </TableRow>
                      ) : (
                        filteredIncidents.map((incident) => {
                          const statusValue = resolveIncidentStatus(incident);
                          return (
                            <TableRow key={incident.id}>
                              <TableCell className="font-medium">
                                {incident.source_type}
                                {incident.source_type === "Otro" && incident.source_other_detail
                                  ? ` (${incident.source_other_detail})`
                                  : ""}
                              </TableCell>
                              <TableCell>{incident.stakeholder}</TableCell>
                              <TableCell>
                                {
                                  incidentClassificationOptions.find(
                                    (option) => option.value === incident.classification,
                                  )?.label
                                }
                              </TableCell>
                              <TableCell className="max-w-[260px] truncate">{incident.description}</TableCell>
                              <TableCell>
                                {incident.evidence_url ? (
                                  <a
                                    href={incident.evidence_url}
                                    target="_blank"
                                    rel="noreferrer"
                                    className="text-emerald-600 hover:underline"
                                  >
                                    Ver evidencia
                                  </a>
                                ) : (
                                  "-"
                                )}
                              </TableCell>
                              <TableCell className="max-w-[220px] truncate">{incident.action_plan ?? "-"}</TableCell>
                              <TableCell>{incident.plan_owner ?? "-"}</TableCell>
                              <TableCell>{formatDate(incident.target_date)}</TableCell>
                              <TableCell>
                                <Badge className={incidentStatusBadge[statusValue]}>
                                  {incidentStatusLabels[statusValue]}
                                </Badge>
                              </TableCell>
                              <TableCell className="max-w-[200px] truncate">{incident.result ?? "-"}</TableCell>
                              {canManage && (
                                <TableCell className="text-right">
                                  <div className="flex justify-end gap-2">
                                    <Button
                                      variant="outline"
                                      size="icon"
                                      onClick={() => openIncidentDialog(incident)}
                                    >
                                      <Pencil className="h-4 w-4" />
                                    </Button>
                                    <Button
                                      variant="destructive"
                                      size="icon"
                                      onClick={() => setIncidentDeleteTarget(incident)}
                                    >
                                      <Trash2 className="h-4 w-4" />
                                    </Button>
                                  </div>
                                </TableCell>
                              )}
                            </TableRow>
                          );
                        })
                      )}
                    </TableBody>
                  </Table>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>

      <Dialog open={matrixDialogOpen} onOpenChange={setMatrixDialogOpen}>
        <DialogContent className="max-w-3xl">
          <DialogHeader>
            <DialogTitle>{editingMatrix ? "Editar registro" : "Nuevo registro"}</DialogTitle>
          </DialogHeader>
          <div className="grid gap-4 md:grid-cols-2">
            <div className="space-y-2">
              <Label>Categoría</Label>
              <Input
                value={matrixForm.category}
                onChange={(event) => setMatrixForm((prev) => ({ ...prev, category: event.target.value }))}
              />
            </div>
            <div className="space-y-2">
              <Label>Tipo de comunicación</Label>
              <Input
                value={matrixForm.communication_type}
                onChange={(event) =>
                  setMatrixForm((prev) => ({ ...prev, communication_type: event.target.value }))
                }
              />
            </div>
            <div className="space-y-2 md:col-span-2">
              <Label>¿Qué comunicar?</Label>
              <Textarea
                value={matrixForm.what_to_communicate}
                onChange={(event) =>
                  setMatrixForm((prev) => ({ ...prev, what_to_communicate: event.target.value }))
                }
              />
            </div>
            <div className="space-y-2">
              <Label>¿Cuándo comunicar?</Label>
              <Input
                value={matrixForm.when_to_communicate}
                onChange={(event) =>
                  setMatrixForm((prev) => ({ ...prev, when_to_communicate: event.target.value }))
                }
              />
            </div>
            <div className="space-y-2">
              <Label>¿A quién comunicar?</Label>
              <Input
                value={matrixForm.who_to_communicate}
                onChange={(event) =>
                  setMatrixForm((prev) => ({ ...prev, who_to_communicate: event.target.value }))
                }
              />
            </div>
            <div className="space-y-2">
              <Label>¿Cómo comunicar?</Label>
              <Input
                value={matrixForm.how_to_communicate}
                onChange={(event) =>
                  setMatrixForm((prev) => ({ ...prev, how_to_communicate: event.target.value }))
                }
              />
            </div>
            <div className="space-y-2">
              <Label>¿Quién comunica?</Label>
              <Input
                value={matrixForm.who_communicates}
                onChange={(event) =>
                  setMatrixForm((prev) => ({ ...prev, who_communicates: event.target.value }))
                }
              />
            </div>
            <div className="space-y-2 md:col-span-2">
              <Label>Responsable de seguimiento (opcional)</Label>
              <Input
                value={matrixForm.followup_responsible}
                onChange={(event) =>
                  setMatrixForm((prev) => ({ ...prev, followup_responsible: event.target.value }))
                }
              />
            </div>
          </div>
          <div className="flex justify-end gap-2">
            <Button variant="outline" onClick={() => setMatrixDialogOpen(false)}>
              Cancelar
            </Button>
            <Button onClick={saveMatrix}>{editingMatrix ? "Guardar cambios" : "Crear registro"}</Button>
          </div>
        </DialogContent>
      </Dialog>

      <Dialog open={postDialogOpen} onOpenChange={setPostDialogOpen}>
        <DialogContent className="max-w-3xl">
          <DialogHeader>
            <DialogTitle>{editingPost ? "Editar comunicado" : "Nuevo comunicado"}</DialogTitle>
          </DialogHeader>
          <div className="max-h-[70vh] overflow-y-auto pr-2">
            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2 md:col-span-2">
                <Label>Título</Label>
                <Input
                  value={postForm.title}
                  onChange={(event) => setPostForm((prev) => ({ ...prev, title: event.target.value }))}
                />
              </div>
              <div className="space-y-2">
                <Label>Categoría</Label>
                <Input
                  value={postForm.category}
                  onChange={(event) => setPostForm((prev) => ({ ...prev, category: event.target.value }))}
                />
              </div>
              <div className="space-y-2">
                <Label>Fecha de publicación</Label>
                <Input
                  type="date"
                  value={postForm.publish_date}
                  onChange={(event) => setPostForm((prev) => ({ ...prev, publish_date: event.target.value }))}
                />
              </div>
              <div className="space-y-2">
                <Label>Estatus</Label>
                <Select
                  value={postForm.status}
                  onValueChange={(value) =>
                    setPostForm((prev) => ({ ...prev, status: value as CommunicationPost["status"] }))
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona un estatus" />
                  </SelectTrigger>
                  <SelectContent>
                    {statusOptions.map((status) => (
                      <SelectItem key={status.value} value={status.value}>
                        {status.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Vigencia inicio (opcional)</Label>
                <Input
                  type="date"
                  value={postForm.valid_from}
                  onChange={(event) => setPostForm((prev) => ({ ...prev, valid_from: event.target.value }))}
                />
              </div>
              <div className="space-y-2">
                <Label>Vigencia fin (opcional)</Label>
                <Input
                  type="date"
                  value={postForm.valid_to}
                  onChange={(event) => setPostForm((prev) => ({ ...prev, valid_to: event.target.value }))}
                />
              </div>
              <div className="space-y-2 md:col-span-2">
                <Label>Resumen</Label>
                <Textarea
                  value={postForm.summary}
                  onChange={(event) => setPostForm((prev) => ({ ...prev, summary: event.target.value }))}
                />
              </div>
              <div className="space-y-2 md:col-span-2">
                <Label>Contenido</Label>
                <Textarea
                  value={postForm.content}
                  onChange={(event) => setPostForm((prev) => ({ ...prev, content: event.target.value }))}
                />
              </div>
              <div className="space-y-2 md:col-span-2">
                <Label>Adjuntos (PDF, imágenes, etc.)</Label>
                <div className="flex items-center gap-3">
                  <Input type="file" multiple onChange={(event) => setPostFiles(event.target.files)} />
                  <Upload className="h-4 w-4 text-muted-foreground" />
                </div>
              </div>
            </div>
          </div>
          <div className="flex justify-end gap-2">
            <Button variant="outline" onClick={() => setPostDialogOpen(false)}>
              Cancelar
            </Button>
            <Button onClick={savePost}>{editingPost ? "Guardar cambios" : "Crear comunicado"}</Button>
          </div>
        </DialogContent>
      </Dialog>

      <Dialog open={incidentDialogOpen} onOpenChange={setIncidentDialogOpen}>
        <DialogContent className="max-w-4xl">
          <DialogHeader>
            <DialogTitle>{editingIncident ? "Editar incidencia" : "Nueva incidencia"}</DialogTitle>
          </DialogHeader>
          <div className="max-h-[70vh] overflow-y-auto pr-2">
            <div className="grid gap-4 md:grid-cols-2">
              <div className="space-y-2">
                <Label>Fuente de la comunicación</Label>
                <Select
                  value={incidentForm.source_type}
                  onValueChange={(value) => setIncidentForm((prev) => ({ ...prev, source_type: value }))}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona una fuente" />
                  </SelectTrigger>
                  <SelectContent>
                    {incidentSourceOptions.map((source) => (
                      <SelectItem key={source} value={source}>
                        {source}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Parte interesada</Label>
                <Input
                  value={incidentForm.stakeholder}
                  onChange={(event) => setIncidentForm((prev) => ({ ...prev, stakeholder: event.target.value }))}
                />
              </div>
              {incidentForm.source_type === "Otro" && (
                <div className="space-y-2 md:col-span-2">
                  <Label>Especificar fuente</Label>
                  <Input
                    value={incidentForm.source_other_detail}
                    onChange={(event) =>
                      setIncidentForm((prev) => ({ ...prev, source_other_detail: event.target.value }))
                    }
                  />
                </div>
              )}
              <div className="space-y-2">
                <Label>Clasificación</Label>
                <Select
                  value={incidentForm.classification}
                  onValueChange={(value) =>
                    setIncidentForm((prev) => ({
                      ...prev,
                      classification: value as CommunicationIncident["classification"],
                    }))
                  }
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Selecciona una clasificación" />
                  </SelectTrigger>
                  <SelectContent>
                    {incidentClassificationOptions.map((option) => (
                      <SelectItem key={option.value} value={option.value}>
                        {option.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label>Estado calculado</Label>
                <div className="flex items-center gap-2 rounded-md border border-slate-200 px-3 py-2 text-sm">
                  <Badge className={incidentStatusBadge[resolveIncidentStatus(incidentForm)]}>
                    {incidentStatusLabels[resolveIncidentStatus(incidentForm)]}
                  </Badge>
                  <span className="text-xs text-muted-foreground">Automático según fechas y responsable.</span>
                </div>
              </div>
              <div className="space-y-2 md:col-span-2">
                <Label>Descripción</Label>
                <Textarea
                  value={incidentForm.description}
                  onChange={(event) => setIncidentForm((prev) => ({ ...prev, description: event.target.value }))}
                />
              </div>
              <div className="space-y-2 md:col-span-2">
                <Label>Evidencia (link o archivo)</Label>
                <div className="grid gap-3 md:grid-cols-[2fr,1fr]">
                  <Input
                    placeholder="https://"
                    value={incidentForm.evidence_url}
                    onChange={(event) => setIncidentForm((prev) => ({ ...prev, evidence_url: event.target.value }))}
                  />
                  <div className="flex items-center gap-3">
                    <Input
                      type="file"
                      onChange={(event) => setIncidentEvidenceFile(event.target.files?.[0] ?? null)}
                    />
                    <Upload className="h-4 w-4 text-muted-foreground" />
                  </div>
                </div>
                {editingIncident?.evidence_url && (
                  <a
                    href={editingIncident.evidence_url}
                    target="_blank"
                    rel="noreferrer"
                    className="text-xs text-emerald-600 hover:underline"
                  >
                    Ver evidencia actual
                  </a>
                )}                
              </div>
              <div className="space-y-2 md:col-span-2">
                <Label>Plan de acción</Label>
                <Textarea
                  value={incidentForm.action_plan}
                  onChange={(event) => setIncidentForm((prev) => ({ ...prev, action_plan: event.target.value }))}
                  disabled={!canManage}
                />
              </div>
              <div className="space-y-2">
                <Label>Responsable del plan</Label>
                <Input
                  value={incidentForm.plan_owner}
                  onChange={(event) => setIncidentForm((prev) => ({ ...prev, plan_owner: event.target.value }))}
                  disabled={!canManage}
                />
              </div>
              <div className="space-y-2">
                <Label>Fecha objetivo</Label>
                <Input
                  type="date"
                  value={incidentForm.target_date}
                  onChange={(event) => setIncidentForm((prev) => ({ ...prev, target_date: event.target.value }))}
                  disabled={!canManage}
                />
              </div>
              <div className="space-y-2">
                <Label>Fecha de inicio</Label>
                <Input
                  type="date"
                  value={incidentForm.start_date}
                  onChange={(event) => setIncidentForm((prev) => ({ ...prev, start_date: event.target.value }))}
                  disabled={!canManage}
                />
              </div>
              <div className="space-y-2">
                <Label>Fecha de cierre</Label>
                <Input
                  type="date"
                  value={incidentForm.close_date}
                  onChange={(event) => setIncidentForm((prev) => ({ ...prev, close_date: event.target.value }))}
                  disabled={!canManage}
                />
              </div>
              <div className="space-y-2 md:col-span-2">
                <Label>Resultado / Conclusión</Label>
                <Textarea
                  value={incidentForm.result}
                  onChange={(event) => setIncidentForm((prev) => ({ ...prev, result: event.target.value }))}
                  disabled={!canManage}
                />
              </div>
              <div className="space-y-2">
                <Label>Fecha de confirmación</Label>
                <Input
                  type="date"
                  value={incidentForm.confirmation_date}
                  onChange={(event) => setIncidentForm((prev) => ({ ...prev, confirmation_date: event.target.value }))}
                  disabled={!canManage}
                />
              </div>
              {!canManage && (
                <div className="flex items-center gap-2 text-xs text-muted-foreground md:col-span-2">
                  <Filter className="h-4 w-4" />
                  Solo managers pueden asignar responsables, fechas y cerrar incidencias.
                </div>
              )}
            </div>
          </div>
          <div className="flex justify-end gap-2">
            <Button variant="outline" onClick={() => setIncidentDialogOpen(false)}>
              Cancelar
            </Button>
            <Button onClick={saveIncident}>{editingIncident ? "Guardar cambios" : "Registrar incidencia"}</Button>
          </div>
        </DialogContent>
      </Dialog>

      <Dialog open={!!detailPost} onOpenChange={() => setDetailPost(null)}>
        <DialogContent className="max-w-3xl">
          <DialogHeader>
            <DialogTitle>{detailPost?.title}</DialogTitle>
          </DialogHeader>
          {detailPost && (
            <div className="space-y-4">
              <div className="flex flex-wrap items-center gap-2">
                <Badge variant="secondary" className="bg-emerald-50 text-emerald-600">
                  {detailPost.category}
                </Badge>
                <Badge className={statusOptions.find((status) => status.value === detailPost.status)?.badge}>
                  {statusOptions.find((status) => status.value === detailPost.status)?.label}
                </Badge>
                <span className="text-xs text-muted-foreground">
                  {formatDate(detailPost.publish_date)} · {detailPost.created_by ?? "SGI"}
                </span>
              </div>
              <div className="space-y-2">
                <p className="text-sm text-muted-foreground">Resumen</p>
                <p className="text-sm text-slate-700">{detailPost.summary || "Sin resumen."}</p>
              </div>
              <div className="space-y-2">
                <p className="text-sm text-muted-foreground">Contenido</p>
                <p className="whitespace-pre-line text-sm text-slate-700">
                  {detailPost.content || "Sin contenido."}
                </p>
              </div>
              <div className="space-y-2">
                <p className="text-sm text-muted-foreground">Adjuntos</p>
                {postAttachments[detailPost.id]?.length ? (
                  <div className="space-y-3">
                    {postAttachments[detailPost.id].some(isImageAttachment) && (
                      <div className="grid gap-3 sm:grid-cols-2">
                        {postAttachments[detailPost.id]
                          .filter(isImageAttachment)
                          .map((attachment) => {
                            const publicUrl = getPublicFileUrl(attachment.file_path);
                            return (
                              <div
                                key={attachment.id}
                                className="overflow-hidden rounded-lg border border-slate-200 bg-slate-50"
                              >
                                <img
                                  src={publicUrl}
                                  alt={attachment.file_name}
                                  className="h-40 w-full object-cover"
                                  loading="lazy"
                                />
                                <div className="flex items-center justify-between gap-2 px-3 py-2 text-xs text-slate-600">
                                  <span className="truncate">{attachment.file_name}</span>
                                  <a
                                    href={publicUrl}
                                    target="_blank"
                                    rel="noreferrer"
                                    className="text-emerald-600 hover:underline"
                                  >
                                    Descargar
                                  </a>
                                </div>
                              </div>
                            );
                          })}
                      </div>
                    )}
                    {postAttachments[detailPost.id].some((attachment) => !isImageAttachment(attachment)) && (
                      <ul className="space-y-2">
                        {postAttachments[detailPost.id]
                          .filter((attachment) => !isImageAttachment(attachment))
                          .map((attachment) => (
                            <li key={attachment.id} className="flex items-center gap-2 text-sm">
                              <FileText className="h-4 w-4 text-emerald-600" />
                              <a
                                href={getPublicFileUrl(attachment.file_path)}
                                target="_blank"
                                rel="noreferrer"
                                className="text-emerald-600 hover:underline"
                              >
                                {attachment.file_name}
                              </a>
                            </li>
                          ))}
                      </ul>
                    )}
                  </div>
                ) : (
                  <p className="text-sm text-muted-foreground">Sin adjuntos.</p>
                )}
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>

      <AlertDialog open={!!matrixDeleteTarget} onOpenChange={() => setMatrixDeleteTarget(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>¿Eliminar este registro de la matriz?</AlertDialogTitle>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancelar</AlertDialogCancel>
            <AlertDialogAction onClick={confirmDeleteMatrix}>Eliminar</AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      <AlertDialog open={!!postDeleteTarget} onOpenChange={() => setPostDeleteTarget(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>¿Eliminar este comunicado?</AlertDialogTitle>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancelar</AlertDialogCancel>
            <AlertDialogAction onClick={confirmDeletePost}>Eliminar</AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>

      <AlertDialog open={!!incidentDeleteTarget} onOpenChange={() => setIncidentDeleteTarget(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>¿Eliminar esta incidencia?</AlertDialogTitle>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancelar</AlertDialogCancel>
            <AlertDialogAction onClick={confirmDeleteIncident}>Eliminar</AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </SGIDashboardLayout>
  );
}