"use client";

import { useEffect, useState } from 'react';
import { useRouter } from 'next/navigation';
import Image from 'next/image';
import { getClientPortalAuth, isMeiwaCorePortalAuth } from '@/app/client-portal-auth';
import { supabase } from '@/lib/supabase';
import { User } from '@supabase/supabase-js';
import { LoginForm } from '@/components/auth/login-form';
import { RegisterForm } from '@/components/auth/register-form';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import MeiwaTasksLogo from '@/app/meiwa-core/LOGO/LOGOMTMS.png';
import MeiwaSgiLogo from '@/app/meiwa-core/LOGO/LOGOMEIWASGI.png';
import Meiwa5sLogo from '@/app/meiwa-core/LOGO/LOGOMEIWA5S.png';

export default function Home() {
  const [user, setUser] = useState<User | null>(null);
  const [loading, setLoading] = useState(true);
  const [portalAuthorized, setPortalAuthorized] = useState(false);
  const router = useRouter();

  useEffect(() => {
    const portalAuth = getClientPortalAuth();
    if (!isMeiwaCorePortalAuth(portalAuth)) {
      router.replace('/client-authlog');
      return;
    }
    setPortalAuthorized(true);

    const getSession = async () => {
      const { data: { session } } = await supabase.auth.getSession();
      setUser(session?.user ?? null);
      setLoading(false);
      
      if (session?.user) {
        router.push('/meiwa-core/selector');
      }
    };

    getSession();

    const {
      data: { subscription },
    } = supabase.auth.onAuthStateChange((_event, session) => {
      setUser(session?.user ?? null);
      if (session?.user) {
        router.push('/meiwa-core/selector');
      }
    });

    return () => subscription.unsubscribe();
  }, [router]);

  if (!portalAuthorized || loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-blue-50 to-indigo-100">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (user) {
    return null; // Will redirect to selector
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 via-indigo-50 to-purple-50">
      {/* Header */}
      <div className="relative overflow-hidden">
        <div className="absolute inset-0 bg-gradient-to-r from-blue-600 to-purple-600 opacity-10"></div>
        <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 pt-20 pb-16">
          <div className="text-center">
            <div className="flex items-center justify-center mb-8">
              <Image
                src="/meiwa-core.png"
                alt="Meiwa Core logo"
                width={200}
                height={200}
                priority
                className="h-60 w-60 object-contain"
              />
            </div>
            <h1 className="text-4xl sm:text-5xl font-bold text-gray-900 mb-4">
              MEIWA CORE
            </h1>
            <p className="text-xl text-gray-600 max-w-2xl mx-auto mb-8">
              Una plataforma central de gestión que integra múltiples aplicaciones como
              MEIWA TASKS, MEIWA SGI, MEIWA 5S, MEIWA POINTS, MEIWA DIAGRAMS y más.
            </p>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6 max-w-4xl mx-auto">
              <div className="text-center">
                <div className="bg-white rounded-full p-3 w-20 h-20 mx-auto mb-4 shadow-md flex items-center justify-center">
                  <Image
                    src={MeiwaTasksLogo}
                    alt="Logo MEIWA TASKS"
                    width={56}
                    height={56}
                    className="h-14 w-14 object-contain"
                  />
                </div>
                <h3 className="font-semibold text-gray-900">MEIWA TASKS</h3>
                <p className="text-gray-600">Gestión de tareas y productividad</p>
              </div>
              <div className="text-center">
                <div className="bg-white rounded-full p-3 w-20 h-20 mx-auto mb-4 shadow-md flex items-center justify-center">
                  <Image
                    src={MeiwaSgiLogo}
                    alt="Logo MEIWA SGI"
                    width={56}
                    height={56}
                    className="h-14 w-14 object-contain"
                  />
                </div>
                <h3 className="font-semibold text-gray-900">MEIWA SGI</h3>
                <p className="text-gray-600">Sistema de gestión integral</p>
              </div>
              <div className="text-center">
                <div className="bg-white rounded-full p-3 w-20 h-20 mx-auto mb-4 shadow-md flex items-center justify-center">
                  <Image
                    src={Meiwa5sLogo}
                    alt="Logo MEIWA 5S"
                    width={56}
                    height={56}
                    className="h-14 w-14 object-contain"
                  />
                </div>
                <h3 className="font-semibold text-gray-900">MEIWA 5S</h3>
                <p className="text-gray-600">Estandarización y mejora continua</p>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Auth Forms */}
      <div className="max-w-md mx-auto px-4 pb-20">
        <Card className="backdrop-blur-sm bg-white/80 border-0 shadow-xl">
          <CardHeader className="text-center">
            <CardTitle className="text-2xl font-bold">¡Comienza ahora!</CardTitle>
            <CardDescription>
              Únete para comenzar a organizar tus tareas de manera eficiente.
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Tabs defaultValue="login" className="w-full">
              <TabsList className="grid w-full grid-cols-2">
                <TabsTrigger value="login">Iniciar Sesión</TabsTrigger>
                <TabsTrigger value="register">Registrarse</TabsTrigger>
              </TabsList>
              <TabsContent value="login" className="mt-6">
                <LoginForm />
              </TabsContent>
              <TabsContent value="register" className="mt-6">
                <RegisterForm />
              </TabsContent>
            </Tabs>
          </CardContent>
        </Card>
      </div>

      {/* Footer */}
      <footer className="bg-gray-900 text-white py-12">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center">
            <div className="flex items-center justify-center mb-4">
              <Image
                src="/meiwa-core.png"
                alt="Meiwa Core logo"
                width={36}
                height={36}
                className="h-9 w-9 object-contain mr-2"
              />
              <span className="text-2xl font-bold">MEIWA CORE</span>
            </div>
            <p className="text-gray-400 mb-4">
              MEIWAMOLD MEXICO S.A. DE C.V.
            </p>
            <p className="text-gray-500 text-sm">
              2026 © MEIWA CORE | TODOS LOS DERECHOS RESERVADOS
            </p>
          </div>
        </div>
      </footer>
    </div>
  );
}