"use client";

import { useCallback, useEffect, useMemo, useState } from "react";
import { toast } from "sonner";
import { ChevronDown, ChevronUp, Plus, Trash2 } from "lucide-react";

import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { Textarea } from "@/components/ui/textarea";
import { supabase } from "@/lib/supabase";

const categoryOptions = [
  { value: "environment", label: "Medio Ambiente" },
  { value: "quality", label: "Calidad" },
  { value: "safety", label: "Seguridad" },
  { value: "general", label: "General" },
];

type ContentBlockType = "text" | "link" | "pdf" | "video";

type ContentBlock = {
  type: ContentBlockType;
  value: string;
};

type LessonFormState = {
  id?: string;
  title: string;
  is_active: boolean;
  order_index?: number | null;
  content_blocks: ContentBlock[];
};

type QuizQuestion = {
  id?: string;
  question: string;
  options: string[];
  correctIndex: number;
  points: number;
};

type ModuleFormState = {
  title: string;
  description: string;
  category: string;
  points_available: string;
  pass_score: string;
  is_active: boolean;
  order_index: string;
};

const defaultModuleState: ModuleFormState = {
  title: "",
  description: "",
  category: "general",
  points_available: "100",
  pass_score: "",
  is_active: true,
  order_index: "",
};

const createLesson = (): LessonFormState => ({
  title: "",
  is_active: true,
  order_index: null,
  content_blocks: [{ type: "text", value: "" }],
});

const createQuestion = (): QuizQuestion => ({
  question: "",
  options: [""],
  correctIndex: 0,
  points: 10,
});

const normalizeContentBlocks = (value: unknown): ContentBlock[] => {
  if (!Array.isArray(value)) return [];
  return value
    .filter((item) => item && typeof item === "object")
    .map((block) => {
      const typedBlock = block as { type?: ContentBlockType; value?: string };
      return {
        type: typedBlock.type ?? "text",
        value: typedBlock.value ?? "",
      };
    });
};

const normalizeQuizQuestions = (value: unknown): QuizQuestion[] => {
  if (!value || typeof value !== "object") return [];
  const maybeQuestions = (value as { questions?: unknown }).questions;
  if (!Array.isArray(maybeQuestions)) return [];
  return maybeQuestions.map((question) => {
    const typed = question as {
      question?: string;
      options?: string[];
      correctIndex?: number;
      points?: number;
    };
    return {
      question: typed.question ?? "",
      options: typed.options ?? [""],
      correctIndex: typed.correctIndex ?? 0,
      points: typed.points ?? 10,
    };
  });
};

type AwarenessModuleFormProps = {
  mode: "create" | "edit";
  moduleId?: string;
  onSuccess: () => void;
};

export function AwarenessModuleForm({ mode, moduleId, onSuccess }: AwarenessModuleFormProps) {
  const [moduleState, setModuleState] = useState<ModuleFormState>(defaultModuleState);
  const [lessons, setLessons] = useState<LessonFormState[]>([createLesson()]);
  const [questions, setQuestions] = useState<QuizQuestion[]>([createQuestion()]);
  const [loading, setLoading] = useState(mode === "edit");
  const [saving, setSaving] = useState(false);

  const isEdit = mode === "edit";

  const lessonCount = lessons.length;
  const questionCount = questions.length;

  const loadModule = useCallback(async () => {
    if (!moduleId) return;
    setLoading(true);

    const [moduleResponse, lessonsResponse, quizResponse] = await Promise.all([
      supabase.from("awareness_modules").select("*").eq("id", moduleId).single(),
      supabase
        .from("awareness_lessons")
        .select("*")
        .eq("module_id", moduleId)
        .order("order_index", { ascending: true }),
      supabase.from("awareness_quizzes").select("*").eq("module_id", moduleId).maybeSingle(),
    ]);

    if (moduleResponse.error) {
      toast.error("No se pudo cargar el módulo.");
      console.error(moduleResponse.error);
      setLoading(false);
      return;
    }

    const moduleData = moduleResponse.data;
    setModuleState({
      title: moduleData.title ?? "",
      description: moduleData.description ?? "",
      category: moduleData.category ?? "general",
      points_available: moduleData.points_available?.toString() ?? "0",
      pass_score: moduleData.pass_score?.toString() ?? "",
      is_active: moduleData.is_active ?? true,
      order_index: moduleData.order_index?.toString() ?? "",
    });

    if (lessonsResponse.error) {
      toast.error("No se pudieron cargar las lecciones.");
      console.error(lessonsResponse.error);
    }

    const lessonData = lessonsResponse.data ?? [];
    setLessons(
      lessonData.length
        ? lessonData.map((lesson) => ({
            id: lesson.id,
            title: lesson.title ?? "",
            is_active: lesson.is_active ?? true,
            order_index: lesson.order_index,
            content_blocks: normalizeContentBlocks(lesson.content_blocks),
          }))
        : [createLesson()],
    );

    if (quizResponse.error) {
      toast.error("No se pudo cargar el cuestionario.");
      console.error(quizResponse.error);
    }

    const quizQuestions = normalizeQuizQuestions(quizResponse.data?.quiz_json);
    setQuestions(quizQuestions.length ? quizQuestions : [createQuestion()]);

    setLoading(false);
  }, [moduleId]);

  useEffect(() => {
    if (isEdit) {
      void loadModule();
    }
  }, [isEdit, loadModule]);

  const updateModuleField = <K extends keyof ModuleFormState>(key: K, value: ModuleFormState[K]) => {
    setModuleState((prev) => ({
      ...prev,
      [key]: value,
    }));
  };

  const addLesson = () => setLessons((prev) => [...prev, createLesson()]);

  const updateLesson = (index: number, patch: Partial<LessonFormState>) => {
    setLessons((prev) =>
      prev.map((lesson, lessonIndex) =>
        lessonIndex === index
          ? {
              ...lesson,
              ...patch,
            }
          : lesson,
      ),
    );
  };

  const moveLesson = (index: number, direction: "up" | "down") => {
    setLessons((prev) => {
      const targetIndex = direction === "up" ? index - 1 : index + 1;
      if (targetIndex < 0 || targetIndex >= prev.length) return prev;
      const next = [...prev];
      const [removed] = next.splice(index, 1);
      next.splice(targetIndex, 0, removed);
      return next;
    });
  };

  const removeLesson = (index: number) => {
    setLessons((prev) => prev.filter((_, lessonIndex) => lessonIndex !== index));
  };

  const addContentBlock = (lessonIndex: number) => {
    setLessons((prev) =>
      prev.map((lesson, index) =>
        index === lessonIndex
          ? {
              ...lesson,
              content_blocks: [...lesson.content_blocks, { type: "text", value: "" }],
            }
          : lesson,
      ),
    );
  };

  const updateContentBlock = (
    lessonIndex: number,
    blockIndex: number,
    patch: Partial<ContentBlock>,
  ) => {
    setLessons((prev) =>
      prev.map((lesson, index) => {
        if (index !== lessonIndex) return lesson;
        return {
          ...lesson,
          content_blocks: lesson.content_blocks.map((block, innerIndex) =>
            innerIndex === blockIndex ? { ...block, ...patch } : block,
          ),
        };
      }),
    );
  };

  const removeContentBlock = (lessonIndex: number, blockIndex: number) => {
    setLessons((prev) =>
      prev.map((lesson, index) => {
        if (index !== lessonIndex) return lesson;
        return {
          ...lesson,
          content_blocks: lesson.content_blocks.filter((_, innerIndex) => innerIndex !== blockIndex),
        };
      }),
    );
  };

  const addQuestion = () => setQuestions((prev) => [...prev, createQuestion()]);

  const updateQuestion = (index: number, patch: Partial<QuizQuestion>) => {
    setQuestions((prev) =>
      prev.map((question, questionIndex) =>
        questionIndex === index
          ? {
              ...question,
              ...patch,
            }
          : question,
      ),
    );
  };

  const addOption = (questionIndex: number) => {
    setQuestions((prev) =>
      prev.map((question, index) =>
        index === questionIndex
          ? {
              ...question,
              options: [...question.options, ""],
            }
          : question,
      ),
    );
  };

  const updateOption = (questionIndex: number, optionIndex: number, value: string) => {
    setQuestions((prev) =>
      prev.map((question, index) => {
        if (index !== questionIndex) return question;
        const nextOptions = question.options.map((option, optIndex) =>
          optIndex === optionIndex ? value : option,
        );
        const correctIndex = Math.min(question.correctIndex, nextOptions.length - 1);
        return {
          ...question,
          options: nextOptions,
          correctIndex,
        };
      }),
    );
  };

  const removeOption = (questionIndex: number, optionIndex: number) => {
    setQuestions((prev) =>
      prev.map((question, index) => {
        if (index !== questionIndex) return question;
        const nextOptions = question.options.filter((_, optIndex) => optIndex !== optionIndex);
        const correctIndex = Math.min(question.correctIndex, nextOptions.length - 1);
        return {
          ...question,
          options: nextOptions.length ? nextOptions : [""],
          correctIndex: correctIndex < 0 ? 0 : correctIndex,
        };
      }),
    );
  };

  const totalQuizScore = useMemo(() => {
    return questions.reduce((total, question) => total + (Number(question.points) || 0), 0);
  }, [questions]);

  const handleSave = async () => {
    if (!moduleState.title.trim()) {
      toast.error("El título del módulo es obligatorio.");
      return;
    }

    if (!moduleState.category) {
      toast.error("Selecciona la categoría del módulo.");
      return;
    }

    if (!moduleState.points_available.trim()) {
      toast.error("Ingresa los puntos disponibles.");
      return;
    }

    setSaving(true);

    const payload = {
      title: moduleState.title.trim(),
      description: moduleState.description.trim() || null,
      category: moduleState.category,
      points_available: Number(moduleState.points_available) || 0,
      pass_score: moduleState.pass_score ? Number(moduleState.pass_score) : null,
      is_active: moduleState.is_active,
      order_index: moduleState.order_index ? Number(moduleState.order_index) : null,
    };

    let moduleRecordId = moduleId;

    if (isEdit && moduleRecordId) {
      const { error } = await supabase.from("awareness_modules").update(payload).eq("id", moduleRecordId);
      if (error) {
        toast.error("No se pudieron guardar los cambios del módulo.");
        console.error(error);
        setSaving(false);
        return;
      }
    } else {
      const { data, error } = await supabase
        .from("awareness_modules")
        .insert(payload)
        .select("id")
        .single();
      if (error || !data) {
        toast.error("No se pudo crear el módulo.");
        console.error(error);
        setSaving(false);
        return;
      }
      moduleRecordId = data.id;
    }

    if (!moduleRecordId) {
      toast.error("No se pudo determinar el módulo guardado.");
      setSaving(false);
      return;
    }

    if (isEdit) {
      const [lessonDelete, quizDelete] = await Promise.all([
        supabase.from("awareness_lessons").delete().eq("module_id", moduleRecordId),
        supabase.from("awareness_quizzes").delete().eq("module_id", moduleRecordId),
      ]);

      if (lessonDelete.error) {
        toast.error("No se pudieron actualizar las lecciones.");
        console.error(lessonDelete.error);
        setSaving(false);
        return;
      }

      if (quizDelete.error) {
        toast.error("No se pudo actualizar el cuestionario.");
        console.error(quizDelete.error);
        setSaving(false);
        return;
      }
    }

    const lessonPayload = lessons.map((lesson, index) => ({
      module_id: moduleRecordId,
      title: lesson.title.trim() || `Lección ${index + 1}`,
      content_blocks: lesson.content_blocks,
      order_index: lesson.order_index ?? index + 1,
      is_active: lesson.is_active,
    }));

    if (lessonPayload.length) {
      const { error } = await supabase.from("awareness_lessons").insert(lessonPayload);
      if (error) {
        toast.error("No se pudieron guardar las lecciones.");
        console.error(error);
        setSaving(false);
        return;
      }
    }

    if (questions.length) {
      const quizPayload = {
        module_id: moduleRecordId,
        quiz_json: {
          questions: questions.map((question) => ({
            question: question.question,
            options: question.options,
            correctIndex: question.correctIndex,
            points: question.points,
          })),
        },
        max_score: totalQuizScore > 0 ? totalQuizScore : 100,
      };

      const { error } = await supabase.from("awareness_quizzes").insert(quizPayload);
      if (error) {
        toast.error("No se pudo guardar el cuestionario.");
        console.error(error);
        setSaving(false);
        return;
      }
    }

    toast.success(isEdit ? "Módulo actualizado" : "Módulo creado");
    setSaving(false);
    onSuccess();
  };

  if (loading) {
    return (
      <Card>
        <CardContent className="py-10 text-center text-sm text-muted-foreground">
          Cargando datos del módulo...
        </CardContent>
      </Card>
    );
  }

  return (
    <div className="space-y-6">
      <Card>
        <CardHeader>
          <CardTitle>Datos generales</CardTitle>
        </CardHeader>
        <CardContent className="grid gap-6 md:grid-cols-2">
          <div className="space-y-2">
            <Label>Título</Label>
            <Input
              value={moduleState.title}
              onChange={(event) => updateModuleField("title", event.target.value)}
              placeholder="Título del módulo"
            />
          </div>
          <div className="space-y-2">
            <Label>Categoría</Label>
            <Select
              value={moduleState.category}
              onValueChange={(value) => updateModuleField("category", value)}
            >
              <SelectTrigger>
                <SelectValue placeholder="Selecciona una categoría" />
              </SelectTrigger>
              <SelectContent>
                {categoryOptions.map((option) => (
                  <SelectItem key={option.value} value={option.value}>
                    {option.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
          <div className="space-y-2 md:col-span-2">
            <Label>Descripción</Label>
            <Textarea
              value={moduleState.description}
              onChange={(event) => updateModuleField("description", event.target.value)}
              placeholder="Descripción del módulo"
            />
          </div>
          <div className="space-y-2">
            <Label>Puntos disponibles</Label>
            <Input
              type="number"
              min="0"
              value={moduleState.points_available}
              onChange={(event) => updateModuleField("points_available", event.target.value)}
            />
          </div>
          <div className="space-y-2">
            <Label>Pass score (opcional)</Label>
            <Input
              type="number"
              min="0"
              max="100"
              value={moduleState.pass_score}
              onChange={(event) => updateModuleField("pass_score", event.target.value)}
              placeholder="Ej: 80"
            />
          </div>
          <div className="space-y-2">
            <Label>Orden</Label>
            <Input
              type="number"
              min="0"
              value={moduleState.order_index}
              onChange={(event) => updateModuleField("order_index", event.target.value)}
              placeholder="1"
            />
          </div>
          <div className="flex items-center gap-3">
            <Switch
              checked={moduleState.is_active}
              onCheckedChange={(checked) => updateModuleField("is_active", checked)}
            />
            <Label>Activo</Label>
          </div>
        </CardContent>
      </Card>

      <Card>
        <CardHeader className="flex flex-row items-center justify-between">
          <div>
            <CardTitle>Lecciones</CardTitle>
            <p className="text-sm text-muted-foreground">{lessonCount} lecciones configuradas</p>
          </div>
          <Button variant="outline" onClick={addLesson}>
            <Plus className="mr-2 h-4 w-4" />
            Agregar lección
          </Button>
        </CardHeader>
        <CardContent className="space-y-6">
          {lessons.map((lesson, lessonIndex) => (
            <Card key={`lesson-${lessonIndex}`} className="border-dashed">
              <CardHeader className="flex flex-row items-start justify-between gap-4">
                <div>
                  <CardTitle className="text-base">Lección {lessonIndex + 1}</CardTitle>
                  <p className="text-sm text-muted-foreground">Configura el contenido principal.</p>
                </div>
                <div className="flex items-center gap-2">
                  <Button
                    size="icon"
                    variant="ghost"
                    onClick={() => moveLesson(lessonIndex, "up")}
                    disabled={lessonIndex === 0}
                  >
                    <ChevronUp className="h-4 w-4" />
                  </Button>
                  <Button
                    size="icon"
                    variant="ghost"
                    onClick={() => moveLesson(lessonIndex, "down")}
                    disabled={lessonIndex === lessons.length - 1}
                  >
                    <ChevronDown className="h-4 w-4" />
                  </Button>
                  <Button
                    size="icon"
                    variant="ghost"
                    onClick={() => removeLesson(lessonIndex)}
                    disabled={lessons.length === 1}
                  >
                    <Trash2 className="h-4 w-4" />
                  </Button>
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid gap-4 md:grid-cols-2">
                  <div className="space-y-2">
                    <Label>Título de la lección</Label>
                    <Input
                      value={lesson.title}
                      onChange={(event) =>
                        updateLesson(lessonIndex, { title: event.target.value })
                      }
                    />
                  </div>
                  <div className="flex items-center gap-3">
                    <Switch
                      checked={lesson.is_active}
                      onCheckedChange={(checked) =>
                        updateLesson(lessonIndex, { is_active: checked })
                      }
                    />
                    <Label>Activa</Label>
                  </div>
                </div>

                <div className="space-y-3">
                  <div className="flex items-center justify-between">
                    <Label>Contenido por bloques</Label>
                    <Button variant="ghost" size="sm" onClick={() => addContentBlock(lessonIndex)}>
                      <Plus className="mr-2 h-4 w-4" />
                      Agregar bloque
                    </Button>
                  </div>
                  <div className="space-y-4">
                    {lesson.content_blocks.map((block, blockIndex) => (
                      <div
                        key={`lesson-${lessonIndex}-block-${blockIndex}`}
                        className="rounded-lg border border-muted/50 bg-muted/20 p-4"
                      >
                        <div className="flex flex-wrap items-center gap-3">
                          <Select
                            value={block.type}
                            onValueChange={(value) =>
                              updateContentBlock(lessonIndex, blockIndex, {
                                type: value as ContentBlockType,
                              })
                            }
                          >
                            <SelectTrigger className="w-[200px]">
                              <SelectValue placeholder="Tipo" />
                            </SelectTrigger>
                            <SelectContent>
                              <SelectItem value="text">Texto</SelectItem>
                              <SelectItem value="link">Link</SelectItem>
                              <SelectItem value="pdf">PDF</SelectItem>
                              <SelectItem value="video">Video</SelectItem>
                            </SelectContent>
                          </Select>
                          <Input
                            className="flex-1"
                            value={block.value}
                            onChange={(event) =>
                              updateContentBlock(lessonIndex, blockIndex, {
                                value: event.target.value,
                              })
                            }
                            placeholder="Contenido o URL"
                          />
                          <Button
                            size="icon"
                            variant="ghost"
                            onClick={() => removeContentBlock(lessonIndex, blockIndex)}
                            disabled={lesson.content_blocks.length === 1}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </CardContent>
      </Card>

      <Card>
        <CardHeader className="flex flex-row items-start justify-between gap-4">
          <div>
            <CardTitle>Cuestionario final</CardTitle>
            <p className="text-sm text-muted-foreground">
              {questionCount} preguntas configuradas · Puntaje total: {totalQuizScore}
            </p>
          </div>
          <Button variant="outline" onClick={addQuestion}>
            <Plus className="mr-2 h-4 w-4" />
            Agregar pregunta
          </Button>
        </CardHeader>
        <CardContent className="space-y-6">
          {questions.map((question, questionIndex) => (
            <Card key={`question-${questionIndex}`} className="border-dashed">
              <CardHeader className="flex flex-row items-start justify-between gap-4">
                <div>
                  <CardTitle className="text-base">Pregunta {questionIndex + 1}</CardTitle>
                  <p className="text-sm text-muted-foreground">
                    Selecciona la respuesta correcta y asigna puntaje.
                  </p>
                </div>
                <Button
                  size="icon"
                  variant="ghost"
                  onClick={() =>
                    setQuestions((prev) => prev.filter((_, index) => index !== questionIndex))
                  }
                  disabled={questions.length === 1}
                >
                  <Trash2 className="h-4 w-4" />
                </Button>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <Label>Pregunta</Label>
                  <Textarea
                    value={question.question}
                    onChange={(event) => updateQuestion(questionIndex, { question: event.target.value })}
                  />
                </div>
                <div className="grid gap-4 md:grid-cols-2">
                  <div className="space-y-2">
                    <Label>Puntaje</Label>
                    <Input
                      type="number"
                      min="0"
                      value={question.points}
                      onChange={(event) =>
                        updateQuestion(questionIndex, { points: Number(event.target.value) || 0 })
                      }
                    />
                  </div>
                  <div className="space-y-2">
                    <Label>Respuesta correcta</Label>
                    <Select
                      value={question.correctIndex.toString()}
                      onValueChange={(value) =>
                        updateQuestion(questionIndex, { correctIndex: Number(value) })
                      }
                    >
                      <SelectTrigger>
                        <SelectValue placeholder="Selecciona la correcta" />
                      </SelectTrigger>
                      <SelectContent>
                        {question.options.map((_, optionIndex) => (
                          <SelectItem key={`correct-${optionIndex}`} value={optionIndex.toString()}>
                            Opción {optionIndex + 1}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                </div>

                <div className="space-y-3">
                  <div className="flex items-center justify-between">
                    <Label>Opciones</Label>
                    <Button variant="ghost" size="sm" onClick={() => addOption(questionIndex)}>
                      <Plus className="mr-2 h-4 w-4" />
                      Agregar opción
                    </Button>
                  </div>
                  <div className="space-y-3">
                    {question.options.map((option, optionIndex) => (
                      <div key={`option-${optionIndex}`} className="flex items-center gap-3">
                        <Badge variant={optionIndex === question.correctIndex ? "default" : "secondary"}>
                          {optionIndex === question.correctIndex ? "Correcta" : `Opción ${optionIndex + 1}`}
                        </Badge>
                        <Input
                          className="flex-1"
                          value={option}
                          onChange={(event) =>
                            updateOption(questionIndex, optionIndex, event.target.value)
                          }
                        />
                        <Button
                          size="icon"
                          variant="ghost"
                          onClick={() => removeOption(questionIndex, optionIndex)}
                          disabled={question.options.length === 1}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    ))}
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </CardContent>
      </Card>

      <div className="flex flex-wrap items-center justify-end gap-3">
        <Button variant="outline" onClick={onSuccess}>
          Cancelar
        </Button>
        <Button onClick={handleSave} disabled={saving}>
          {saving ? "Guardando..." : isEdit ? "Guardar cambios" : "Crear módulo"}
        </Button>
      </div>
    </div>
  );
}