"use client";

import { type Dispatch, type SetStateAction, useMemo, useState } from "react";
import { Input } from "@/components/ui/input";
import { Search } from "lucide-react";
import { TaskFilter } from "./task-filter";
import { TaskItem } from "./task-item";
import { UnifiedTask } from "./types";
import { useUnifiedTasks } from "./use-unified-tasks";

interface TaskListProps {
  refreshKey?: number;
  tasks?: UnifiedTask[];
  setTasks?: Dispatch<SetStateAction<UnifiedTask[]>>;
  loading?: boolean;  
}

export function TaskList({ refreshKey, tasks: externalTasks, setTasks: externalSetTasks, loading: externalLoading }: TaskListProps) {
  const { tasks: fetchedTasks, setTasks: internalSetTasks, loading: internalLoading } = useUnifiedTasks(
    refreshKey,
    { skip: Boolean(externalTasks) }
  );
  const tasks = externalTasks ?? fetchedTasks;
  const setTasks = externalSetTasks ?? internalSetTasks;
  const loading = externalLoading ?? internalLoading;
  const [searchTerm, setSearchTerm] = useState("");
  const [filter, setFilter] = useState<"all" | "completed" | "pending" | "in_progress">("all");

  const filteredTasks = useMemo(() => {
    let filtered = tasks;

    if (filter === "completed") {
      filtered = filtered.filter((task) => task.completed || task.status === "completed");
    } else if (filter === "pending") {
      filtered = filtered.filter((task) => (!task.completed && task.status !== "completed") || task.status === "pending");
    } else if (filter === "in_progress") {
      filtered = filtered.filter((task) => task.status === "in_progress");
    }

    if (searchTerm) {
      filtered = filtered.filter(
        (task) =>
          task.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
          task.description?.toLowerCase().includes(searchTerm.toLowerCase()) ||
          task.assigned_by_name?.toLowerCase().includes(searchTerm.toLowerCase()) ||
          task.assignee_name?.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    return filtered;
  }, [filter, searchTerm, tasks]);

  const handleTaskUpdate = (taskId: string, updates: Partial<UnifiedTask>) => {
    setTasks((prevTasks) =>
      prevTasks.map((task) => (task.id === taskId ? { ...task, ...updates } : task))
    );
  };

  const handleTaskCreate = (task: UnifiedTask) => {
    setTasks((prevTasks) => [task, ...prevTasks]);
  };

  const handleTaskDelete = (taskId: string) => {
    setTasks((prevTasks) => prevTasks.filter((task) => task.id !== taskId));
  };

  if (loading) {
    return (
      <div className="p-6 text-center">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto"></div>
        <p className="mt-2 text-gray-600">Cargando tareas...</p>
      </div>
    );
  }

  return (
    <div className="space-y-4">
      <div className="p-6 border-b space-y-4">
        <div className="relative">
          <Search className="absolute left-3 top-3 h-4 w-4 text-gray-400" />
          <Input
            type="text"
            placeholder="Buscar tareas..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            className="pl-10"
          />
        </div>
        <TaskFilter filter={filter} onFilterChange={setFilter} />
      </div>

      <div className="p-6">
        {filteredTasks.length === 0 ? (
          <div className="text-center py-12">
            <div className="text-gray-400 text-lg mb-2">
              {searchTerm || filter !== "all" ? "No se encontraron tareas" : "No tienes tareas aún"}
            </div>
            <p className="text-gray-500">
              {searchTerm || filter !== "all" ? "Intenta cambiar los filtros" : "Crea tu primera tarea para comenzar"}
            </p>
          </div>
        ) : (
          <div className="space-y-3">
            {filteredTasks.map((task) => (
              <TaskItem
                key={`${task.source}-${task.id}`}
                task={task}
                onUpdate={handleTaskUpdate}
                onDelete={handleTaskDelete}
                onCreate={handleTaskCreate}
              />
            ))}
          </div>
        )}
      </div>
    </div>
  );
}