"use client";

import { useEffect, useState } from "react";
import { supabase } from "@/lib/supabase";

type UseSgiAdminPermissionResult = {
  canManage: boolean;
  loading: boolean;
};

export const useSgiAdminPermission = (userId: string | null | undefined, pagePath: string): UseSgiAdminPermissionResult => {
  const [canManage, setCanManage] = useState(false);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    let isActive = true;

    if (!userId) {
      setCanManage(false);
      setLoading(false);
      return;
    }

    const loadPermission = async () => {
      setLoading(true);
      const normalizedPath = pagePath.endsWith("/") ? pagePath.slice(0, -1) : pagePath;
      const alternatePath = normalizedPath === pagePath ? `${normalizedPath}/` : normalizedPath;
      const lastSlashIndex = normalizedPath.lastIndexOf("/");
      const parentPath = lastSlashIndex > 0 ? normalizedPath.slice(0, lastSlashIndex) : null;
      const parentAlternatePath = parentPath ? `${parentPath}/` : null;
      const pathsToCheck = [normalizedPath, alternatePath];

      if (parentPath) {
        pathsToCheck.push(parentPath);
      }

      if (parentAlternatePath) {
        pathsToCheck.push(parentAlternatePath);
      }

      const { data: pageData, error: pageError } = await supabase
        .from("sgi_pages")
        .select("id, path")
        .in("path", pathsToCheck)
        .order("created_at", { ascending: true });

      if (!isActive) {
        return;
      }

      const pageIds = pageData?.map((page) => page.id) ?? [];

      if (pageError || pageIds.length === 0) {
        setCanManage(false);
        setLoading(false);
        return;
      }

      const { data: permissionData, error: permissionError } = await supabase
        .from("sgi_admin_permissions")
        .select("id")
        .eq("user_id", userId)
        .in("page_id", pageIds)

      if (!isActive) {
        return;
      }

      setCanManage(Boolean(permissionData?.length) && !permissionError);
      setLoading(false);
    };

    loadPermission();

    const channel = supabase
      .channel(`sgi-admin-permissions-${userId}-${pagePath}`)
      .on(
        "postgres_changes",
        {
          event: "*",
          schema: "public",
          table: "sgi_admin_permissions",
          filter: `user_id=eq.${userId}`,
        },
        () => {
          loadPermission();
        },
      )
      .subscribe();

    return () => {
      isActive = false;
      supabase.removeChannel(channel);
    };
  }, [pagePath, userId]);

  return { canManage, loading };
};