import { supabase } from "@/lib/supabase";
import type { User } from "@supabase/supabase-js";

export type PointsSummary = {
  user_id: string;
  points_earned: number;
  points_spent: number;
  points_available: number;
};

export const getCurrentUser = async () => {
  const {
    data: { session },
  } = await supabase.auth.getSession();
  return session?.user ?? null;
};

export const fetchRewards = async () => {
  return supabase
    .from("mp_rewards")
    .select("id, name, description, points_value, stock_available, image_url, storage_path")
    .order("created_at", { ascending: false });
};

export const fetchActivities = async () => {
  return supabase
    .from("mp_activities")
    .select("id, date, name, description, points_value")
    .order("date", { ascending: false });
};

export const fetchAllocations = async () => {
  return supabase
    .from("mp_allocations")
    .select(
      "id, user_id, activity_id, points, assigned_at, assigned_by, comment, profiles(full_name, email), mp_activities(name, points_value)"
    )
    .order("assigned_at", { ascending: false });
};

export const fetchUserStatuses = async () => {
  return supabase
    .from("mp_user_status")
    .select("id, user_id, status, profiles(full_name, email)")
    .order("status", { ascending: true });
};

export const fetchUsers = async () => {
  return supabase.from("profiles").select("id, full_name, email").order("full_name");
};

export const fetchRewardsMatrix = async () => {
  return supabase
    .from("mp_rewards")
    .select("id, name, description, points_value, stock_available, image_url, storage_path")
    .order("created_at", { ascending: false });
};

export const fetchRedemptions = async () => {
  return supabase
    .from("mp_redemptions")
    .select(
      "id, reward_id, user_id, points, stock_before, stock_after, requested_at, status, mp_rewards(name, description, points_value), profiles(full_name, email)"
    )
    .order("requested_at", { ascending: false });
};

export const fetchUserAllocations = async (userId: string) => {
  return supabase
    .from("mp_allocations")
    .select(
      "id, user_id, activity_id, points, assigned_at, assigned_by, comment, mp_activities(name, description, date)"
    )
    .eq("user_id", userId)
    .order("assigned_at", { ascending: false });
};

export const fetchUserRedemptions = async (userId: string) => {
  return supabase
    .from("mp_redemptions")
    .select("id, reward_id, user_id, points, requested_at, status, mp_rewards(name, description)")
    .eq("user_id", userId)
    .order("requested_at", { ascending: false });
};

export const fetchUserPointsSummary = async () => {
  return supabase
    .from("mp_user_points_summary")
    .select("user_id, points_earned, points_spent, points_available");
};

export const fetchUserPointsSummaryForUser = async (userId: string) => {
  return supabase
    .from("mp_user_points_summary")
    .select("user_id, points_earned, points_spent, points_available")
    .eq("user_id", userId)
    .maybeSingle();
};

export const requestRewardRedemption = async (rewardId: string, user: User) => {
  return supabase.rpc("request_redemption", { reward_id: rewardId, user_id: user.id });
};