-- Make the signup trigger resilient so user creation never fails because of
-- auxiliary inserts (profiles/default categories).
CREATE OR REPLACE FUNCTION handle_new_user()
RETURNS TRIGGER AS $$
BEGIN
  -- Create or refresh the user's profile
  INSERT INTO profiles (id, full_name, email, avatar_url)
  VALUES (
    NEW.id,
    NEW.raw_user_meta_data ->> 'full_name',
    NEW.email,
    NEW.raw_user_meta_data ->> 'avatar_url'
  )
  ON CONFLICT (id) DO UPDATE
    SET full_name = EXCLUDED.full_name,
        email = EXCLUDED.email,
        avatar_url = EXCLUDED.avatar_url;

  -- Seed default categories without blocking signup if something goes wrong
  BEGIN
    INSERT INTO categories (name, color, user_id) VALUES
      ('Trabajo', '#3B82F6', NEW.id),
      ('Personal', '#10B981', NEW.id),
      ('Estudios', '#F59E0B', NEW.id),
      ('Hogar', '#EF4444', NEW.id);
  EXCEPTION WHEN OTHERS THEN
    -- Log the issue but keep allowing the user account to be created
    RAISE WARNING 'Default categories could not be created for %: %', NEW.id, SQLERRM;
  END;

  RETURN NEW;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER;

-- Ensure the trigger points to the refreshed function
DROP TRIGGER IF EXISTS on_auth_user_created ON auth.users;
CREATE TRIGGER on_auth_user_created
  AFTER INSERT ON auth.users
  FOR EACH ROW EXECUTE FUNCTION handle_new_user();

