-- Create team members table if it doesn't exist
create table if not exists team_members (
  id uuid primary key default gen_random_uuid(),
  owner_id uuid not null references auth.users(id) on delete cascade,
  member_id uuid not null references auth.users(id) on delete cascade,
  email text not null,
  role text,
  status text not null default 'pending' check (status in ('pending', 'active', 'removed')),
  created_at timestamptz not null default now(),
  constraint team_members_unique_pair unique (owner_id, member_id)
);

-- Create team tasks table if it doesn't exist
create table if not exists team_tasks (
  id uuid primary key default gen_random_uuid(),
  member_id uuid not null references auth.users(id) on delete cascade,
  assigned_by uuid not null references auth.users(id) on delete cascade,
  title text not null,
  description text,
  status text not null default 'pending' check (status in ('pending', 'in_progress', 'completed')),
  due_date date,
  created_at timestamptz not null default now()
);

-- Enable RLS for both tables
alter table if exists team_members enable row level security;
alter table if exists team_tasks enable row level security;

-- Policies for team_members
create policy if not exists "Owners can view their team" on team_members
  for select
  to authenticated
  using (owner_id = auth.uid());

create policy if not exists "Owners can add team members" on team_members
  for insert
  to authenticated
  with check (owner_id = auth.uid());

create policy if not exists "Owners can update team members" on team_members
  for update
  to authenticated
  using (owner_id = auth.uid());

-- Policies for team_tasks
create policy if not exists "View tasks related to user" on team_tasks
  for select
  to authenticated
  using (assigned_by = auth.uid() or member_id = auth.uid());

create policy if not exists "Owners can assign tasks" on team_tasks
  for insert
  to authenticated
  with check (assigned_by = auth.uid());

create policy if not exists "Task owners can update" on team_tasks
  for update
  to authenticated
  using (assigned_by = auth.uid());

create index if not exists idx_team_members_owner on team_members(owner_id);
create index if not exists idx_team_tasks_assigned_by on team_tasks(assigned_by);
create index if not exists idx_team_tasks_member on team_tasks(member_id);
