set search_path = public, extensions, pg_temp;

-- Teams table to allow multiple named groups per owner
create table if not exists teams (
  id uuid primary key default gen_random_uuid(),
  owner_id uuid not null references auth.users(id) on delete cascade,
  name text not null,
  created_at timestamptz not null default now()
);

alter table if exists teams enable row level security;

do $$
begin
  if not exists (
    select 1 from pg_policies where schemaname = 'public' and tablename = 'teams' and policyname = 'Owners can manage teams'
  ) then
    create policy "Owners can manage teams" on teams
      for all
      to authenticated
      using (owner_id = auth.uid())
      with check (owner_id = auth.uid());
  end if;
end;
$$;

-- Team members now reference a team so users can belong to multiple groups
alter table if exists team_members
  add column if not exists member_id uuid references auth.users(id) on delete cascade,
  add column if not exists team_id uuid references teams(id) on delete cascade;

-- Build a default team for existing owners and attach their members
do $$
declare
  owner_record record;
  created_team uuid;
begin
  for owner_record in select distinct owner_id from team_members where team_id is null loop
    insert into teams (owner_id, name)
    values (
      owner_record.owner_id,
      coalesce(
        (select 'Equipo de ' || coalesce(full_name, 'mi equipo') from profiles where id = owner_record.owner_id limit 1),
        'Equipo principal'
      )
    )
    on conflict do nothing
    returning id into created_team;

    if created_team is null then
      select id into created_team from teams where owner_id = owner_record.owner_id order by created_at asc limit 1;
    end if;

    update team_members
      set team_id = created_team
      where owner_id = owner_record.owner_id and team_id is null;
  end loop;
end;
$$;

alter table if exists team_members drop constraint if exists team_members_unique_pair;
alter table if exists team_members
  add constraint team_members_unique_team_member unique (team_id, member_id);

alter table if exists team_members alter column team_id set not null;

-- Refresh policies so owners can manage members within their teams
do $$
begin
  if exists (
    select 1 from pg_policies where schemaname = 'public' and tablename = 'team_members' and policyname = 'Owners can view their team'
  ) then
    drop policy "Owners can view their team" on team_members;
  end if;

  if exists (
    select 1 from pg_policies where schemaname = 'public' and tablename = 'team_members' and policyname = 'Owners can add team members'
  ) then
    drop policy "Owners can add team members" on team_members;
  end if;

  if exists (
    select 1 from pg_policies where schemaname = 'public' and tablename = 'team_members' and policyname = 'Owners can update team members'
  ) then
    drop policy "Owners can update team members" on team_members;
  end if;

  create policy "Owners can view their teams" on team_members
    for select
    to authenticated
    using (
      team_id in (select id from teams where owner_id = auth.uid())
      or member_id = auth.uid()
    );

  create policy "Owners can add team members" on team_members
    for insert
    to authenticated
    with check (
      team_id in (select id from teams where owner_id = auth.uid())
    );

  create policy "Owners can update team members" on team_members
    for update
    to authenticated
    using (
      team_id in (select id from teams where owner_id = auth.uid())
    );
end;
$$;

-- Update invitation helper to require a target team
create or replace function invite_team_member(
  invite_email text,
  invite_role text default 'Colaborador',
  target_team uuid default null
)
returns team_members
language plpgsql
security definer
as $$
declare
  owner uuid;
  target_profile record;
  inserted_row team_members;
begin
  owner := auth.uid();

  if owner is null then
    raise exception using errcode = 'P0001', message = 'NOT_AUTHENTICATED';
  end if;

  if target_team is null then
    raise exception using errcode = 'P0001', message = 'TEAM_REQUIRED';
  end if;

  if not exists (select 1 from teams where id = target_team and owner_id = owner) then
    raise exception using errcode = 'P0001', message = 'NOT_TEAM_OWNER';
  end if;

  select *
  into target_profile
  from find_profile_by_email(invite_email)
  limit 1;

  if target_profile.id is null then
    raise exception using errcode = 'P0001', message = 'PROFILE_NOT_FOUND';
  end if;

 insert into team_members(owner_id, member_id, email, role, status, team_id)
  values (
    owner,
    target_profile.id,
    coalesce(target_profile.email, lower(invite_email)),
    invite_role,
    'active',
    target_team
  )
  on conflict (team_id, member_id) do update
    set
      role = excluded.role,
      status = 'active',
      email = excluded.email
  returning *
  into inserted_row;

  return inserted_row;
end;
$$;

grant execute on function invite_team_member(text, text, uuid) to authenticated;

-- Allow listing registered profiles (basic fields only) for team selection
create or replace function list_profiles_for_team(search text default '')
returns table (
  id uuid,
  full_name text,
  email text,
  avatar_url text
) as $$
begin
  return query
  select p.id, p.full_name, p.email, p.avatar_url
  from profiles p
  where search = ''
    or p.full_name ilike ('%' || search || '%')
    or p.email ilike ('%' || search || '%');
end;
$$ language plpgsql security definer;

grant execute on function list_profiles_for_team(text) to authenticated;